<?php

namespace Tests\Feature;

use App\Models\ChartOfAccount;
use App\Models\Company;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ChartOfAccountAutoSetupTest extends TestCase
{
    use RefreshDatabase;

    /**
     * اختبار إنشاء الحسابات الأساسية تلقائياً عند فتح صفحة دليل الحسابات لأول مرة
     *
     * @return void
     */
    public function test_default_accounts_are_created_automatically_on_first_visit()
    {
        // إنشاء شركة ومستخدم
        $company = Company::factory()->create();
        $user = User::factory()->create(['company_id' => $company->id]);

        // التأكد من عدم وجود حسابات قبل الزيارة
        $this->assertEquals(0, ChartOfAccount::where('company_id', $company->id)->count());

        // زيارة صفحة دليل الحسابات
        $response = $this->actingAs($user)->get(route('chart-of-accounts.index'));

        // التحقق من نجاح الطلب
        $response->assertStatus(200);

        // التحقق من إنشاء 5 حسابات أساسية
        $this->assertEquals(5, ChartOfAccount::where('company_id', $company->id)->count());

        // التحقق من وجود رسالة النجاح
        $response->assertSessionHas('success', 'تم إنشاء الحسابات الأساسية بنجاح! يمكنك الآن إضافة حسابات فرعية.');
    }

    /**
     * اختبار أن الحسابات الأساسية تحتوي على البيانات الصحيحة
     *
     * @return void
     */
    public function test_default_accounts_have_correct_data()
    {
        // إنشاء شركة ومستخدم
        $company = Company::factory()->create();
        $user = User::factory()->create(['company_id' => $company->id]);

        // زيارة صفحة دليل الحسابات لإنشاء الحسابات الأساسية
        $this->actingAs($user)->get(route('chart-of-accounts.index'));

        // التحقق من حساب الأصول
        $assetsAccount = ChartOfAccount::where('company_id', $company->id)
            ->where('account_code', '1000')
            ->first();

        $this->assertNotNull($assetsAccount);
        $this->assertEquals('الأصول', $assetsAccount->name);
        $this->assertEquals('Assets', $assetsAccount->name_en);
        $this->assertEquals('asset', $assetsAccount->account_type);
        $this->assertTrue($assetsAccount->is_parent);
        $this->assertEquals(1, $assetsAccount->level);
        $this->assertNull($assetsAccount->parent_id);
        $this->assertEquals(0, $assetsAccount->opening_balance);
        $this->assertEquals('SAR', $assetsAccount->currency);
        $this->assertTrue($assetsAccount->is_active);

        // التحقق من حساب الخصوم
        $liabilitiesAccount = ChartOfAccount::where('company_id', $company->id)
            ->where('account_code', '2000')
            ->first();

        $this->assertNotNull($liabilitiesAccount);
        $this->assertEquals('الخصوم', $liabilitiesAccount->name);
        $this->assertEquals('liability', $liabilitiesAccount->account_type);

        // التحقق من حساب رأس المال
        $equityAccount = ChartOfAccount::where('company_id', $company->id)
            ->where('account_code', '3000')
            ->first();

        $this->assertNotNull($equityAccount);
        $this->assertEquals('رأس المال وحقوق الملكية', $equityAccount->name);
        $this->assertEquals('equity', $equityAccount->account_type);

        // التحقق من حساب الإيرادات
        $revenueAccount = ChartOfAccount::where('company_id', $company->id)
            ->where('account_code', '4000')
            ->first();

        $this->assertNotNull($revenueAccount);
        $this->assertEquals('الإيرادات', $revenueAccount->name);
        $this->assertEquals('revenue', $revenueAccount->account_type);

        // التحقق من حساب المصروفات
        $expenseAccount = ChartOfAccount::where('company_id', $company->id)
            ->where('account_code', '5000')
            ->first();

        $this->assertNotNull($expenseAccount);
        $this->assertEquals('المصروفات', $expenseAccount->name);
        $this->assertEquals('expense', $expenseAccount->account_type);
    }

    /**
     * اختبار أن الحسابات لا تُنشأ مرة أخرى عند الزيارة الثانية
     *
     * @return void
     */
    public function test_default_accounts_are_not_created_again_on_second_visit()
    {
        // إنشاء شركة ومستخدم
        $company = Company::factory()->create();
        $user = User::factory()->create(['company_id' => $company->id]);

        // الزيارة الأولى - إنشاء الحسابات
        $this->actingAs($user)->get(route('chart-of-accounts.index'));
        $this->assertEquals(5, ChartOfAccount::where('company_id', $company->id)->count());

        // الزيارة الثانية - لا يجب إنشاء حسابات جديدة
        $response = $this->actingAs($user)->get(route('chart-of-accounts.index'));
        $this->assertEquals(5, ChartOfAccount::where('company_id', $company->id)->count());

        // التحقق من عدم وجود رسالة النجاح في الزيارة الثانية
        $response->assertSessionMissing('success');
    }

    /**
     * اختبار عزل الحسابات بين الشركات المختلفة
     *
     * @return void
     */
    public function test_default_accounts_are_isolated_between_companies()
    {
        // إنشاء شركتين ومستخدمين
        $company1 = Company::factory()->create();
        $user1 = User::factory()->create(['company_id' => $company1->id]);

        $company2 = Company::factory()->create();
        $user2 = User::factory()->create(['company_id' => $company2->id]);

        // زيارة صفحة دليل الحسابات للشركة الأولى
        $this->actingAs($user1)->get(route('chart-of-accounts.index'));

        // زيارة صفحة دليل الحسابات للشركة الثانية
        $this->actingAs($user2)->get(route('chart-of-accounts.index'));

        // التحقق من أن كل شركة لها 5 حسابات
        $this->assertEquals(5, ChartOfAccount::where('company_id', $company1->id)->count());
        $this->assertEquals(5, ChartOfAccount::where('company_id', $company2->id)->count());

        // التحقق من أن إجمالي الحسابات هو 10 (5 لكل شركة)
        $this->assertEquals(10, ChartOfAccount::count());

        // التحقق من أن حسابات الشركة الأولى لا تظهر للشركة الثانية
        $company1Accounts = ChartOfAccount::where('company_id', $company1->id)->pluck('id')->toArray();
        $company2Accounts = ChartOfAccount::where('company_id', $company2->id)->pluck('id')->toArray();

        // التأكد من عدم وجود تداخل بين الحسابات
        $this->assertEmpty(array_intersect($company1Accounts, $company2Accounts));
    }

    /**
     * اختبار أن المستخدم بدون شركة لا يمكنه الوصول لدليل الحسابات
     *
     * @return void
     */
    public function test_user_without_company_cannot_access_chart_of_accounts()
    {
        // إنشاء مستخدم بدون شركة
        $user = User::factory()->create(['company_id' => null]);

        // محاولة زيارة صفحة دليل الحسابات
        $response = $this->actingAs($user)->get(route('chart-of-accounts.index'));

        // التحقق من إعادة التوجيه مع رسالة خطأ
        $response->assertRedirect(route('dashboard'));
        $response->assertSessionHas('error', 'لا يوجد شركة مرتبطة بالمستخدم الحالي');

        // التحقق من عدم إنشاء أي حسابات
        $this->assertEquals(0, ChartOfAccount::count());
    }

    /**
     * اختبار أن الحسابات الأساسية تُنشأ بأكواد فريدة
     *
     * @return void
     */
    public function test_default_accounts_have_unique_codes()
    {
        // إنشاء شركة ومستخدم
        $company = Company::factory()->create();
        $user = User::factory()->create(['company_id' => $company->id]);

        // زيارة صفحة دليل الحسابات
        $this->actingAs($user)->get(route('chart-of-accounts.index'));

        // الحصول على جميع أكواد الحسابات
        $accountCodes = ChartOfAccount::where('company_id', $company->id)
            ->pluck('account_code')
            ->toArray();

        // التحقق من أن جميع الأكواد فريدة
        $this->assertEquals(count($accountCodes), count(array_unique($accountCodes)));

        // التحقق من وجود الأكواد المتوقعة
        $expectedCodes = ['1000', '2000', '3000', '4000', '5000'];
        $this->assertEquals($expectedCodes, $accountCodes);
    }
}
