<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;
use App\Models\User;
use App\Models\CashRegister;
use App\Models\CashTransaction;
use App\Models\ChartOfAccount;
use App\Models\Company;
use App\Models\FiscalYear;
use App\Models\AccountingPeriod;
use App\Http\Controllers\CashTransactionController;
use Illuminate\Support\Facades\DB;

class CashDepositJournalEntryTest extends TestCase
{
    /**
     * اختبار إنشاء قيد محاسبي للإيداع النقدي
     *
     * @return void
     */
    public function testCreateJournalEntryForDeposit()
    {
        // إنشاء بيانات الاختبار
        $company = Company::first();

        if (!$company) {
            $this->markTestSkipped('لا توجد شركات في قاعدة البيانات');
        }

        // التأكد من وجود حساب خزينة نشط
        $cashAccount = ChartOfAccount::where('company_id', $company->id)
            ->where('is_active', true)
            ->where('is_cash_account', true)
            ->first();

        if (!$cashAccount) {
            // إنشاء حساب خزينة إذا لم يكن موجودًا
            $cashAccount = new ChartOfAccount();
            $cashAccount->company_id = $company->id;
            $cashAccount->parent_id = null;
            $cashAccount->code = '1010';
            $cashAccount->name = 'حساب الخزينة';
            $cashAccount->name_en = 'Cash Account';
            $cashAccount->account_type = 'asset';
            $cashAccount->is_active = true;
            $cashAccount->is_cash_account = true;
            $cashAccount->save();

            echo "تم إنشاء حساب خزينة جديد برقم {$cashAccount->id}\n";
        } else {
            echo "تم العثور على حساب خزينة موجود برقم {$cashAccount->id}\n";
        }

        // التأكد من وجود حساب للطرف الدائن
        $creditAccount = ChartOfAccount::where('company_id', $company->id)
            ->where('is_active', true)
            ->where('name', 'حساب العميل')
            ->first();

        if (!$creditAccount) {
            // إنشاء حساب للطرف الدائن إذا لم يكن موجودًا
            $creditAccount = new ChartOfAccount();
            $creditAccount->company_id = $company->id;
            $creditAccount->parent_id = null;
            $creditAccount->code = '2010';
            $creditAccount->name = 'حساب العميل';
            $creditAccount->name_en = 'Customer Account';
            $creditAccount->account_type = 'liability';
            $creditAccount->is_active = true;
            $creditAccount->save();

            echo "تم إنشاء حساب للطرف الدائن برقم {$creditAccount->id}\n";
        } else {
            echo "تم العثور على حساب للطرف الدائن موجود برقم {$creditAccount->id}\n";
        }

        // التأكد من وجود سنة مالية نشطة
        $fiscalYear = FiscalYear::where('company_id', $company->id)
            ->where('is_active', true)
            ->where('is_closed', false)
            ->first();

        if (!$fiscalYear) {
            // إنشاء سنة مالية إذا لم تكن موجودة
            $fiscalYear = new FiscalYear();
            $fiscalYear->company_id = $company->id;
            $fiscalYear->name = 'السنة المالية 2025';
            $fiscalYear->start_date = '2025-01-01';
            $fiscalYear->end_date = '2025-12-31';
            $fiscalYear->is_active = true;
            $fiscalYear->is_closed = false;
            $fiscalYear->save();

            echo "تم إنشاء سنة مالية جديدة برقم {$fiscalYear->id}\n";
        } else {
            echo "تم العثور على سنة مالية موجودة برقم {$fiscalYear->id}\n";
        }

        // التأكد من وجود فترة محاسبية نشطة
        $accountingPeriod = AccountingPeriod::where('fiscal_year_id', $fiscalYear->id)
            ->where('is_closed', false)
            ->where('start_date', '<=', now())
            ->where('end_date', '>=', now())
            ->first();

        if (!$accountingPeriod) {
            // إنشاء فترة محاسبية إذا لم تكن موجودة
            $accountingPeriod = new AccountingPeriod();
            $accountingPeriod->fiscal_year_id = $fiscalYear->id;
            $accountingPeriod->name = 'الفترة المحاسبية الحالية';
            $accountingPeriod->start_date = now()->startOfMonth();
            $accountingPeriod->end_date = now()->endOfMonth();
            $accountingPeriod->is_closed = false;
            $accountingPeriod->save();

            echo "تم إنشاء فترة محاسبية جديدة برقم {$accountingPeriod->id}\n";
        } else {
            echo "تم العثور على فترة محاسبية موجودة برقم {$accountingPeriod->id}\n";
        }

        // التأكد من وجود خزينة نشطة
        $cashRegister = CashRegister::where('company_id', $company->id)
            ->where('is_active', true)
            ->first();

        if (!$cashRegister) {
            // إنشاء خزينة إذا لم تكن موجودة
            $cashRegister = new CashRegister();
            $cashRegister->company_id = $company->id;
            $cashRegister->name = 'الخزينة الرئيسية';
            $cashRegister->current_balance = 0;
            $cashRegister->is_active = true;
            $cashRegister->save();

            echo "تم إنشاء خزينة جديدة برقم {$cashRegister->id}\n";
        } else {
            echo "تم العثور على خزينة موجودة برقم {$cashRegister->id}\n";
        }

        // التأكد من وجود مستخدم
        $user = User::where('company_id', $company->id)->first();

        if (!$user) {
            $this->markTestSkipped('لا يوجد مستخدمين في قاعدة البيانات');
        }

        // إنشاء معاملة إيداع نقدي
        $transaction = new CashTransaction();
        $transaction->company_id = $company->id;
        $transaction->cash_register_id = $cashRegister->id;
        $transaction->user_id = $user->id;
        $transaction->type = 'deposit';
        $transaction->amount = 100;
        $transaction->party_name = $creditAccount->name;
        $transaction->party_type = 'customer';
        $transaction->notes = 'اختبار إنشاء قيد محاسبي للإيداع النقدي';
        $transaction->save();

        echo "تم إنشاء معاملة إيداع نقدي جديدة برقم {$transaction->id}\n";

        // تحديث رصيد الخزينة
        $cashRegister->current_balance += $transaction->amount;
        $cashRegister->save();

        // إنشاء قيد محاسبي للإيداع
        $controller = new CashTransactionController();
        $entry = $controller->createJournalEntryForDeposit($transaction);

        // التحقق من نجاح إنشاء القيد المحاسبي
        if ($entry) {
            echo "تم إنشاء قيد محاسبي بنجاح برقم {$entry->id}\n";

            // التحقق من وجود بنود القيد المحاسبي
            $items = DB::table('journal_entry_items')
                ->where('journal_entry_id', $entry->id)
                ->get();

            echo "عدد بنود القيد المحاسبي: {$items->count()}\n";

            // التحقق من وجود سجل في جدول الربط
            $link = DB::table('cash_transaction_entries')
                ->where('cash_transaction_id', $transaction->id)
                ->where('journal_entry_id', $entry->id)
                ->first();

            if ($link) {
                echo "تم إنشاء سجل الربط بين المعاملة والقيد المحاسبي بنجاح\n";
            } else {
                echo "فشل إنشاء سجل الربط بين المعاملة والقيد المحاسبي\n";
            }

            $this->assertTrue(true);
        } else {
            echo "فشل إنشاء قيد محاسبي للإيداع النقدي\n";
            $this->assertTrue(false, 'فشل إنشاء قيد محاسبي للإيداع النقدي');
        }
    }
}
