<?php

require_once __DIR__ . '/vendor/autoload.php';

$app = require_once __DIR__ . '/bootstrap/app.php';
$app->make(\Illuminate\Contracts\Console\Kernel::class)->bootstrap();

use App\Models\CashTransaction;
use App\Http\Controllers\CashTransactionController;
use App\Models\ChartOfAccount;
use App\Models\JournalEntry;
use App\Models\JournalEntryItem;
use Illuminate\Support\Facades\DB;

// Obtener la transacción con ID 10
$transaction = CashTransaction::find(10);

if (!$transaction) {
    echo "No se encontró la transacción con ID 8\n";
    exit;
}

// Verificar si ya existe un registro en cash_transaction_entries para esta transacción
$existingEntry = DB::table('cash_transaction_entries')
    ->where('cash_transaction_id', $transaction->id)
    ->first();

if ($existingEntry) {
    echo "Ya existe un asiento contable para esta transacción (ID: {$existingEntry->journal_entry_id})\n";

    // Eliminar el registro existente para poder crear uno nuevo
    DB::table('cash_transaction_entries')
        ->where('cash_transaction_id', $transaction->id)
        ->delete();

    echo "Se ha eliminado el registro existente para crear uno nuevo.\n\n";
}

// Verificar la cuenta de caja
$cashAccount = ChartOfAccount::where('company_id', $transaction->company_id)
    ->where('is_active', true)
    ->where('is_cash_account', true)
    ->first();

if (!$cashAccount) {
    $cashAccount = ChartOfAccount::where('company_id', $transaction->company_id)
        ->where('is_active', true)
        ->where(function ($query) {
            $query->where('name', 'النقدية والبنوك')
                ->orWhere('name_en', 'Cash and Banks');
        })
        ->first();
}

echo "Cuenta de caja encontrada: " . ($cashAccount ? "Sí - {$cashAccount->name} (ID: {$cashAccount->id})" : "No") . "\n";

// Verificar la cuenta de gastos
$debitAccount = ChartOfAccount::where('company_id', $transaction->company_id)
    ->where('is_active', true)
    ->where(function ($query) use ($transaction) {
        $query->where('name', $transaction->party_name)
            ->orWhere('name_en', $transaction->party_name);
    })
    ->first();

echo "Cuenta de gastos encontrada: " . ($debitAccount ? "Sí - {$debitAccount->name} (ID: {$debitAccount->id})" : "No") . "\n\n";

// Crear una instancia del controlador
$controller = new CashTransactionController();

// Usar reflexión para acceder al método privado
$method = new ReflectionMethod(CashTransactionController::class, 'createJournalEntryForWithdrawal');
$method->setAccessible(true);

// Intentar crear el asiento contable
try {
    DB::beginTransaction();

    $result = $method->invoke($controller, $transaction);

    if ($result) {
        echo "¡Éxito! Se ha creado el asiento contable con ID: " . $result->id . "\n";

        // Mostrar detalles del asiento
        echo "Detalles del asiento:\n";
        echo "- Número: " . $result->entry_number . "\n";
        echo "- Fecha: " . $result->entry_date . "\n";
        echo "- Descripción: " . $result->description . "\n";

        // Mostrar los elementos del asiento
        echo "\nElementos del asiento:\n";
        foreach ($result->items as $item) {
            $account = ChartOfAccount::find($item->account_id);
            echo "- Cuenta: " . $account->name . " (" . $account->account_code . ")\n";
            echo "  Debe: " . $item->debit . ", Haber: " . $item->credit . "\n";
            echo "  Descripción: " . $item->description . "\n\n";
        }

        DB::commit();
        echo "Transacción completada con éxito.\n";
    } else {
        echo "Error: No se pudo crear el asiento contable\n";
        DB::rollBack();
    }
} catch (\Exception $e) {
    DB::rollBack();
    echo "Error: " . $e->getMessage() . "\n";
    echo "Archivo: " . $e->getFile() . " en la línea " . $e->getLine() . "\n";

    // Mostrar la traza completa para depuración
    echo "\nTraza de error:\n";
    echo $e->getTraceAsString() . "\n";
}
