<?php

require __DIR__ . '/vendor/autoload.php';

$app = require_once __DIR__ . '/bootstrap/app.php';
$app->make(\Illuminate\Contracts\Console\Kernel::class)->bootstrap();

echo "Running fix:cash-transaction-entries command...\n";

// Get all cash transactions
$allTransactions = \App\Models\CashTransaction::count();
echo "Total cash transactions: {$allTransactions}\n";

// Check if there are any journal entries
$journalEntries = \App\Models\JournalEntry::count();
echo "Total journal entries: {$journalEntries}\n";

// Check if there are any cash transaction entries
$cashTransactionEntries = DB::table('cash_transaction_entries')->count();
echo "Total cash transaction entries: {$cashTransactionEntries}\n";

// Get all cash transactions that don't have journal entries
$transactions = \App\Models\CashTransaction::whereNotIn('id', function ($query) {
    $query->select('cash_transaction_id')
        ->from('cash_transaction_entries');
})->get();

echo "Cash transactions needing journal entries: {$transactions->count()}\n";

// Get active fiscal year
$fiscalYear = \App\Models\FiscalYear::where('is_active', true)->first();
echo "Active fiscal year: " . ($fiscalYear ? $fiscalYear->name : 'None') . "\n";

// Get current accounting period
if ($fiscalYear) {
    $accountingPeriod = \App\Models\AccountingPeriod::where('fiscal_year_id', $fiscalYear->id)
        ->where('start_date', '<=', now())
        ->where('end_date', '>=', now())
        ->first();

    if (!$accountingPeriod) {
        $accountingPeriod = \App\Models\AccountingPeriod::where('fiscal_year_id', $fiscalYear->id)->first();
    }

    echo "Accounting period: " . ($accountingPeriod ? $accountingPeriod->name : 'None') . "\n";
}

// Get cash account
$cashAccount = \App\Models\ChartOfAccount::where(function ($query) {
    $query->where('name', 'like', '%cash%')
        ->orWhere('name', 'like', '%الصندوق%')
        ->orWhere('name', 'like', '%خزينة%')
        ->orWhere('name_en', 'like', '%cash%')
        ->orWhere('is_cash_account', true)
        ->orWhere('account_type', 'cash');
})->first();

echo "Cash account: " . ($cashAccount ? $cashAccount->name : 'None') . "\n";

// Get revenue account
$revenueAccount = \App\Models\ChartOfAccount::where(function ($query) {
    $query->where('name', 'like', '%revenue%')
        ->orWhere('name', 'like', '%إيراد%')
        ->orWhere('name', 'like', '%دخل%')
        ->orWhere('name_en', 'like', '%revenue%')
        ->orWhere('account_type', 'revenue');
})->first();

echo "Revenue account: " . ($revenueAccount ? $revenueAccount->name : 'None') . "\n";

// Process each transaction
$createdCount = 0;
foreach ($transactions as $transaction) {
    echo "\nProcessing cash transaction ID: {$transaction->id}, Type: {$transaction->type}, Amount: {$transaction->amount}\n";

    try {
        // Create journal entry
        $entry = new \App\Models\JournalEntry();
        $entry->company_id = $transaction->company_id ?? 1;
        $entry->fiscal_year_id = $fiscalYear->id;
        $entry->accounting_period_id = $accountingPeriod->id;
        $entry->entry_number = \App\Models\JournalEntry::generateEntryNumber($transaction->company_id ?? 1);
        $entry->entry_date = $transaction->created_at ?? now();
        $entry->reference_type = 'App\\Models\\CashTransaction';
        $entry->reference_id = $transaction->id;
        $entry->description = ($transaction->type == 'deposit' ? 'إيداع نقدي' : 'سحب نقدي') . ' - ' . ($transaction->notes ?? 'معاملة خزينة');
        $entry->created_by = $transaction->user_id ?? 1;
        $entry->is_posted = false;
        $entry->is_approved = false;
        $entry->save();

        echo "Created journal entry ID: {$entry->id}, Entry number: {$entry->entry_number}\n";

        // Create journal entry items
        if ($transaction->type == 'deposit') {
            // Debit cash account
            $debitItem = new \App\Models\JournalEntryItem();
            $debitItem->journal_entry_id = $entry->id;
            $debitItem->account_id = $cashAccount->id;
            $debitItem->debit = $transaction->amount;
            $debitItem->credit = 0;
            $debitItem->description = 'إيداع في الصندوق';
            $debitItem->save();

            // Credit revenue account
            $creditItem = new \App\Models\JournalEntryItem();
            $creditItem->journal_entry_id = $entry->id;
            $creditItem->account_id = $revenueAccount->id;
            $creditItem->debit = 0;
            $creditItem->credit = $transaction->amount;
            $creditItem->description = 'إيراد من معاملة خزينة';
            $creditItem->save();
        } elseif ($transaction->type == 'withdrawal') {
            // Credit cash account
            $creditItem = new \App\Models\JournalEntryItem();
            $creditItem->journal_entry_id = $entry->id;
            $creditItem->account_id = $cashAccount->id;
            $creditItem->debit = 0;
            $creditItem->credit = $transaction->amount;
            $creditItem->description = 'سحب من الصندوق';
            $creditItem->save();

            // Debit expense account (using revenue for now)
            $debitItem = new \App\Models\JournalEntryItem();
            $debitItem->journal_entry_id = $entry->id;
            $debitItem->account_id = $revenueAccount->id;
            $debitItem->debit = $transaction->amount;
            $debitItem->credit = 0;
            $debitItem->description = 'مصروف من معاملة خزينة';
            $debitItem->save();
        }

        // Create record in cash_transaction_entries table
        DB::table('cash_transaction_entries')->insert([
            'cash_transaction_id' => $transaction->id,
            'journal_entry_id' => $entry->id,
            'created_at' => now(),
            'updated_at' => now()
        ]);

        $createdCount++;
        echo "Successfully created journal entry for cash transaction ID: {$transaction->id}\n";
    } catch (\Exception $e) {
        echo "Error creating journal entry for transaction ID {$transaction->id}: {$e->getMessage()}\n";
        echo "Error details: {$e->getTraceAsString()}\n";
    }
}

echo "\nFix completed. Fixed {$createdCount} relationships.\n";
