@extends('layouts.app')

@section('title', 'إنشاء فاتورة مشتريات جديدة')

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">إنشاء فاتورة مشتريات جديدة</h3>
                </div>
                
                <form action="{{ route('inventory.purchase-invoices.store') }}" method="POST" enctype="multipart/form-data">
                    @csrf
                    
                    <div class="card-body pb-0">
                        <!-- بيانات الفاتورة الأساسية - الصف الأول -->
                        <div class="row">
                            <div class="col-md-3">
                                <div class="form-group">
                                    <label for="invoice_number">رقم الفاتورة <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control @error('invoice_number') is-invalid @enderror" id="invoice_number" name="invoice_number" value="{{ old('invoice_number') }}" required>
                                    @error('invoice_number')
                                        <span class="invalid-feedback">{{ $message }}</span>
                                    @enderror
                                </div>
                            </div>
                            
                            <div class="col-md-3">
                                <div class="form-group">
                                    <label for="supplier_invoice_number">رقم فاتورة المورد</label>
                                    <input type="text" class="form-control @error('supplier_invoice_number') is-invalid @enderror" id="supplier_invoice_number" name="supplier_invoice_number" value="{{ old('supplier_invoice_number') }}">
                                    @error('supplier_invoice_number')
                                        <span class="invalid-feedback">{{ $message }}</span>
                                    @enderror
                                </div>
                            </div>
                            
                            <div class="col-md-3">
                                <div class="form-group">
                                    <label for="invoice_date">تاريخ الفاتورة <span class="text-danger">*</span></label>
                                    <input type="date" class="form-control @error('invoice_date') is-invalid @enderror" id="invoice_date" name="invoice_date" value="{{ old('invoice_date', date('Y-m-d')) }}" required>
                                    @error('invoice_date')
                                        <span class="invalid-feedback">{{ $message }}</span>
                                    @enderror
                                </div>
                            </div>
                            
                            <div class="col-md-3">
                                <div class="form-group">
                                    <label for="due_date">تاريخ الاستحقاق</label>
                                    <input type="date" class="form-control @error('due_date') is-invalid @enderror" id="due_date" name="due_date" value="{{ old('due_date') }}">
                                    @error('due_date')
                                        <span class="invalid-feedback">{{ $message }}</span>
                                    @enderror
                                </div>
                            </div>
                        </div>
                        
                        <!-- بيانات المورد والمستودع - الصف الثاني -->
                        <div class="row">
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="supplier_id">المورد <span class="text-danger">*</span></label>
                                    <select class="form-control select2 @error('supplier_id') is-invalid @enderror" id="supplier_id" name="supplier_id" required>
                                        <option value="">-- اختر المورد --</option>
                                        @foreach($suppliers as $supplier)
                                            <option value="{{ $supplier->id }}" {{ old('supplier_id') == $supplier->id ? 'selected' : '' }}>
                                                {{ $supplier->name }}
                                            </option>
                                        @endforeach
                                    </select>
                                    @error('supplier_id')
                                        <span class="invalid-feedback">{{ $message }}</span>
                                    @enderror
                                </div>
                            </div>
                            
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="warehouse_id">المستودع <span class="text-danger">*</span></label>
                                    <select class="form-control select2 @error('warehouse_id') is-invalid @enderror" id="warehouse_id" name="warehouse_id" required>
                                        <option value="">-- اختر المستودع --</option>
                                        @foreach($warehouses as $warehouse)
                                            <option value="{{ $warehouse->id }}" {{ old('warehouse_id') == $warehouse->id ? 'selected' : '' }}>
                                                {{ $warehouse->name }}
                                            </option>
                                        @endforeach
                                    </select>
                                    @error('warehouse_id')
                                        <span class="invalid-feedback">{{ $message }}</span>
                                    @enderror
                                </div>
                            </div>
                            
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="purchase_order_id">أمر الشراء المرتبط</label>
                                    <select class="form-control select2 @error('purchase_order_id') is-invalid @enderror" id="purchase_order_id" name="purchase_order_id">
                                        <option value="">-- اختر أمر الشراء --</option>
                                        @foreach($purchaseOrders as $order)
                                            <option value="{{ $order->id }}" {{ old('purchase_order_id') == $order->id ? 'selected' : '' }}>
                                                {{ $order->order_number }} - {{ $order->supplier->name }}
                                            </option>
                                        @endforeach
                                    </select>
                                    @error('purchase_order_id')
                                        <span class="invalid-feedback">{{ $message }}</span>
                                    @enderror
                                </div>
                            </div>
                        </div>
                        
                        <!-- المرفقات والملاحظات - الصف الثالث -->
                        <div class="row">
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="attachment">مرفق الفاتورة</label>
                                    <div class="input-group">
                                        <div class="custom-file">
                                            <input type="file" class="custom-file-input @error('attachment') is-invalid @enderror" id="attachment" name="attachment">
                                            <label class="custom-file-label" for="attachment">اختر ملف</label>
                                        </div>
                                    </div>
                                    @error('attachment')
                                        <span class="text-danger small">{{ $message }}</span>
                                    @enderror
                                </div>
                            </div>
                            
                            <div class="col-md-8">
                                <div class="form-group">
                                    <label for="notes">ملاحظات</label>
                                    <input type="text" class="form-control @error('notes') is-invalid @enderror" id="notes" name="notes" value="{{ old('notes') }}">
                                    @error('notes')
                                        <span class="invalid-feedback">{{ $message }}</span>
                                    @enderror
                                </div>
                            </div>
                        </div>
                        
                        <!-- عناصر الفاتورة والإجماليات -->
                        <div class="row mt-3">
                            <div class="col-md-8">
                                <h5 class="mb-3">عناصر الفاتورة</h5>
                                <div class="table-responsive">
                                    <table class="table table-bordered table-striped" id="items-table">
                                        <thead class="thead-light">
                                            <tr>
                                                <th width="30%">العنصر</th>
                                                <th width="10%">الكمية</th>
                                                <th width="15%">سعر الوحدة</th>
                                                <th width="10%">الضريبة %</th>
                                                <th width="10%">الخصم %</th>
                                                <th width="15%">الإجمالي</th>
                                                <th width="10%"></th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <tr class="item-row">
                                                <td>
                                                    <select class="form-control select2 item-select" name="items[0][item_id]" required>
                                                        <option value="">-- اختر العنصر --</option>
                                                        @foreach($items as $item)
                                                            <option value="{{ $item->id }}" data-price="{{ $item->purchase_price }}">
                                                                {{ $item->name }} ({{ $item->sku }})
                                                            </option>
                                                        @endforeach
                                                    </select>
                                                    <input type="hidden" name="items[0][purchase_order_item_id]" value="">
                                                </td>
                                                <td>
                                                    <input type="number" class="form-control item-quantity" name="items[0][quantity]" min="0.01" step="0.01" value="1" required>
                                                </td>
                                                <td>
                                                    <input type="number" class="form-control item-price" name="items[0][unit_price]" min="0" step="0.01" value="0" required>
                                                </td>
                                                <td>
                                                    <input type="number" class="form-control item-tax-rate" name="items[0][tax_rate]" min="0" step="0.01" value="0">
                                                    <input type="hidden" class="item-tax-amount" name="items[0][tax_amount]" value="0">
                                                </td>
                                                <td>
                                                    <input type="number" class="form-control item-discount-rate" name="items[0][discount_rate]" min="0" step="0.01" value="0">
                                                    <input type="hidden" class="item-discount-amount" name="items[0][discount_amount]" value="0">
                                                </td>
                                                <td>
                                                    <input type="number" class="form-control item-total" name="items[0][total_price]" min="0" step="0.01" value="0" readonly>
                                                </td>
                                                <td class="text-center">
                                                    <button type="button" class="btn btn-danger btn-sm remove-item">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </td>
                                            </tr>
                                        </tbody>
                                        <tfoot>
                                            <tr>
                                                <td colspan="7">
                                                    <button type="button" class="btn btn-success btn-sm" id="add-item">
                                                        <i class="fas fa-plus"></i> إضافة عنصر
                                                    </button>
                                                </td>
                                            </tr>
                                        </tfoot>
                                    </table>
                                </div>
                            </div>
                            
                            <div class="col-md-4">
                                <div class="card">
                                    <div class="card-header bg-light">
                                        <h5 class="mb-0">ملخص الفاتورة</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="form-group row">
                                            <label class="col-sm-6 col-form-label">المجموع الفرعي</label>
                                            <div class="col-sm-6">
                                                <input type="number" class="form-control text-left" id="subtotal" name="subtotal" min="0" step="0.01" value="0" readonly>
                                            </div>
                                        </div>
                                        <div class="form-group row">
                                            <label class="col-sm-6 col-form-label">إجمالي الضريبة</label>
                                            <div class="col-sm-6">
                                                <input type="number" class="form-control text-left" id="tax_amount" name="tax_amount" min="0" step="0.01" value="0" readonly>
                                            </div>
                                        </div>
                                        <div class="form-group row">
                                            <label class="col-sm-6 col-form-label">إجمالي الخصم</label>
                                            <div class="col-sm-6">
                                                <input type="number" class="form-control text-left" id="discount_amount" name="discount_amount" min="0" step="0.01" value="0" readonly>
                                            </div>
                                        </div>
                                        <hr>
                                        <div class="form-group row">
                                            <label class="col-sm-6 col-form-label font-weight-bold">الإجمالي النهائي</label>
                                            <div class="col-sm-6">
                                                <input type="number" class="form-control text-left font-weight-bold" id="total_amount" name="total_amount" min="0" step="0.01" value="0" readonly>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="card-footer">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save ml-1"></i> حفظ الفاتورة
                        </button>
                        <a href="{{ route('inventory.purchase-invoices.index') }}" class="btn btn-secondary">
                            <i class="fas fa-times ml-1"></i> إلغاء
                        </a>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
    $(document).ready(function() {
        // تهيئة السيلكت2
        $('.select2').select2({
            width: '100%'
        });
        
        // تحديث المورد والمستودع عند اختيار أمر شراء
        $('#purchase_order_id').change(function() {
            const orderId = $(this).val();
            if (orderId) {
                $.ajax({
                    url: `/inventory/purchase-orders/${orderId}/details`,
                    type: 'GET',
                    dataType: 'json',
                    success: function(data) {
                        $('#supplier_id').val(data.supplier_id).trigger('change');
                        $('#warehouse_id').val(data.warehouse_id).trigger('change');
                        
                        // حذف جميع العناصر الحالية
                        $('.item-row:not(:first)').remove();
                        
                        // إعادة تعيين الصف الأول
                        const firstRow = $('.item-row:first');
                        firstRow.find('.item-select').val('').trigger('change');
                        firstRow.find('.item-quantity').val(1);
                        firstRow.find('.item-price').val(0);
                        firstRow.find('.item-tax-rate').val(0);
                        firstRow.find('.item-discount-rate').val(0);
                        
                        // إضافة عناصر أمر الشراء
                        if (data.items && data.items.length > 0) {
                            // استخدام الصف الأول للعنصر الأول
                            const firstItem = data.items[0];
                            firstRow.find('.item-select').val(firstItem.item_id).trigger('change');
                            firstRow.find('input[name="items[0][purchase_order_item_id]"]').val(firstItem.id);
                            firstRow.find('.item-quantity').val(firstItem.received_quantity);
                            firstRow.find('.item-price').val(firstItem.unit_price);
                            firstRow.find('.item-tax-rate').val(firstItem.tax_rate);
                            firstRow.find('.item-discount-rate').val(firstItem.discount_rate);
                            
                            // حساب إجمالي الصف الأول
                            calculateItemTotal(firstRow);
                            
                            // إضافة باقي العناصر
                            for (let i = 1; i < data.items.length; i++) {
                                const item = data.items[i];
                                addItemRow(item.item_id, item.id, item.received_quantity, item.unit_price, item.tax_rate, item.discount_rate);
                            }
                        }
                        
                        // تحديث الإجماليات
                        updateTotals();
                    }
                });
            }
        });
        
        // إضافة صف عنصر جديد
        $('#add-item').click(function() {
            addItemRow();
            // تحديث الإجماليات بعد إضافة صف جديد
            updateTotals();
        });
        
        // حذف صف عنصر
        $(document).on('click', '.remove-item', function() {
            if ($('.item-row').length > 1) {
                $(this).closest('tr').remove();
                updateRowIndices();
                // تحديث الإجماليات بعد حذف صف
                updateTotals();
            } else {
                alert('يجب أن تحتوي الفاتورة على عنصر واحد على الأقل');
            }
        });
        
        // تحديث الحسابات عند تغيير قيم العناصر
        $(document).on('change', '.item-select', function() {
            const row = $(this).closest('tr');
            const selectedOption = $(this).find('option:selected');
            const price = selectedOption.data('price') || 0;
            
            row.find('.item-price').val(price);
            calculateItemTotal(row);
        });
        
        // تحديث الحسابات عند تغيير الكمية أو السعر أو نسبة الضريبة أو نسبة الخصم
        $(document).on('input', '.item-quantity, .item-price, .item-tax-rate, .item-discount-rate', function() {
            calculateItemTotal($(this).closest('tr'));
        });
        
        // دالة إضافة صف عنصر جديد
        function addItemRow(itemId = '', purchaseOrderItemId = '', quantity = 1, price = 0, taxRate = 0, discountRate = 0) {
            const rowCount = $('.item-row').length;
            const newRow = $('.item-row:first').clone();
            
            // إعادة تعيين القيم
            newRow.find('.item-select').val(itemId);
            newRow.find('input[name^="items[0][purchase_order_item_id]"]').val(purchaseOrderItemId);
            newRow.find('.item-quantity').val(quantity);
            newRow.find('.item-price').val(price);
            newRow.find('.item-tax-rate').val(taxRate);
            newRow.find('.item-tax-amount').val(0);
            newRow.find('.item-discount-rate').val(discountRate);
            newRow.find('.item-discount-amount').val(0);
            newRow.find('.item-total').val(0);
            
            // تحديث أسماء الحقول
            updateRowInputNames(newRow, rowCount);
            
            // إضافة الصف الجديد
            $('#items-table tbody').append(newRow);
            
            // تهيئة السيلكت2 للصف الجديد
            newRow.find('.select2').select2({
                width: '100%'
            });
            
            // حساب إجمالي العنصر
            if (itemId) {
                // حساب الضريبة والخصم والإجمالي للصف الجديد
                const subtotal = quantity * price;
                const discountAmount = (subtotal * discountRate) / 100;
                const taxableAmount = subtotal - discountAmount;
                const taxAmount = (taxableAmount * taxRate) / 100;
                const total = taxableAmount + taxAmount;
                
                newRow.find('.item-tax-amount').val(taxAmount.toFixed(2));
                newRow.find('.item-discount-amount').val(discountAmount.toFixed(2));
                newRow.find('.item-total').val(total.toFixed(2));
            }
            
            calculateItemTotal(newRow);
            
            // تحديث الإجماليات الكلية بعد إضافة صف جديد
            updateTotals();
            
            return newRow;
        }
        
        // دالة تحديث أسماء حقول الإدخال في الصف
        function updateRowInputNames(row, index) {
            row.find('select[name^="items["]').attr('name', `items[${index}][item_id]`);
            row.find('input[name^="items["][name$="[purchase_order_item_id]"]').attr('name', `items[${index}][purchase_order_item_id]`);
            row.find('input[name^="items["][name$="[quantity]"]').attr('name', `items[${index}][quantity]`);
            row.find('input[name^="items["][name$="[unit_price]"]').attr('name', `items[${index}][unit_price]`);
            row.find('input[name^="items["][name$="[tax_rate]"]').attr('name', `items[${index}][tax_rate]`);
            row.find('input[name^="items["][name$="[tax_amount]"]').attr('name', `items[${index}][tax_amount]`);
            row.find('input[name^="items["][name$="[discount_rate]"]').attr('name', `items[${index}][discount_rate]`);
            row.find('input[name^="items["][name$="[discount_amount]"]').attr('name', `items[${index}][discount_amount]`);
            row.find('input[name^="items["][name$="[total_price]"]').attr('name', `items[${index}][total_price]`);
        }
        
        // دالة تحديث فهارس الصفوف بعد الحذف
        function updateRowIndices() {
            $('.item-row').each(function(index) {
                updateRowInputNames($(this), index);
            });
        }
        
        // دالة حساب إجمالي العنصر
        function calculateItemTotal(row) {
            const quantity = parseFloat(row.find('.item-quantity').val()) || 0;
            const price = parseFloat(row.find('.item-price').val()) || 0;
            const taxRate = parseFloat(row.find('.item-tax-rate').val()) || 0;
            const discountRate = parseFloat(row.find('.item-discount-rate').val()) || 0;
            
            const subtotal = quantity * price;
            const discountAmount = (subtotal * discountRate) / 100;
            const taxableAmount = subtotal - discountAmount;
            const taxAmount = (taxableAmount * taxRate) / 100;
            const total = taxableAmount + taxAmount;
            
            // تحديث قيم الحقول مع التقريب إلى رقمين عشريين
            row.find('.item-tax-amount').val(taxAmount.toFixed(2));
            row.find('.item-discount-amount').val(discountAmount.toFixed(2));
            row.find('.item-total').val(total.toFixed(2));
            
            // تحديث الإجماليات بعد تغيير قيم العنصر
            updateTotals();
        }
        
        // دالة تحديث الإجماليات
        function updateTotals() {
            let subtotal = 0;
            let totalTax = 0;
            let totalDiscount = 0;
            let grandTotal = 0;
            
            // حساب الإجماليات من جميع الصفوف
            $('.item-row').each(function() {
                const quantity = parseFloat($(this).find('.item-quantity').val()) || 0;
                const price = parseFloat($(this).find('.item-price').val()) || 0;
                const taxAmount = parseFloat($(this).find('.item-tax-amount').val()) || 0;
                const discountAmount = parseFloat($(this).find('.item-discount-amount').val()) || 0;
                
                // حساب المجموع الفرعي والإجماليات
                const rowSubtotal = quantity * price;
                subtotal += rowSubtotal;
                totalTax += taxAmount;
                totalDiscount += discountAmount;
                grandTotal += (rowSubtotal - discountAmount + taxAmount);
            });
            
            // تحديث حقول الإجماليات في النموذج
            $('#subtotal').val(subtotal.toFixed(2));
            $('#tax_amount').val(totalTax.toFixed(2));
            $('#discount_amount').val(totalDiscount.toFixed(2));
            $('#total_amount').val(grandTotal.toFixed(2));
        }
        
        // تهيئة أول صف وحساب الإجماليات عند تحميل الصفحة
        calculateItemTotal($('.item-row:first'));
        updateTotals();
        
        // إضافة مستمع للتأكد من تحديث الإجماليات بعد تحميل الصفحة بالكامل
        $(window).on('load', function() {
            setTimeout(function() {
                calculateItemTotal($('.item-row:first'));
                updateTotals();
            }, 500);
        });
        
        // تفعيل البوبوفر للمساعدة
        $('[data-toggle="popover"]').popover({
            trigger: 'hover',
            placement: 'top',
            html: true
        });
        
        // تفعيل اختيار الملف المرفق
        $('.custom-file-input').on('change', function() {
            let fileName = $(this).val().split('\\').pop();
            $(this).next('.custom-file-label').addClass("selected").html(fileName || 'اختر ملف');
        });
    });
</script>
@endpush