@extends('layouts.modern')

@section('title', 'Low Stock Alerts')

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-md-12">
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">Low Stock Alerts</h3>
                    <div class="card-tools">
                        <div class="btn-group">
                            <a href="{{ route('inventory.alerts.settings') }}" class="btn btn-default">
                                <i class="fas fa-cog"></i> Alert Settings
                            </a>
                            @if($alertsEnabled)
                                <form action="{{ route('inventory.alerts.send-low-stock') }}" method="POST" class="d-inline">
                                    @csrf
                                    <button type="submit" class="btn btn-primary">
                                        <i class="fas fa-envelope"></i> Send Alerts
                                    </button>
                                </form>
                            @endif
                        </div>
                    </div>
                </div>
                <div class="card-body">
                    @if(session('success'))
                        <div class="alert alert-success">{{ session('success') }}</div>
                    @endif
                    @if(session('error'))
                        <div class="alert alert-danger">{{ session('error') }}</div>
                    @endif
                    @if(session('info'))
                        <div class="alert alert-info">{{ session('info') }}</div>
                    @endif

                    @if(!$alertsEnabled)
                        <div class="alert alert-warning">
                            <i class="fas fa-exclamation-triangle"></i> Low stock alerts are currently disabled. 
                            <a href="{{ route('inventory.alerts.settings') }}">Enable them in settings</a> to monitor your inventory levels.
                        </div>
                    @else
                        <div class="alert alert-info">
                            <i class="fas fa-info-circle"></i> Showing items that have fallen below {{ $threshold }}% of their minimum stock level.
                        </div>

                        @if($items->isEmpty())
                            <div class="alert alert-success">
                                <i class="fas fa-check-circle"></i> Good news! No items are currently below the low stock threshold.
                            </div>
                        @else
                            <div class="table-responsive">
                                <table class="table table-bordered table-striped">
                                    <thead>
                                        <tr>
                                            <th>Item</th>
                                            <th>SKU</th>
                                            <th>Category</th>
                                            <th>Current Stock</th>
                                            <th>Min. Stock Level</th>
                                            <th>Status</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        @foreach($items as $item)
                                            @php
                                                $percentage = ($item->available_stock / $item->min_stock_level) * 100;
                                                if ($percentage <= 5) {
                                                    $statusClass = 'danger';
                                                    $statusText = 'Critical';
                                                } elseif ($percentage <= 25) {
                                                    $statusClass = 'warning';
                                                    $statusText = 'Low';
                                                } else {
                                                    $statusClass = 'info';
                                                    $statusText = 'Warning';
                                                }
                                            @endphp
                                            <tr>
                                                <td>{{ $item->name }}</td>
                                                <td>{{ $item->sku }}</td>
                                                <td>{{ $item->category->name ?? 'N/A' }}</td>
                                                <td>{{ $item->available_stock }}</td>
                                                <td>{{ $item->min_stock_level }}</td>
                                                <td>
                                                    <span class="badge badge-{{ $statusClass }}">
                                                        {{ $statusText }} ({{ number_format($percentage, 1) }}%)
                                                    </span>
                                                </td>
                                                <td>
                                                    <div class="btn-group">
                                                        <a href="{{ route('inventory.items.show', $item->id) }}" class="btn btn-sm btn-info">
                                                            <i class="fas fa-eye"></i> View
                                                        </a>
                                                        <a href="{{ route('inventory.purchase-orders.create') }}?item_id={{ $item->id }}" class="btn btn-sm btn-success">
                                                            <i class="fas fa-shopping-cart"></i> Order
                                                        </a>
                                                    </div>
                                                </td>
                                            </tr>
                                        @endforeach
                                    </tbody>
                                </table>
                            </div>
                        @endif
                    @endif
                </div>
            </div>
        </div>
    </div>

    @if($alertsEnabled && !$items->isEmpty())
    <div class="row">
        <div class="col-md-12">
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">Stock Level Analysis</h3>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <canvas id="stockLevelChart" height="300"></canvas>
                        </div>
                        <div class="col-md-6">
                            <div class="info-box bg-danger">
                                <span class="info-box-icon"><i class="fas fa-exclamation-circle"></i></span>
                                <div class="info-box-content">
                                    <span class="info-box-text">Critical Items (0-5%)</span>
                                    <span class="info-box-number">{{ $items->filter(function($item) { return ($item->available_stock / $item->min_stock_level) * 100 <= 5; })->count() }}</span>
                                </div>
                            </div>
                            <div class="info-box bg-warning">
                                <span class="info-box-icon"><i class="fas fa-exclamation-triangle"></i></span>
                                <div class="info-box-content">
                                    <span class="info-box-text">Low Items (6-25%)</span>
                                    <span class="info-box-number">{{ $items->filter(function($item) { 
                                        $percentage = ($item->available_stock / $item->min_stock_level) * 100;
                                        return $percentage > 5 && $percentage <= 25; 
                                    })->count() }}</span>
                                </div>
                            </div>
                            <div class="info-box bg-info">
                                <span class="info-box-icon"><i class="fas fa-info-circle"></i></span>
                                <div class="info-box-content">
                                    <span class="info-box-text">Warning Items (26-{{ $threshold }}%)</span>
                                    <span class="info-box-number">{{ $items->filter(function($item) use ($threshold) { 
                                        $percentage = ($item->available_stock / $item->min_stock_level) * 100;
                                        return $percentage > 25 && $percentage <= $threshold; 
                                    })->count() }}</span>
                                </div>
                            </div>
                            <div class="alert alert-warning">
                                <h5><i class="fas fa-lightbulb"></i> Recommendation</h5>
                                <p>Consider creating purchase orders for the critical and low stock items to maintain adequate inventory levels.</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    @endif
</div>

@if($alertsEnabled && !$items->isEmpty())
@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
    $(document).ready(function() {
        // Prepare data for chart
        var criticalCount = {{ $items->filter(function($item) { return ($item->available_stock / $item->min_stock_level) * 100 <= 5; })->count() }};
        var lowCount = {{ $items->filter(function($item) { 
            $percentage = ($item->available_stock / $item->min_stock_level) * 100;
            return $percentage > 5 && $percentage <= 25; 
        })->count() }};
        var warningCount = {{ $items->filter(function($item) use ($threshold) { 
            $percentage = ($item->available_stock / $item->min_stock_level) * 100;
            return $percentage > 25 && $percentage <= $threshold; 
        })->count() }};
        
        // Create chart
        var ctx = document.getElementById('stockLevelChart').getContext('2d');
        var stockLevelChart = new Chart(ctx, {
            type: 'pie',
            data: {
                labels: ['Critical (0-5%)', 'Low (6-25%)', 'Warning (26-{{ $threshold }}%)'],
                datasets: [{
                    data: [criticalCount, lowCount, warningCount],
                    backgroundColor: ['#dc3545', '#ffc107', '#17a2b8'],
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        position: 'bottom',
                    },
                    title: {
                        display: true,
                        text: 'Low Stock Items by Severity'
                    }
                }
            }
        });
    });
</script>
@endpush
@endif
@endsection