@extends('layouts.modern')

@section('title', 'Expiry Alerts')

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-md-12">
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">Expiry Alerts</h3>
                    <div class="card-tools">
                        <div class="btn-group">
                            <a href="{{ route('inventory.alerts.settings') }}" class="btn btn-default">
                                <i class="fas fa-cog"></i> Alert Settings
                            </a>
                            @if($alertsEnabled)
                                <form action="{{ route('inventory.alerts.send-expiry') }}" method="POST" class="d-inline">
                                    @csrf
                                    <button type="submit" class="btn btn-primary">
                                        <i class="fas fa-envelope"></i> Send Alerts
                                    </button>
                                </form>
                            @endif
                        </div>
                    </div>
                </div>
                <div class="card-body">
                    @if(session('success'))
                        <div class="alert alert-success">{{ session('success') }}</div>
                    @endif
                    @if(session('error'))
                        <div class="alert alert-danger">{{ session('error') }}</div>
                    @endif
                    @if(session('info'))
                        <div class="alert alert-info">{{ session('info') }}</div>
                    @endif

                    @if(!$alertsEnabled)
                        <div class="alert alert-warning">
                            <i class="fas fa-exclamation-triangle"></i> Expiry alerts are currently disabled. 
                            <a href="{{ route('inventory.alerts.settings') }}">Enable them in settings</a> to monitor your inventory expiry dates.
                        </div>
                    @else
                        <div class="alert alert-info">
                            <i class="fas fa-info-circle"></i> Showing items that will expire within the next {{ $thresholdDays }} days.
                        </div>

                        @if($batches->isEmpty())
                            <div class="alert alert-success">
                                <i class="fas fa-check-circle"></i> Good news! No items are set to expire within the next {{ $thresholdDays }} days.
                            </div>
                        @else
                            <div class="table-responsive">
                                <table class="table table-bordered table-striped">
                                    <thead>
                                        <tr>
                                            <th>Item</th>
                                            <th>SKU</th>
                                            <th>Batch Number</th>
                                            <th>Warehouse</th>
                                            <th>Quantity</th>
                                            <th>Expiry Date</th>
                                            <th>Days Left</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        @foreach($batches as $batch)
                                            @php
                                                $daysRemaining = now()->diffInDays($batch->expiry_date);
                                                if ($daysRemaining <= 7) {
                                                    $statusClass = 'danger';
                                                } elseif ($daysRemaining <= 14) {
                                                    $statusClass = 'warning';
                                                } else {
                                                    $statusClass = 'info';
                                                }
                                            @endphp
                                            <tr>
                                                <td>{{ $batch->item->name }}</td>
                                                <td>{{ $batch->item->sku }}</td>
                                                <td>{{ $batch->batch_number }}</td>
                                                <td>{{ $batch->warehouse->name }}</td>
                                                <td>{{ $batch->quantity }}</td>
                                                <td>{{ $batch->expiry_date->format('Y-m-d') }}</td>
                                                <td>
                                                    <span class="badge badge-{{ $statusClass }}">
                                                        {{ $daysRemaining }} days
                                                    </span>
                                                </td>
                                                <td>
                                                    <div class="btn-group">
                                                        <a href="{{ route('inventory.items.show', $batch->item_id) }}" class="btn btn-sm btn-info">
                                                            <i class="fas fa-eye"></i> View Item
                                                        </a>
                                                        <a href="{{ route('inventory.batches.show', $batch->id) }}" class="btn btn-sm btn-primary">
                                                            <i class="fas fa-box"></i> View Batch
                                                        </a>
                                                    </div>
                                                </td>
                                            </tr>
                                        @endforeach
                                    </tbody>
                                </table>
                            </div>
                        @endif
                    @endif
                </div>
            </div>
        </div>
    </div>

    @if($alertsEnabled && !$batches->isEmpty())
    <div class="row">
        <div class="col-md-12">
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">Expiry Analysis</h3>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <canvas id="expiryChart" height="300"></canvas>
                        </div>
                        <div class="col-md-6">
                            <div class="info-box bg-danger">
                                <span class="info-box-icon"><i class="fas fa-exclamation-circle"></i></span>
                                <div class="info-box-content">
                                    <span class="info-box-text">Critical (0-7 days)</span>
                                    <span class="info-box-number">{{ $batches->filter(function($batch) { 
                                        return now()->diffInDays($batch->expiry_date) <= 7; 
                                    })->count() }}</span>
                                </div>
                            </div>
                            <div class="info-box bg-warning">
                                <span class="info-box-icon"><i class="fas fa-exclamation-triangle"></i></span>
                                <div class="info-box-content">
                                    <span class="info-box-text">Warning (8-14 days)</span>
                                    <span class="info-box-number">{{ $batches->filter(function($batch) { 
                                        $days = now()->diffInDays($batch->expiry_date);
                                        return $days > 7 && $days <= 14; 
                                    })->count() }}</span>
                                </div>
                            </div>
                            <div class="info-box bg-info">
                                <span class="info-box-icon"><i class="fas fa-info-circle"></i></span>
                                <div class="info-box-content">
                                    <span class="info-box-text">Upcoming (15-{{ $thresholdDays }} days)</span>
                                    <span class="info-box-number">{{ $batches->filter(function($batch) use ($thresholdDays) { 
                                        $days = now()->diffInDays($batch->expiry_date);
                                        return $days > 14 && $days <= $thresholdDays; 
                                    })->count() }}</span>
                                </div>
                            </div>
                            <div class="alert alert-warning">
                                <h5><i class="fas fa-lightbulb"></i> Recommendations</h5>
                                <ul>
                                    <li>Use items with the earliest expiry dates first</li>
                                    <li>Consider offering discounts for items approaching expiry</li>
                                    <li>Check if items can be returned to suppliers</li>
                                    <li>Plan for proper disposal if necessary</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    @endif
</div>

@if($alertsEnabled && !$batches->isEmpty())
@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
    $(document).ready(function() {
        // Prepare data for chart
        var criticalCount = {{ $batches->filter(function($batch) { 
            return now()->diffInDays($batch->expiry_date) <= 7; 
        })->count() }};
        
        var warningCount = {{ $batches->filter(function($batch) { 
            $days = now()->diffInDays($batch->expiry_date);
            return $days > 7 && $days <= 14; 
        })->count() }};
        
        var upcomingCount = {{ $batches->filter(function($batch) use ($thresholdDays) { 
            $days = now()->diffInDays($batch->expiry_date);
            return $days > 14 && $days <= $thresholdDays; 
        })->count() }};
        
        // Create chart
        var ctx = document.getElementById('expiryChart').getContext('2d');
        var expiryChart = new Chart(ctx, {
            type: 'pie',
            data: {
                labels: ['Critical (0-7 days)', 'Warning (8-14 days)', 'Upcoming (15-{{ $thresholdDays }} days)'],
                datasets: [{
                    data: [criticalCount, warningCount, upcomingCount],
                    backgroundColor: ['#dc3545', '#ffc107', '#17a2b8'],
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        position: 'bottom',
                    },
                    title: {
                        display: true,
                        text: 'Expiring Items by Timeframe'
                    }
                }
            }
        });
    });
</script>
@endpush
@endif
@endsection