@extends('layouts.app')

@section('title', 'تحويل بنكي جديد')

@section('content')
<div class="container-fluid">
    <div class="row justify-content-center">
        <div class="col-md-12">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h3 class="mb-0">تحويل بنكي جديد</h3>
                    <a href="{{ route('bank-transactions.index') }}" class="btn btn-secondary">
                        <i class="fas fa-arrow-right"></i> العودة للقائمة
                    </a>
                </div>

                <div class="card-body">
                    @if(session('error'))
                        <div class="alert alert-danger">
                            {{ session('error') }}
                        </div>
                    @endif

                    <form action="{{ route('bank-transactions.store-transfer') }}" method="POST" enctype="multipart/form-data">
                        @csrf
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="from_account_id">من الحساب <span class="text-danger">*</span></label>
                                    <select class="form-control @error('from_account_id') is-invalid @enderror" id="from_account_id" name="from_account_id" required>
                                        <option value="">-- اختر الحساب المصدر --</option>
                                        @foreach($bankAccounts as $account)
                                            <option value="{{ $account->id }}" {{ old('from_account_id', request('from_account_id')) == $account->id ? 'selected' : '' }}
                                                data-currency="{{ $account->currency }}" data-balance="{{ $account->current_balance }}">
                                                {{ $account->bank->name }} - {{ $account->account_number }} ({{ $account->currency }}) - الرصيد: {{ number_format($account->current_balance, 2) }}
                                            </option>
                                        @endforeach
                                    </select>
                                    <small class="form-text text-muted" id="from-balance-info"></small>
                                    @error('from_account_id')
                                        <span class="invalid-feedback" role="alert">
                                            <strong>{{ $message }}</strong>
                                        </span>
                                    @enderror
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="to_account_id">إلى الحساب <span class="text-danger">*</span></label>
                                    <select class="form-control @error('to_account_id') is-invalid @enderror" id="to_account_id" name="to_account_id" required>
                                        <option value="">-- اختر الحساب الهدف --</option>
                                        @foreach($bankAccounts as $account)
                                            <option value="{{ $account->id }}" {{ old('to_account_id', request('to_account_id')) == $account->id ? 'selected' : '' }}
                                                data-currency="{{ $account->currency }}">
                                                {{ $account->bank->name }} - {{ $account->account_number }} ({{ $account->currency }})
                                            </option>
                                        @endforeach
                                    </select>
                                    <small class="form-text text-muted" id="currency-warning" style="display: none; color: red;">
                                        تنبيه: العملات مختلفة بين الحسابين. سيتم تطبيق سعر الصرف.
                                    </small>
                                    @error('to_account_id')
                                        <span class="invalid-feedback" role="alert">
                                            <strong>{{ $message }}</strong>
                                        </span>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="amount">المبلغ <span class="text-danger">*</span></label>
                                    <div class="input-group">
                                        <input type="number" step="0.01" min="0.01" class="form-control @error('amount') is-invalid @enderror" id="amount" name="amount" value="{{ old('amount') }}" required>
                                        <div class="input-group-append">
                                            <span class="input-group-text" id="from-currency-addon">العملة</span>
                                        </div>
                                    </div>
                                    @error('amount')
                                        <span class="invalid-feedback" role="alert">
                                            <strong>{{ $message }}</strong>
                                        </span>
                                    @enderror
                                </div>
                            </div>
                            <div class="col-md-6" id="exchange-rate-container" style="display: none;">
                                <div class="form-group">
                                    <label for="exchange_rate">سعر الصرف <span class="text-danger">*</span></label>
                                    <div class="input-group">
                                        <input type="number" step="0.0001" min="0.0001" class="form-control @error('exchange_rate') is-invalid @enderror" id="exchange_rate" name="exchange_rate" value="{{ old('exchange_rate', 1) }}">
                                        <div class="input-group-append">
                                            <span class="input-group-text" id="exchange-rate-text">1:1</span>
                                        </div>
                                    </div>
                                    @error('exchange_rate')
                                        <span class="invalid-feedback" role="alert">
                                            <strong>{{ $message }}</strong>
                                        </span>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row" id="converted-amount-container" style="display: none;">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="converted_amount">المبلغ المحول (بعد التحويل)</label>
                                    <div class="input-group">
                                        <input type="number" step="0.01" class="form-control" id="converted_amount" name="converted_amount" value="{{ old('converted_amount') }}" readonly>
                                        <div class="input-group-append">
                                            <span class="input-group-text" id="to-currency-addon">العملة</span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="transaction_date">تاريخ التحويل <span class="text-danger">*</span></label>
                                    <input type="date" class="form-control @error('transaction_date') is-invalid @enderror" id="transaction_date" name="transaction_date" value="{{ old('transaction_date', date('Y-m-d')) }}" required>
                                    @error('transaction_date')
                                        <span class="invalid-feedback" role="alert">
                                            <strong>{{ $message }}</strong>
                                        </span>
                                    @enderror
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="reference_number">رقم المرجع</label>
                                    <input type="text" class="form-control @error('reference_number') is-invalid @enderror" id="reference_number" name="reference_number" value="{{ old('reference_number') }}">
                                    @error('reference_number')
                                        <span class="invalid-feedback" role="alert">
                                            <strong>{{ $message }}</strong>
                                        </span>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="transfer_method">طريقة التحويل <span class="text-danger">*</span></label>
                                    <select class="form-control @error('transfer_method') is-invalid @enderror" id="transfer_method" name="transfer_method" required>
                                        <option value="">-- اختر طريقة التحويل --</option>
                                        <option value="online" {{ old('transfer_method') == 'online' ? 'selected' : '' }}>تحويل إلكتروني</option>
                                        <option value="wire" {{ old('transfer_method') == 'wire' ? 'selected' : '' }}>تحويل بنكي</option>
                                        <option value="cheque" {{ old('transfer_method') == 'cheque' ? 'selected' : '' }}>شيك</option>
                                        <option value="internal" {{ old('transfer_method') == 'internal' ? 'selected' : '' }}>تحويل داخلي</option>
                                        <option value="other" {{ old('transfer_method') == 'other' ? 'selected' : '' }}>أخرى</option>
                                    </select>
                                    @error('transfer_method')
                                        <span class="invalid-feedback" role="alert">
                                            <strong>{{ $message }}</strong>
                                        </span>
                                    @enderror
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="transfer_fees">رسوم التحويل</label>
                                    <div class="input-group">
                                        <input type="number" step="0.01" min="0" class="form-control @error('transfer_fees') is-invalid @enderror" id="transfer_fees" name="transfer_fees" value="{{ old('transfer_fees', 0) }}">
                                        <div class="input-group-append">
                                            <span class="input-group-text" id="fees-currency-addon">العملة</span>
                                        </div>
                                    </div>
                                    @error('transfer_fees')
                                        <span class="invalid-feedback" role="alert">
                                            <strong>{{ $message }}</strong>
                                        </span>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-12">
                                <div class="form-group">
                                    <label for="description">الوصف <span class="text-danger">*</span></label>
                                    <textarea class="form-control @error('description') is-invalid @enderror" id="description" name="description" rows="3" required>{{ old('description') }}</textarea>
                                    @error('description')
                                        <span class="invalid-feedback" role="alert">
                                            <strong>{{ $message }}</strong>
                                        </span>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="attachment">المرفقات</label>
                                    <input type="file" class="form-control-file @error('attachment') is-invalid @enderror" id="attachment" name="attachment">
                                    <small class="form-text text-muted">يمكنك إرفاق صورة أو مستند للتحويل (PDF, JPG, PNG)</small>
                                    @error('attachment')
                                        <span class="invalid-feedback" role="alert">
                                            <strong>{{ $message }}</strong>
                                        </span>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="form-group text-center mt-4">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-exchange-alt"></i> إجراء التحويل
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
    $(document).ready(function() {
        // تحديث معلومات العملة والرصيد
        function updateCurrencyInfo() {
            var fromAccount = $('#from_account_id').find(':selected');
            var toAccount = $('#to_account_id').find(':selected');
            
            var fromCurrency = fromAccount.data('currency');
            var toCurrency = toAccount.data('currency');
            var fromBalance = fromAccount.data('balance');
            
            // تحديث عملة المبلغ
            $('#from-currency-addon').text(fromCurrency || 'العملة');
            $('#fees-currency-addon').text(fromCurrency || 'العملة');
            $('#to-currency-addon').text(toCurrency || 'العملة');
            
            // عرض معلومات الرصيد
            if (fromBalance !== undefined) {
                $('#from-balance-info').text('الرصيد المتاح: ' + fromBalance.toFixed(2) + ' ' + fromCurrency);
            } else {
                $('#from-balance-info').text('');
            }
            
            // التحقق من اختلاف العملات
            if (fromCurrency && toCurrency && fromCurrency !== toCurrency) {
                $('#currency-warning').show();
                $('#exchange-rate-container').show();
                $('#converted-amount-container').show();
                $('#exchange-rate-text').text('1 ' + fromCurrency + ' = ? ' + toCurrency);
            } else {
                $('#currency-warning').hide();
                $('#exchange-rate-container').hide();
                $('#converted-amount-container').hide();
            }
            
            // حساب المبلغ المحول
            calculateConvertedAmount();
        }
        
        // حساب المبلغ المحول بعد تطبيق سعر الصرف
        function calculateConvertedAmount() {
            var amount = parseFloat($('#amount').val()) || 0;
            var exchangeRate = parseFloat($('#exchange_rate').val()) || 1;
            var convertedAmount = amount * exchangeRate;
            
            $('#converted_amount').val(convertedAmount.toFixed(2));
        }
        
        // تحديث المعلومات عند تغيير الحسابات
        $('#from_account_id, #to_account_id').change(function() {
            updateCurrencyInfo();
        });
        
        // تحديث المبلغ المحول عند تغيير المبلغ أو سعر الصرف
        $('#amount, #exchange_rate').on('input', function() {
            calculateConvertedAmount();
        });
        
        // تحديث المعلومات عند تحميل الصفحة
        updateCurrencyInfo();
        
        // التحقق من صحة البيانات قبل الإرسال
        $('form').submit(function(e) {
            var fromAccount = $('#from_account_id').val();
            var toAccount = $('#to_account_id').val();
            var amount = parseFloat($('#amount').val());
            var fromBalance = parseFloat($('#from_account_id').find(':selected').data('balance'));
            
            // التحقق من اختيار حسابين مختلفين
            if (fromAccount === toAccount) {
                e.preventDefault();
                alert('لا يمكن التحويل إلى نفس الحساب!');
                return false;
            }
            
            // التحقق من كفاية الرصيد
            if (amount > fromBalance) {
                e.preventDefault();
                alert('المبلغ المطلوب تحويله أكبر من الرصيد المتاح في الحساب المصدر!');
                return false;
            }
        });
    });
</script>
@endsection