<?php

// Cargar la aplicación
require __DIR__ . '/vendor/autoload.php';
$app = require_once __DIR__ . '/bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Http\Kernel::class);
$response = $kernel->handle(
    $request = Illuminate\Http\Request::capture()
);

// Obtener el contenido actual del modelo CustomerStatement
$modelPath = __DIR__ . '/app/Models/CustomerStatement.php';
$currentContent = file_get_contents($modelPath);

// Crear una copia de seguridad del archivo original
$backupPath = $modelPath . '.backup.' . date('Y-m-d-H-i-s');
file_put_contents($backupPath, $currentContent);
echo "Se ha creado una copia de seguridad del modelo en: " . $backupPath . "\n";

// Mejorar el método createForJournalEntry para detectar mejor las cuentas de clientes
$improvedMethod = '
    /**
     * Create a customer statement record for a journal entry
     * 
     * @param JournalEntry $journalEntry
     * @return CustomerStatement|null
     */
    public static function createForJournalEntry(JournalEntry $journalEntry)
    {
        try {
            // أولاً، نحذف أي سجلات موجودة لهذا القيد المحاسبي
            self::where(\'reference_type\', \'App\\\\Models\\\\JournalEntry\')
                ->where(\'reference_id\', $journalEntry->id)
                ->delete();

            // Find customer account items in this journal entry
            $customerAccountItems = JournalEntryItem::where(\'journal_entry_id\', $journalEntry->id)
                ->whereHas(\'account\', function ($query) {
                    $query->where(\'account_type\', \'asset\')
                        ->where(\'sub_type\', \'accounts_receivable\');
                })
                ->get();

            // If no customer account items found, we can\'t create a statement
            if ($customerAccountItems->isEmpty()) {
                return null;
            }

            $statementsCreated = [];

            // For each customer account item, create a statement
            foreach ($customerAccountItems as $item) {
                // Get the customer account
                $customerAccount = ChartOfAccount::find($item->account_id);
                if (!$customerAccount) continue;

                // Extract customer ID from account code (AR-{customer_id})
                $accountCode = $customerAccount->account_code;
                $customerId = null;

                if (strpos($accountCode, \'AR-\') === 0) {
                    $customerId = intval(substr($accountCode, 3));
                }

                // If we couldn\'t extract customer ID, try to find by name
                if (!$customerId) {
                    // Extract customer name from account name (ذمم العميل: {customer_name})
                    $customerName = str_replace(\'ذمم العميل: \', \'\', $customerAccount->name);
                    $customerName = str_replace(\'Customer Receivable: \', \'\', $customerName);
                    
                    $customer = Customer::where(\'company_id\', $journalEntry->company_id)
                        ->where(function($query) use ($customerName) {
                            $query->where(\'name\', \'like\', \'%\' . $customerName . \'%\')
                                ->orWhere(\'name_en\', \'like\', \'%\' . $customerName . \'%\');
                        })
                        ->first();

                    if ($customer) {
                        $customerId = $customer->id;
                    } else {
                        // Try to find customer by searching all customers and comparing with account name
                        $allCustomers = Customer::where(\'company_id\', $journalEntry->company_id)->get();
                        foreach ($allCustomers as $potentialCustomer) {
                            if (stripos($customerAccount->name, $potentialCustomer->name) !== false || 
                                (isset($potentialCustomer->name_en) && stripos($customerAccount->name, $potentialCustomer->name_en) !== false)) {
                                $customerId = $potentialCustomer->id;
                                break;
                            }
                        }
                        
                        if (!$customerId) {
                            continue; // Skip if we can\'t find the customer
                        }
                    }
                } else {
                    // Verify customer exists
                    $customer = Customer::find($customerId);
                    if (!$customer) continue;
                }

                // Get the last balance for this customer
                $lastStatement = self::where(\'customer_id\', $customerId)
                    ->orderBy(\'id\', \'desc\')
                    ->first();

                $currentBalance = $lastStatement ? $lastStatement->balance : 0;

                // Calculate the new balance
                // If debit > 0, it increases the customer balance (customer owes more)
                // If credit > 0, it decreases the customer balance (customer paid)
                $debitAmount = $item->debit;
                $creditAmount = $item->credit;
                $netAmount = $creditAmount - $debitAmount;

                // If net amount is credit (positive), it reduces the customer balance
                // If net amount is debit (negative), it increases the customer balance
                $newBalance = $currentBalance - $netAmount;

                // Get reference information based on journal entry reference
                $referenceType = $journalEntry->reference_type;
                $referenceId = $journalEntry->reference_id;
                $description = $journalEntry->description;

                // If reference is an invoice, get more details
                if ($referenceType == \'App\\\\Models\\\\Invoice\' && $referenceId) {
                    $invoice = Invoice::find($referenceId);
                    if ($invoice) {
                        $description = \'فاتورة رقم \' . $invoice->invoice_number . \' (قيد محاسبي: \' . $journalEntry->entry_number . \')\';
                    }
                }
                // If reference is a payment, get more details
                else if ($referenceType == \'App\\\\Models\\\\Payment\' && $referenceId) {
                    $payment = Payment::find($referenceId);
                    if ($payment && $payment->invoice) {
                        $description = \'دفعة للفاتورة رقم \' . $payment->invoice->invoice_number . \' (قيد محاسبي: \' . $journalEntry->entry_number . \')\';
                    }
                }

                // Create the statement record
                $statement = self::create([
                    \'company_id\' => $journalEntry->company_id,
                    \'customer_id\' => $customerId,
                    \'branch_id\' => null, // We don\'t have branch info from journal entry
                    \'transaction_date\' => $journalEntry->entry_date,
                    \'reference_number\' => $journalEntry->entry_number,
                    \'reference_type\' => \'App\\\\Models\\\\JournalEntry\',
                    \'reference_id\' => $journalEntry->id,
                    \'description\' => $description,
                    \'debit\' => $debitAmount,
                    \'credit\' => $creditAmount,
                    \'balance\' => $newBalance
                ]);

                $statementsCreated[] = $statement;
            }

            return count($statementsCreated) > 0 ? $statementsCreated[0] : null;
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error(\'Error creating customer statement from journal entry: \' . $e->getMessage());
            return null;
        }

        return null;
    }';

// Reemplazar el método existente con el mejorado
$pattern = '/public static function createForJournalEntry\(JournalEntry \$journalEntry\).*?return null;\s*\}/s';
$newContent = preg_replace($pattern, $improvedMethod, $currentContent);

// Guardar el contenido modificado
file_put_contents($modelPath, $newContent);
echo "Se ha actualizado el modelo CustomerStatement con un método mejorado para detectar cuentas de clientes.\n";

// Crear un script para regenerar todos los estados de cuenta de clientes
$regenerateScript = __DIR__ . '/regenerate_customer_statements.php';
$scriptContent = '<?php

// Cargar la aplicación
require __DIR__ . \'/vendor/autoload.php\';
$app = require_once __DIR__ . \'/bootstrap/app.php\';
$kernel = $app->make(Illuminate\Contracts\Http\Kernel::class);
$response = $kernel->handle(
    $request = Illuminate\Http\Request::capture()
);

echo "Regenerando todos los estados de cuenta de clientes...\n";

// Eliminar todos los registros existentes en customer_statements
\App\Models\CustomerStatement::truncate();
echo "Se han eliminado todos los registros existentes en customer_statements.\n";

// Obtener todos los asientos contables publicados
$entries = \App\Models\JournalEntry::where(\'is_posted\', true)->get();
echo "Total de asientos contables publicados: " . $entries->count() . "\n";

$created = 0;
$skipped = 0;

foreach ($entries as $entry) {
    echo "Procesando asiento ID: " . $entry->id . " - Número: " . $entry->entry_number . "... ";
    
    $result = \App\Models\CustomerStatement::createForJournalEntry($entry);
    
    if ($result) {
        echo "OK\n";
        $created++;
    } else {
        echo "Omitido\n";
        $skipped++;
    }
}

echo "\nResumen:\n";
echo "Total de asientos procesados: " . $entries->count() . "\n";
echo "Estados de cuenta creados: " . $created . "\n";
echo "Asientos omitidos: " . $skipped . "\n";';

file_put_contents($regenerateScript, $scriptContent);
echo "Se ha creado un script para regenerar todos los estados de cuenta de clientes en: " . $regenerateScript . "\n";

echo "\nPara aplicar la solución completa, ejecute el siguiente comando:\n";
echo "php regenerate_customer_statements.php\n";
