<?php

/**
 * هذا الملف يقوم بتحسين استخراج معرف العميل في دالة createForJournalEntry
 * لتعتمد على معرف العميل (ID) بدلاً من رقم العميل
 */

require_once __DIR__ . '/vendor/autoload.php';

use Illuminate\Support\Facades\Log;

// مسار ملف النموذج
$modelPath = __DIR__ . '/app/Models/CustomerStatement.php';

// قراءة محتوى الملف الحالي
$currentContent = file_get_contents($modelPath);

// إنشاء نسخة احتياطية من الملف الأصلي
$backupPath = $modelPath . '.backup.' . date('Y-m-d-H-i-s');
file_put_contents($backupPath, $currentContent);
echo "تم إنشاء نسخة احتياطية من النموذج في: " . $backupPath . "\n";

// تحسين دالة createForJournalEntry لاستخدام معرف العميل (ID) بدلاً من رقم العميل
$improvedMethod = '
	/**
	 * Create a customer statement record for a journal entry
	 * 
	 * @param JournalEntry $journalEntry
	 * @return CustomerStatement|null
	 */
	public static function createForJournalEntry(JournalEntry $journalEntry)
	{
		try {
			// أولاً، نحذف أي سجلات موجودة لهذا القيد المحاسبي
			self::where(\'reference_type\', \'App\\Models\\JournalEntry\')
				->where(\'reference_id\', $journalEntry->id)
				->delete();

			// Find customer account items in this journal entry
			$customerAccountItems = JournalEntryItem::where(\'journal_entry_id\', $journalEntry->id)
				->whereHas(\'account\', function ($query) {
					$query->where(\'account_type\', \'asset\')
						->where(function ($q) {
							$q->where(\'sub_type\', \'accounts_receivable\')
								->orWhere(\'sub_type\', \'customer_receivable\');
						});
				})
				->get();

			// If no customer account items found, we can\'t create a statement
			if ($customerAccountItems->isEmpty()) {
				\Illuminate\Support\Facades\Log::info(\'لا توجد عناصر حساب عميل في القيد المحاسبي: \' . $journalEntry->id);
				return null;
			}

			$statementsCreated = [];

			// For each customer account item, create a statement
			foreach ($customerAccountItems as $item) {
				// Get the customer account
				$customerAccount = ChartOfAccount::find($item->account_id);
				if (!$customerAccount) {
					\Illuminate\Support\Facades\Log::warning(\'حساب العميل غير موجود: \' . $item->account_id);
					continue;
				}

				// Extract customer ID from account code (AR-{customer_id})
				$accountCode = $customerAccount->account_code;
				$customerId = null;

				// 1. أولاً، نحاول استخراج معرف العميل من اسم الحساب باستخدام التعبير النمطي
				if (preg_match(\'/عميل رقم(\\d+)/\', $customerAccount->name, $matches)) {
					// البحث عن العميل باستخدام الرقم المستخرج
					$customerNumber = intval($matches[1]);
					\Illuminate\Support\Facades\Log::info(\'استخراج رقم العميل من اسم الحساب: \' . $customerAccount->name . \' -> \' . $customerNumber);
					
					// البحث عن العميل باستخدام رقم العميل
					$customer = Customer::where(\'company_id\', $journalEntry->company_id)
						->where(\'customer_number\', $customerNumber)
						->first();
					
					if ($customer) {
						$customerId = $customer->id;
						\Illuminate\Support\Facades\Log::info(\'تم العثور على العميل برقم العميل: \' . $customerNumber . \' -> معرف العميل: \' . $customerId);
					}
				}
				
				// 2. إذا لم نجد العميل، نحاول استخراج معرف العميل من رمز الحساب
				if (!$customerId && strpos($accountCode, \'AR-\') === 0) {
					$extractedId = intval(substr($accountCode, 3));
					
					// التحقق مما إذا كان المعرف المستخرج هو معرف العميل أو رقم العميل
					$customerById = Customer::find($extractedId);
					if ($customerById) {
						$customerId = $customerById->id;
						\Illuminate\Support\Facades\Log::info(\'تم العثور على العميل بمعرف العميل المستخرج من رمز الحساب: \' . $accountCode . \' -> \' . $customerId);
					} else {
						// البحث عن العميل باستخدام رقم العميل
						$customerByNumber = Customer::where(\'company_id\', $journalEntry->company_id)
							->where(\'customer_number\', $extractedId)
							->first();
						
						if ($customerByNumber) {
							$customerId = $customerByNumber->id;
							\Illuminate\Support\Facades\Log::info(\'تم العثور على العميل برقم العميل المستخرج من رمز الحساب: \' . $accountCode . \' -> \' . $customerId);
						}
					}
				}

				// 3. إذا لم نجد العميل، نحاول البحث عن العميل باستخدام اسم الحساب
				if (!$customerId) {
					// Extract customer name from account name (ذمم العميل: {customer_name})
					$customerName = str_replace(\'ذمم العميل: \', \'\', $customerAccount->name);
					$customerName = str_replace(\'Customer Receivable: \', \'\', $customerName);

					$customer = Customer::where(\'company_id\', $journalEntry->company_id)
						->where(function ($query) use ($customerName) {
							$query->where(\'name\', \'like\', \'%\' . $customerName . \'%\')
								->orWhere(\'name_en\', \'like\', \'%\' . $customerName . \'%\');
						})
						->first();

					if ($customer) {
						$customerId = $customer->id;
						\Illuminate\Support\Facades\Log::info(\'تم العثور على العميل بالاسم: \' . $customerName . \' -> \' . $customerId);
					} else {
						// Try to find customer by searching all customers and comparing with account name
						$allCustomers = Customer::where(\'company_id\', $journalEntry->company_id)->get();
						foreach ($allCustomers as $potentialCustomer) {
							if (
								stripos($customerAccount->name, $potentialCustomer->name) !== false ||
								(isset($potentialCustomer->name_en) && stripos($customerAccount->name, $potentialCustomer->name_en) !== false)
							) {
								$customerId = $potentialCustomer->id;
								\Illuminate\Support\Facades\Log::info(\'تم العثور على العميل بمقارنة الاسم: \' . $customerAccount->name . \' -> \' . $customerId);
								break;
							}
						}

						if (!$customerId) {
							\Illuminate\Support\Facades\Log::warning(\'لم يتم العثور على العميل للحساب: \' . $customerAccount->name . \' (\' . $accountCode . \')\');
							continue; // Skip if we can\'t find the customer
						}
					}
				}

				// Get the last balance for this customer
				$lastStatement = self::where(\'customer_id\', $customerId)
					->orderBy(\'id\', \'desc\')
					->first();

				$currentBalance = $lastStatement ? $lastStatement->balance : 0;

				// Calculate the new balance
				// If debit > 0, it increases the customer balance (customer owes more)
				// If credit > 0, it decreases the customer balance (customer paid)
				$debitAmount = $item->debit;
				$creditAmount = $item->credit;
				$netAmount = $creditAmount - $debitAmount;

				// If net amount is credit (positive), it reduces the customer balance
				// If net amount is debit (negative), it increases the customer balance
				$newBalance = $currentBalance - $netAmount;

				// Get reference information based on journal entry reference
				$referenceType = $journalEntry->reference_type;
				$referenceId = $journalEntry->reference_id;
				$description = $journalEntry->description;

				// If reference is an invoice, get more details
				if ($referenceType == \'App\\Models\\Invoice\' && $referenceId) {
					$invoice = Invoice::find($referenceId);
					if ($invoice) {
						$description = \'فاتورة رقم \' . $invoice->invoice_number . \' (قيد محاسبي: \' . $journalEntry->entry_number . \')\';
					}
				}
				// If reference is a payment, get more details
				else if ($referenceType == \'App\\Models\\Payment\' && $referenceId) {
					$payment = Payment::find($referenceId);
					if ($payment && $payment->invoice) {
						$description = \'دفعة للفاتورة رقم \' . $payment->invoice->invoice_number . \' (قيد محاسبي: \' . $journalEntry->entry_number . \')\';
					}
				}

				// Create the statement record
				$statement = self::create([
					\'company_id\' => $journalEntry->company_id,
					\'customer_id\' => $customerId,
					\'branch_id\' => null, // We don\'t have branch info from journal entry
					\'transaction_date\' => $journalEntry->entry_date,
					\'reference_number\' => $journalEntry->entry_number,
					\'reference_type\' => \'App\\Models\\JournalEntry\',
					\'reference_id\' => $journalEntry->id,
					\'description\' => $description,
					\'debit\' => $debitAmount,
					\'credit\' => $creditAmount,
					\'balance\' => $newBalance
				]);

				\Illuminate\Support\Facades\Log::info(\'تم إنشاء كشف حساب للعميل: \' . $customerId . \' للقيد المحاسبي: \' . $journalEntry->id);
				$statementsCreated[] = $statement;
			}

			return count($statementsCreated) > 0 ? $statementsCreated[0] : null;
		} catch (\Exception $e) {
			\Illuminate\Support\Facades\Log::error(\'خطأ في إنشاء كشف حساب العميل من القيد المحاسبي: \' . $e->getMessage());
			return null;
		}
	}';

// استبدال الدالة الموجودة بالدالة المحسنة
$pattern = '/public static function createForJournalEntry\(JournalEntry \$journalEntry\).*?return null;\s*\}/s';
$newContent = preg_replace($pattern, $improvedMethod, $currentContent);

// حفظ المحتوى المعدل
file_put_contents($modelPath, $newContent);
echo "تم تحديث نموذج CustomerStatement بدالة محسنة لاستخراج معرف العميل.\n";

// إنشاء سكريبت لإعادة إنشاء كشوفات حساب العملاء لجميع القيود المحاسبية
echo "جاري إنشاء سكريبت لإعادة إنشاء كشوفات حساب العملاء...\n";

$regenerateScript = '<?php

/**
 * هذا السكريبت يقوم بإعادة إنشاء كشوفات حساب العملاء لجميع القيود المحاسبية
 * باستخدام الدالة المحسنة التي تعتمد على معرف العميل (ID)
 */

require_once __DIR__ . \'/vendor/autoload.php\';

use Illuminate\Support\Facades\Log;

// تهيئة Laravel
$app = require_once __DIR__ . \'/bootstrap/app.php\';
$app->make(Illuminate\Contracts\Console\Kernel::class)->bootstrap();

echo "بدء إعادة إنشاء كشوفات حساب العملاء لجميع القيود المحاسبية...\n";

// الحصول على جميع القيود المحاسبية المرحلة
$entries = \App\Models\JournalEntry::where(\'is_posted\', true)->get();

echo "تم العثور على " . $entries->count() . " قيد محاسبي مرحل.\n";

$created = 0;
$skipped = 0;

foreach ($entries as $entry) {
    echo "معالجة القيد المحاسبي رقم: " . $entry->id . " - رقم القيد: " . $entry->entry_number . "... ";
    
    // حذف أي سجلات موجودة لهذا القيد المحاسبي
    \App\Models\CustomerStatement::where(\'reference_type\', \'App\\Models\\JournalEntry\')
        ->where(\'reference_id\', $entry->id)
        ->delete();
    
    // إنشاء كشف حساب جديد
    $result = \App\Models\CustomerStatement::createForJournalEntry($entry);
    
    if ($result) {
        echo "تم بنجاح\n";
        $created++;
    } else {
        echo "تم تخطيه\n";
        $skipped++;
    }
}

echo "\nاكتملت العملية!\n";
echo "تم إنشاء كشوفات حساب لـ " . $created . " قيد محاسبي.\n";
echo "تم تخطي " . $skipped . " قيد محاسبي.\n";
';

// حفظ سكريبت إعادة الإنشاء
$regenerateScriptPath = __DIR__ . '/regenerate_customer_statements_improved.php';
file_put_contents($regenerateScriptPath, $regenerateScript);
echo "تم إنشاء سكريبت إعادة إنشاء كشوفات حساب العملاء في: " . $regenerateScriptPath . "\n";

echo "\nتم الانتهاء من التحسينات. يمكنك الآن تشغيل سكريبت إعادة إنشاء كشوفات حساب العملاء باستخدام الأمر التالي:\n";
echo "php regenerate_customer_statements_improved.php\n";
