<?php

require __DIR__ . '/vendor/autoload.php';

$app = require_once __DIR__ . '/bootstrap/app.php';
$app->make(Illuminate\Contracts\Console\Kernel::class)->bootstrap();

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;

// المسار الذي سيتم حفظ ملفات البذور فيه
$seedersPath = __DIR__ . '/database/seeders';

// إنشاء المجلد إذا لم يكن موجودًا
if (!file_exists($seedersPath)) {
    mkdir($seedersPath, 0755, true);
}

// قائمة الجداول الأساسية التي نريد إنشاء بذور لها
$coreTables = [
    'companies',
    'branches',
    'users',
    'roles',
    'permissions',
    'permission_role',
    'role_user',
    'user_permission',
    'departments',
    'job_titles',
    'warehouses',
    'inventory_categories',
    'payment_methods',
    'tax_settings',
    'tax_brackets'
];

// إنشاء ملف DatabaseSeeder الرئيسي
createDatabaseSeeder($coreTables);

// إنشاء ملفات البذور لكل جدول
foreach ($coreTables as $table) {
    generateSeederForTable($table);
}

echo "تم إنشاء ملفات البذور بنجاح في المجلد: {$seedersPath}\n";

/**
 * إنشاء ملف DatabaseSeeder الرئيسي
 */
function createDatabaseSeeder($tables)
{
    global $seedersPath;

    $content = "<?php\n\n";
    $content .= "namespace Database\\Seeders;\n\n";
    $content .= "use Illuminate\\Database\\Seeder;\n\n";
    $content .= "class DatabaseSeeder extends Seeder\n";
    $content .= "{\n";
    $content .= "    /**\n";
    $content .= "     * Seed the application's database.\n";
    $content .= "     */\n";
    $content .= "    public function run(): void\n";
    $content .= "    {\n";

    foreach ($tables as $table) {
        $seederClass = Str::studly(Str::singular($table)) . 'Seeder';
        $content .= "        \$this->call({$seederClass}::class);\n";
    }

    $content .= "    }\n";
    $content .= "}\n";

    file_put_contents($seedersPath . '/DatabaseSeeder.php', $content);
    echo "تم إنشاء ملف DatabaseSeeder الرئيسي\n";
}

/**
 * إنشاء ملف بذور لجدول معين
 */
function generateSeederForTable($tableName)
{
    global $seedersPath;

    // الحصول على بيانات الجدول
    $data = DB::table($tableName)->get();

    if ($data->isEmpty()) {
        echo "الجدول {$tableName} فارغ، لم يتم إنشاء ملف بذور له\n";
        return;
    }

    // إنشاء اسم الصنف
    $className = Str::studly(Str::singular($tableName)) . 'Seeder';

    // إنشاء محتوى ملف البذور
    $content = "<?php\n\n";
    $content .= "namespace Database\\Seeders;\n\n";
    $content .= "use Illuminate\\Database\\Seeder;\n";
    $content .= "use Illuminate\\Support\\Facades\\DB;\n";
    $content .= "use Illuminate\\Support\\Facades\\Hash;\n\n";
    $content .= "class {$className} extends Seeder\n";
    $content .= "{\n";
    $content .= "    /**\n";
    $content .= "     * Run the database seeds.\n";
    $content .= "     */\n";
    $content .= "    public function run(): void\n";
    $content .= "    {\n";

    // إضافة عبارة التعطيل المؤقت للمفاتيح الأجنبية
    $content .= "        // تعطيل فحص المفاتيح الأجنبية مؤقتًا\n";
    $content .= "        DB::statement('SET FOREIGN_KEY_CHECKS=0');\n\n";

    // إضافة عبارة حذف البيانات الموجودة
    $content .= "        // حذف البيانات الموجودة\n";
    $content .= "        DB::table('{$tableName}')->truncate();\n\n";

    // إضافة البيانات
    $content .= "        // إضافة البيانات\n";
    $content .= "        \$data = [\n";

    foreach ($data as $row) {
        $content .= "            [\n";

        foreach ((array)$row as $column => $value) {
            // معالجة القيم الخاصة
            if (is_null($value)) {
                $content .= "                '{$column}' => null,\n";
            } elseif (is_bool($value)) {
                $boolValue = $value ? 'true' : 'false';
                $content .= "                '{$column}' => {$boolValue},\n";
            } elseif (is_numeric($value)) {
                $content .= "                '{$column}' => {$value},\n";
            } elseif ($column === 'password' && $tableName === 'users') {
                // كلمات المرور مشفرة بالفعل، نستخدمها كما هي
                $cleanValue = str_replace("'", "\'", $value);
                $content .= "                '{$column}' => '{$cleanValue}',\n";
            } else {
                // تنظيف النص من الأحرف الخاصة
                $cleanValue = str_replace("'", "\'", $value);
                $content .= "                '{$column}' => '{$cleanValue}',\n";
            }
        }

        $content .= "            ],\n";
    }

    $content .= "        ];\n\n";
    $content .= "        DB::table('{$tableName}')->insert(\$data);\n\n";

    // إضافة عبارة إعادة تفعيل المفاتيح الأجنبية
    $content .= "        // إعادة تفعيل فحص المفاتيح الأجنبية\n";
    $content .= "        DB::statement('SET FOREIGN_KEY_CHECKS=1');\n";

    $content .= "    }\n";
    $content .= "}\n";

    // حفظ ملف البذور
    file_put_contents($seedersPath . '/' . $className . '.php', $content);

    echo "تم إنشاء ملف البذور للجدول: {$tableName}\n";
}
