<?php

require __DIR__ . '/vendor/autoload.php';

$app = require_once __DIR__ . '/bootstrap/app.php';
$app->make(Illuminate\Contracts\Console\Kernel::class)->bootstrap();

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;
use Illuminate\Database\Schema\Blueprint;

// المسار الذي سيتم حفظ ملفات الهجرات فيه
$migrationsPath = __DIR__ . '/database/migrations_generated';

// إنشاء المجلد إذا لم يكن موجودًا
if (!file_exists($migrationsPath)) {
    mkdir($migrationsPath, 0755, true);
}

// الحصول على قائمة بجميع الجداول في قاعدة البيانات
$tables = DB::select('SHOW TABLES');
$dbName = DB::connection()->getDatabaseName();
$tableKey = 'Tables_in_' . $dbName;

// تجاهل جداول Laravel الافتراضية
$ignoreTables = [
    'migrations',
    'password_resets',
    'failed_jobs',
    'personal_access_tokens',
];

// ترتيب الجداول بناءً على العلاقات
$processedTables = [];
$tablesToProcess = [];

foreach ($tables as $table) {
    $tableName = $table->$tableKey;
    if (!in_array($tableName, $ignoreTables)) {
        $tablesToProcess[] = $tableName;
    }
}

// تحديد ترتيب الجداول بناءً على العلاقات
$sortedTables = sortTablesByRelations($tablesToProcess);

// إنشاء ملفات الهجرات
foreach ($sortedTables as $index => $tableName) {
    generateMigrationForTable($tableName, $index + 1);
}

echo "تم إنشاء ملفات الهجرات بنجاح في المجلد: {$migrationsPath}\n";

/**
 * ترتيب الجداول بناءً على العلاقات
 */
function sortTablesByRelations($tables)
{
    $dependencies = [];
    $sorted = [];

    // تحديد العلاقات بين الجداول
    foreach ($tables as $table) {
        $dependencies[$table] = [];
        $foreignKeys = DB::select("
            SELECT REFERENCED_TABLE_NAME
            FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE
            WHERE TABLE_SCHEMA = DATABASE()
            AND TABLE_NAME = '{$table}'
            AND REFERENCED_TABLE_NAME IS NOT NULL
        ");

        foreach ($foreignKeys as $fk) {
            if (in_array($fk->REFERENCED_TABLE_NAME, $tables) && $fk->REFERENCED_TABLE_NAME !== $table) {
                $dependencies[$table][] = $fk->REFERENCED_TABLE_NAME;
            }
        }
    }

    // ترتيب الجداول
    $visited = [];
    $temp = [];

    foreach ($tables as $table) {
        if (!isset($visited[$table])) {
            topologicalSort($table, $visited, $temp, $sorted, $dependencies);
        }
    }

    return array_reverse($sorted);
}

/**
 * ترتيب طوبولوجي للجداول
 */
function topologicalSort($table, &$visited, &$temp, &$sorted, $dependencies)
{
    $visited[$table] = true;
    $temp[$table] = true;

    if (isset($dependencies[$table])) {
        foreach ($dependencies[$table] as $dependency) {
            if (!isset($visited[$dependency])) {
                topologicalSort($dependency, $visited, $temp, $sorted, $dependencies);
            } elseif (isset($temp[$dependency])) {
                // هناك دورة في العلاقات، نتجاهلها
                continue;
            }
        }
    }

    unset($temp[$table]);
    $sorted[] = $table;
}

/**
 * إنشاء ملف هجرة لجدول معين
 */
function generateMigrationForTable($tableName, $order)
{
    global $migrationsPath;

    // الحصول على معلومات الأعمدة
    $columns = DB::select("SHOW FULL COLUMNS FROM `{$tableName}`");

    // الحصول على معلومات المفاتيح الأجنبية
    $foreignKeys = DB::select("
        SELECT
            COLUMN_NAME,
            REFERENCED_TABLE_NAME,
            REFERENCED_COLUMN_NAME
        FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE
        WHERE TABLE_SCHEMA = DATABASE()
        AND TABLE_NAME = '{$tableName}'
        AND REFERENCED_TABLE_NAME IS NOT NULL
    ");

    // الحصول على معلومات المفاتيح
    $indexes = DB::select("SHOW INDEX FROM `{$tableName}`");

    // إنشاء محتوى ملف الهجرة
    $className = 'Create' . Str::studly($tableName) . 'Table';
    $timestamp = date('Y_m_d_His', time() + $order);
    $filename = $timestamp . '_create_' . $tableName . '_table.php';

    $migrationContent = "<?php\n\n";
    $migrationContent .= "use Illuminate\\Database\\Migrations\\Migration;\n";
    $migrationContent .= "use Illuminate\\Database\\Schema\\Blueprint;\n";
    $migrationContent .= "use Illuminate\\Support\\Facades\\Schema;\n\n";
    $migrationContent .= "return new class extends Migration\n";
    $migrationContent .= "{\n";
    $migrationContent .= "    /**\n";
    $migrationContent .= "     * Run the migrations.\n";
    $migrationContent .= "     */\n";
    $migrationContent .= "    public function up(): void\n";
    $migrationContent .= "    {\n";
    $migrationContent .= "        Schema::create('{$tableName}', function (Blueprint \$table) {\n";

    // إضافة الأعمدة
    foreach ($columns as $column) {
        $columnName = $column->Field;
        $columnType = $column->Type;
        $columnNull = $column->Null === 'YES';
        $columnDefault = $column->Default;
        $columnExtra = $column->Extra;
        $columnComment = $column->Comment;

        $migrationContent .= "            " . generateColumnDefinition($columnName, $columnType, $columnNull, $columnDefault, $columnExtra, $columnComment) . "\n";
    }

    // إضافة المفاتيح الأجنبية
    if (!empty($foreignKeys)) {
        $migrationContent .= "\n            // Foreign Keys\n";
        foreach ($foreignKeys as $fk) {
            $migrationContent .= "            \$table->foreign('{$fk->COLUMN_NAME}')->references('{$fk->REFERENCED_COLUMN_NAME}')->on('{$fk->REFERENCED_TABLE_NAME}');\n";
        }
    }

    // إضافة المفاتيح والفهارس
    $uniqueIndexes = [];
    $normalIndexes = [];

    foreach ($indexes as $index) {
        if ($index->Key_name === 'PRIMARY') {
            continue; // تم التعامل مع المفتاح الأساسي بالفعل
        }

        if ($index->Non_unique == 0) {
            $uniqueIndexes[$index->Key_name][] = $index->Column_name;
        } else {
            $normalIndexes[$index->Key_name][] = $index->Column_name;
        }
    }

    if (!empty($uniqueIndexes)) {
        $migrationContent .= "\n            // Unique Indexes\n";
        foreach ($uniqueIndexes as $indexName => $columns) {
            $columnsStr = "'" . implode("', '", $columns) . "'";
            $migrationContent .= "            \$table->unique([{$columnsStr}], '{$indexName}');\n";
        }
    }

    if (!empty($normalIndexes)) {
        $migrationContent .= "\n            // Indexes\n";
        foreach ($normalIndexes as $indexName => $columns) {
            $columnsStr = "'" . implode("', '", $columns) . "'";
            $migrationContent .= "            \$table->index([{$columnsStr}], '{$indexName}');\n";
        }
    }

    $migrationContent .= "        });\n";
    $migrationContent .= "    }\n\n";
    $migrationContent .= "    /**\n";
    $migrationContent .= "     * Reverse the migrations.\n";
    $migrationContent .= "     */\n";
    $migrationContent .= "    public function down(): void\n";
    $migrationContent .= "    {\n";
    $migrationContent .= "        Schema::dropIfExists('{$tableName}');\n";
    $migrationContent .= "    }\n";
    $migrationContent .= "};\n";

    // حفظ ملف الهجرة
    file_put_contents($migrationsPath . '/' . $filename, $migrationContent);

    echo "تم إنشاء ملف الهجرة للجدول: {$tableName}\n";
}

/**
 * إنشاء تعريف العمود في ملف الهجرة
 */
function generateColumnDefinition($name, $type, $nullable, $default, $extra, $comment)
{
    $definition = '';

    // تحديد نوع العمود
    if (strpos($type, 'int') !== false) {
        if (strpos($type, 'tinyint(1)') !== false) {
            $definition = "\$table->boolean('{$name}')";
        } elseif (strpos($type, 'bigint') !== false) {
            $definition = "\$table->bigInteger('{$name}')";
        } elseif (strpos($type, 'smallint') !== false) {
            $definition = "\$table->smallInteger('{$name}')";
        } elseif (strpos($type, 'mediumint') !== false) {
            $definition = "\$table->mediumInteger('{$name}')";
        } else {
            $definition = "\$table->integer('{$name}')";
        }

        // التحقق من الإشارة
        if (strpos($type, 'unsigned') !== false) {
            $definition .= '->unsigned()';
        }
    } elseif (strpos($type, 'varchar') !== false || strpos($type, 'char') !== false) {
        preg_match('/\((\d+)\)/', $type, $matches);
        $length = isset($matches[1]) ? $matches[1] : 255;
        $definition = "\$table->string('{$name}', {$length})";
    } elseif (strpos($type, 'text') !== false) {
        if (strpos($type, 'longtext') !== false) {
            $definition = "\$table->longText('{$name}')";
        } elseif (strpos($type, 'mediumtext') !== false) {
            $definition = "\$table->mediumText('{$name}')";
        } else {
            $definition = "\$table->text('{$name}')";
        }
    } elseif (strpos($type, 'decimal') !== false || strpos($type, 'numeric') !== false) {
        preg_match('/\((\d+),(\d+)\)/', $type, $matches);
        $precision = isset($matches[1]) ? $matches[1] : 8;
        $scale = isset($matches[2]) ? $matches[2] : 2;
        $definition = "\$table->decimal('{$name}', {$precision}, {$scale})";
    } elseif (strpos($type, 'double') !== false) {
        $definition = "\$table->double('{$name}')";
    } elseif (strpos($type, 'float') !== false) {
        $definition = "\$table->float('{$name}')";
    } elseif (strpos($type, 'datetime') !== false) {
        $definition = "\$table->dateTime('{$name}')";
    } elseif (strpos($type, 'timestamp') !== false) {
        $definition = "\$table->timestamp('{$name}')";
    } elseif (strpos($type, 'date') !== false) {
        $definition = "\$table->date('{$name}')";
    } elseif (strpos($type, 'time') !== false) {
        $definition = "\$table->time('{$name}')";
    } elseif (strpos($type, 'enum') !== false) {
        preg_match('/enum\((.*)\)/', $type, $matches);
        $values = str_replace("'", '', $matches[1]);
        $values = explode(',', $values);
        $valuesStr = "'" . implode("', '", $values) . "'";
        $definition = "\$table->enum('{$name}', [{$valuesStr}])";
    } elseif (strpos($type, 'json') !== false) {
        $definition = "\$table->json('{$name}')";
    } else {
        // نوع افتراضي
        $definition = "\$table->string('{$name}')";
    }

    // إضافة خصائص إضافية
    if ($name === 'id' && $extra === 'auto_increment') {
        $definition = "\$table->id()";
    } elseif ($extra === 'auto_increment') {
        $definition .= '->autoIncrement()';
    }

    if ($nullable) {
        $definition .= '->nullable()';
    }

    if ($default !== null) {
        if ($default === 'CURRENT_TIMESTAMP') {
            $definition .= '->useCurrent()';
        } elseif (is_numeric($default)) {
            $definition .= "->default({$default})";
        } else {
            $definition .= "->default('{$default}')";
        }
    }

    if ($extra === 'on update CURRENT_TIMESTAMP') {
        $definition .= '->useCurrentOnUpdate()';
    }

    if (!empty($comment)) {
        $definition .= "->comment('{$comment}')";
    }

    return $definition . ';';
}
