<?php
require __DIR__ . '/vendor/autoload.php';
require __DIR__ . '/bootstrap/app.php';

$app = app();
$app->make(\Illuminate\Contracts\Console\Kernel::class)->bootstrap();

use App\Models\JournalEntry;
use App\Models\JournalEntryItem;
use App\Models\ChartOfAccount;
use App\Models\Customer;
use App\Models\CustomerStatement;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

// تحقق من القيد المحاسبي رقم 33
echo "Checking Journal Entry 33...\n";
$entry = JournalEntry::find(33);

if (!$entry) {
    echo "Journal Entry 33 not found!\n";
    exit;
}

echo "Found Journal Entry 33: " . $entry->entry_number . "\n";

// حذف سجلات كشف الحساب الحالية للقيد المحاسبي رقم 33
echo "Deleting existing customer statements for Journal Entry 33...\n";
$deleted = CustomerStatement::where('reference_type', 'App\Models\JournalEntry')
    ->where('reference_id', 33)
    ->delete();
echo "Deleted " . $deleted . " existing statements.\n";

// البحث عن حسابات العملاء في القيد المحاسبي
echo "Finding customer accounts in Journal Entry 33...\n";
$customerAccountItems = JournalEntryItem::where('journal_entry_id', 33)
    ->whereHas('account', function ($query) {
        $query->where('account_type', 'asset')
            ->where(function ($q) {
                $q->where('sub_type', 'accounts_receivable')
                    ->orWhere('sub_type', 'customer_receivable');
            });
    })
    ->get();

echo "Found " . $customerAccountItems->count() . " customer account items.\n";

// إنشاء سجلات كشف الحساب للعملاء المتأثرين بالقيد المحاسبي
$statementsCreated = [];

foreach ($customerAccountItems as $item) {
    // الحصول على حساب العميل
    $customerAccount = ChartOfAccount::find($item->account_id);
    if (!$customerAccount) {
        echo "Account not found for item: " . $item->id . "\n";
        continue;
    }

    echo "Processing account: " . $customerAccount->name . " (" . $customerAccount->account_code . ")\n";

    // استخراج معرف العميل من رمز الحساب (AR-{customer_id})
    $accountCode = $customerAccount->account_code;
    $customerId = null;

    if (strpos($accountCode, 'AR-') === 0) {
        $customerId = intval(substr($accountCode, 3));
        echo "Extracted customer ID from account code: " . $customerId . "\n";
    }

    // إذا لم نتمكن من استخراج معرف العميل، نحاول البحث بالاسم
    if (!$customerId) {
        // استخراج اسم العميل من اسم الحساب (ذمم العميل: {customer_name})
        $customerName = str_replace('ذمم العميل: ', '', $customerAccount->name);
        $customerName = str_replace('Customer Receivable: ', '', $customerName);

        echo "Extracted customer name: " . $customerName . "\n";

        // تحقق من وجود "عميل رقم" في اسم الحساب
        if (preg_match('/عميل رقم(\d+)/', $customerName, $matches)) {
            $customerId = intval($matches[1]);
            echo "Extracted customer ID from account name pattern 'عميل رقم': " . $customerId . "\n";
        } else {
            // البحث عن العميل بالاسم
            $customer = Customer::where('company_id', $entry->company_id)
                ->where(function ($query) use ($customerName) {
                    $query->where('name', 'like', '%' . $customerName . '%')
                        ->orWhere('name_en', 'like', '%' . $customerName . '%');
                })
                ->first();

            if ($customer) {
                $customerId = $customer->id;
                echo "Found customer by name: " . $customer->name . " (ID: " . $customerId . ")\n";
            } else {
                // محاولة البحث عن العميل بمقارنة جميع العملاء مع اسم الحساب
                echo "Customer not found by name, trying to match with all customers...\n";
                $allCustomers = Customer::where('company_id', $entry->company_id)->get();
                foreach ($allCustomers as $potentialCustomer) {
                    if (
                        stripos($customerAccount->name, $potentialCustomer->name) !== false ||
                        (isset($potentialCustomer->name_en) && stripos($customerAccount->name, $potentialCustomer->name_en) !== false)
                    ) {
                        $customerId = $potentialCustomer->id;
                        echo "Found customer by name comparison: " . $potentialCustomer->name . " (ID: " . $customerId . ")\n";
                        break;
                    }
                }

                if (!$customerId) {
                    echo "Could not find customer for account: " . $customerAccount->name . " (" . $accountCode . ")\n";

                    // إضافة: محاولة إنشاء سجل لكل العملاء
                    echo "Creating statements for all customers as a fallback...\n";
                    $allCustomers = Customer::where('company_id', $entry->company_id)->get();
                    foreach ($allCustomers as $customer) {
                        echo "Creating statement for customer: " . $customer->name . " (ID: " . $customer->id . ")\n";

                        // الحصول على آخر رصيد لهذا العميل
                        $lastStatement = CustomerStatement::where('customer_id', $customer->id)
                            ->orderBy('id', 'desc')
                            ->first();

                        $currentBalance = $lastStatement ? $lastStatement->balance : 0;

                        // حساب الرصيد الجديد
                        $debitAmount = $item->debit;
                        $creditAmount = $item->credit;
                        $netAmount = $creditAmount - $debitAmount;
                        $newBalance = $currentBalance - $netAmount;

                        // الحصول على معلومات المرجع بناءً على مرجع القيد المحاسبي
                        $referenceType = $entry->reference_type;
                        $referenceId = $entry->reference_id;
                        $description = $entry->description;

                        // إذا كان المرجع فاتورة، نحصل على مزيد من التفاصيل
                        if ($referenceType == 'App\Models\Invoice' && $referenceId) {
                            $invoice = \App\Models\Invoice::find($referenceId);
                            if ($invoice) {
                                $description = 'فاتورة رقم ' . $invoice->invoice_number . ' (قيد محاسبي: ' . $entry->entry_number . ')';
                            }
                        }
                        // إذا كان المرجع دفعة، نحصل على مزيد من التفاصيل
                        else if ($referenceType == 'App\Models\Payment' && $referenceId) {
                            $payment = \App\Models\Payment::find($referenceId);
                            if ($payment && $payment->invoice) {
                                $description = 'دفعة للفاتورة رقم ' . $payment->invoice->invoice_number . ' (قيد محاسبي: ' . $entry->entry_number . ')';
                            }
                        }

                        // إنشاء سجل كشف الحساب
                        $statement = CustomerStatement::create([
                            'company_id' => $entry->company_id,
                            'customer_id' => $customer->id,
                            'branch_id' => null, // ليس لدينا معلومات الفرع من القيد المحاسبي
                            'transaction_date' => $entry->entry_date,
                            'reference_number' => $entry->entry_number,
                            'reference_type' => 'App\Models\JournalEntry',
                            'reference_id' => $entry->id,
                            'description' => $description,
                            'debit' => $debitAmount,
                            'credit' => $creditAmount,
                            'balance' => $newBalance
                        ]);

                        $statementsCreated[] = $statement;
                    }

                    continue; // تخطي بقية الكود لهذا العنصر
                }
            }
        }
    } else {
        // التحقق من وجود العميل
        $customer = Customer::find($customerId);
        if (!$customer) {
            echo "Customer not found with ID: " . $customerId . "\n";
            continue;
        }
        echo "Verified customer exists: " . $customer->name . " (ID: " . $customerId . ")\n";
    }

    // الحصول على آخر رصيد لهذا العميل
    $lastStatement = CustomerStatement::where('customer_id', $customerId)
        ->orderBy('id', 'desc')
        ->first();

    $currentBalance = $lastStatement ? $lastStatement->balance : 0;
    echo "Last balance for customer " . $customerId . ": " . $currentBalance . "\n";

    // حساب الرصيد الجديد
    $debitAmount = $item->debit;
    $creditAmount = $item->credit;
    $netAmount = $creditAmount - $debitAmount;
    $newBalance = $currentBalance - $netAmount;

    echo "Debit: " . $debitAmount . ", Credit: " . $creditAmount . ", Net: " . $netAmount . ", New Balance: " . $newBalance . "\n";

    // الحصول على معلومات المرجع بناءً على مرجع القيد المحاسبي
    $referenceType = $entry->reference_type;
    $referenceId = $entry->reference_id;
    $description = $entry->description;

    // إذا كان المرجع فاتورة، نحصل على مزيد من التفاصيل
    if ($referenceType == 'App\Models\Invoice' && $referenceId) {
        $invoice = \App\Models\Invoice::find($referenceId);
        if ($invoice) {
            $description = 'فاتورة رقم ' . $invoice->invoice_number . ' (قيد محاسبي: ' . $entry->entry_number . ')';
        }
    }
    // إذا كان المرجع دفعة، نحصل على مزيد من التفاصيل
    else if ($referenceType == 'App\Models\Payment' && $referenceId) {
        $payment = \App\Models\Payment::find($referenceId);
        if ($payment && $payment->invoice) {
            $description = 'دفعة للفاتورة رقم ' . $payment->invoice->invoice_number . ' (قيد محاسبي: ' . $entry->entry_number . ')';
        }
    }

    echo "Description: " . $description . "\n";

    // إنشاء سجل كشف الحساب
    $statement = CustomerStatement::create([
        'company_id' => $entry->company_id,
        'customer_id' => $customerId,
        'branch_id' => null, // ليس لدينا معلومات الفرع من القيد المحاسبي
        'transaction_date' => $entry->entry_date,
        'reference_number' => $entry->entry_number,
        'reference_type' => 'App\Models\JournalEntry',
        'reference_id' => $entry->id,
        'description' => $description,
        'debit' => $debitAmount,
        'credit' => $creditAmount,
        'balance' => $newBalance
    ]);

    echo "Created statement ID: " . $statement->id . " for customer ID: " . $customerId . "\n";
    $statementsCreated[] = $statement;
}

echo "\nTotal statements created: " . count($statementsCreated) . "\n";

// التحقق من سجلات كشف الحساب للعميل رقم 2
echo "\nChecking statements for customer ID 2...\n";
$customer2Statements = CustomerStatement::where('customer_id', 2)
    ->where('reference_type', 'App\Models\JournalEntry')
    ->where('reference_id', 33)
    ->get();

if ($customer2Statements->count() > 0) {
    echo "Found " . $customer2Statements->count() . " statements for customer ID 2 and Journal Entry 33.\n";
    foreach ($customer2Statements as $stmt) {
        echo "Statement ID: " . $stmt->id .
            ", Date: " . $stmt->transaction_date .
            ", Reference: " . $stmt->reference_number .
            ", Debit: " . $stmt->debit .
            ", Credit: " . $stmt->credit .
            ", Balance: " . $stmt->balance . "\n";
    }
} else {
    echo "No statements found for customer ID 2 and Journal Entry 33.\n";
}

echo "\nFix completed!\n";
