<?php

require __DIR__ . '/vendor/autoload.php';

// Load environment variables
$app = require_once __DIR__ . '/bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use App\Models\ChartOfAccount;

echo "=== بدء إصلاح مشكلة قيد الفريد لحسابات البنك ===\n\n";

try {
    // 1. تحديد ما إذا كان القيد الفريد موجوداً
    $constraintExists = false;
    $constraints = DB::select("
        SELECT CONSTRAINT_NAME
        FROM information_schema.TABLE_CONSTRAINTS
        WHERE TABLE_SCHEMA = DATABASE()
        AND TABLE_NAME = 'chart_of_accounts'
        AND CONSTRAINT_TYPE = 'UNIQUE'
        AND CONSTRAINT_NAME = 'chart_of_accounts_company_account_unique'
    ");

    if (!empty($constraints)) {
        $constraintExists = true;
        echo "تم العثور على قيد الفريد 'chart_of_accounts_company_account_unique'\n";
    } else {
        echo "لم يتم العثور على قيد الفريد 'chart_of_accounts_company_account_unique'\n";
    }

    // 2. إذا كان القيد موجوداً، نقوم بحذفه وإنشاء قيد جديد يستثني السجلات المحذوفة
    if ($constraintExists) {
        echo "جاري حذف القيد الفريد الحالي...\n";

        DB::statement("
            ALTER TABLE chart_of_accounts
            DROP INDEX chart_of_accounts_company_account_unique
        ");

        echo "تم حذف القيد الفريد بنجاح\n";

        echo "جاري إنشاء قيد فريد جديد يستثني السجلات المحذوفة...\n";

        DB::statement("
            ALTER TABLE chart_of_accounts
            ADD CONSTRAINT chart_of_accounts_company_account_active_unique
            UNIQUE (company_id, account_code, deleted_at)
        ");

        echo "تم إنشاء القيد الفريد الجديد بنجاح\n";
    }

    // 3. البحث عن حسابات البنك المتكررة للشركة رقم 3
    $companyId = 3;
    echo "\nفحص حسابات البنك للشركة رقم: {$companyId}\n";

    $bankAccounts = ChartOfAccount::withTrashed()
        ->where('company_id', $companyId)
        ->where('account_code', 'BANK')
        ->get();

    echo "تم العثور على " . $bankAccounts->count() . " حساب بنك رئيسي (BANK) للشركة {$companyId}\n";

    // عرض تفاصيل الحسابات
    foreach ($bankAccounts as $index => $account) {
        $status = $account->deleted_at ? 'محذوف' : 'نشط';
        echo "[{$index}] معرف الحساب: {$account->id}, الحالة: {$status}, تاريخ الإنشاء: {$account->created_at}";
        if ($account->deleted_at) {
            echo ", تاريخ الحذف: {$account->deleted_at}";
        }
        echo "\n";
    }

    // 4. إذا وجدنا أكثر من حساب، نقوم بالإصلاح
    if ($bankAccounts->count() > 1) {
        echo "\n=== بدء عملية الإصلاح ===\n";

        // تحديد الحساب النشط (إن وجد) أو أحدث حساب محذوف
        $activeAccount = $bankAccounts->firstWhere('deleted_at', null);

        if ($activeAccount) {
            echo "تم العثور على حساب نشط برقم معرف: {$activeAccount->id}\n";
            $accountToKeep = $activeAccount;
        } else {
            // إذا كانت جميع الحسابات محذوفة، نختار أحدثها
            $latestAccount = $bankAccounts->sortByDesc('created_at')->first();
            echo "لم يتم العثور على حساب نشط. سيتم استعادة أحدث حساب محذوف برقم معرف: {$latestAccount->id}\n";

            // استعادة الحساب المحذوف
            $latestAccount->restore();
            echo "تم استعادة الحساب رقم {$latestAccount->id} بنجاح\n";
            $accountToKeep = $latestAccount;
        }

        // حذف الحسابات الأخرى نهائياً
        foreach ($bankAccounts as $account) {
            if ($account->id != $accountToKeep->id) {
                echo "حذف نهائي للحساب رقم {$account->id}...\n";

                // تحديث الحسابات الفرعية التي تشير إلى هذا الحساب كأب
                $childAccounts = ChartOfAccount::withTrashed()
                    ->where('parent_id', $account->id)
                    ->get();

                if ($childAccounts->count() > 0) {
                    echo "  تحديث {$childAccounts->count()} حساب فرعي ليشير إلى الحساب الرئيسي الجديد {$accountToKeep->id}...\n";

                    foreach ($childAccounts as $child) {
                        $child->parent_id = $accountToKeep->id;
                        $child->save();
                    }
                }

                // حذف الحساب نهائياً من قاعدة البيانات
                try {
                    $account->forceDelete();
                    echo "  تم حذف الحساب نهائياً\n";
                } catch (\Exception $e) {
                    echo "  فشل في حذف الحساب: " . $e->getMessage() . "\n";

                    // محاولة تحديث الحساب بكود مختلف لتجنب تعارض المفتاح الفريد
                    try {
                        $account->account_code = 'BANK_DELETED_' . $account->id;
                        $account->save();
                        $account->forceDelete();
                        echo "  تم تغيير كود الحساب وحذفه نهائياً\n";
                    } catch (\Exception $e2) {
                        echo "  فشل في تحديث وحذف الحساب: " . $e2->getMessage() . "\n";
                    }
                }
            }
        }

        echo "\n=== تم الانتهاء من عملية الإصلاح ===\n";
    }

    // 5. التحقق من الحسابات بعد الإصلاح
    echo "\n=== التحقق من الحالة النهائية ===\n";
    $finalAccounts = ChartOfAccount::withTrashed()
        ->where('company_id', $companyId)
        ->where('account_code', 'BANK')
        ->get();

    echo "عدد حسابات البنك الرئيسية بعد الإصلاح: " . $finalAccounts->count() . "\n";
    foreach ($finalAccounts as $index => $account) {
        $status = $account->deleted_at ? 'محذوف' : 'نشط';
        echo "[{$index}] معرف الحساب: {$account->id}, الحالة: {$status}\n";
    }

    // 6. اختبار إنشاء حساب بنك جديد
    echo "\n=== اختبار إنشاء حساب بنك جديد ===\n";

    // البحث عن حساب البنك الرئيسي
    $parentBankAccount = ChartOfAccount::where('company_id', $companyId)
        ->where('account_code', 'BANK')
        ->first();

    if (!$parentBankAccount) {
        echo "لم يتم العثور على حساب البنك الرئيسي. جاري إنشاء حساب جديد...\n";

        // البحث عن حساب الأصول المتداولة
        $cashAndBanksAccount = ChartOfAccount::where('company_id', $companyId)
            ->where('account_code', '1110')
            ->first();

        if (!$cashAndBanksAccount) {
            echo "لم يتم العثور على حساب الأصول المتداولة للشركة {$companyId}. لا يمكن إنشاء حساب بنك جديد.\n";
        } else {
            $parentBankAccount = ChartOfAccount::create([
                'company_id' => $companyId,
                'account_code' => 'BANK',
                'name' => 'حسابات بنكية',
                'name_en' => 'Bank Accounts',
                'account_type' => 'asset',
                'sub_type' => 'bank',
                'parent_id' => $cashAndBanksAccount->id,
                'description' => 'حساب الأصول البنكية الرئيسي',
                'is_active' => true,
                'is_system' => true,
                'opening_balance' => 0,
                'current_balance' => 0,
                'currency' => 'SAR',
                'balance_type' => 'debit',
                'is_cash_account' => false,
                'is_bank_account' => true,
                'level' => $cashAndBanksAccount->level + 1,
                'is_parent' => true
            ]);

            echo "تم إنشاء حساب البنك الرئيسي برقم معرف: {$parentBankAccount->id}\n";
        }
    } else {
        echo "تم العثور على حساب البنك الرئيسي برقم معرف: {$parentBankAccount->id}\n";
    }

    if ($parentBankAccount) {
        // إنشاء حساب بنك فرعي جديد
        $newBankName = 'بنك تجريبي جديد ' . date('Y-m-d H:i:s');
        $newAccountCode = 'BANK-' . time();

        $newBankAccount = ChartOfAccount::create([
            'company_id' => $companyId,
            'account_code' => $newAccountCode,
            'name' => $newBankName,
            'name_en' => $newBankName,
            'account_type' => 'asset',
            'sub_type' => 'bank',
            'parent_id' => $parentBankAccount->id,
            'description' => 'حساب البنك: ' . $newBankName,
            'is_active' => true,
            'is_system' => false,
            'opening_balance' => 0,
            'current_balance' => 0,
            'currency' => 'SAR',
            'balance_type' => 'debit',
            'is_cash_account' => false,
            'is_bank_account' => true,
            'level' => $parentBankAccount->level + 1,
            'is_parent' => false
        ]);

        echo "تم إنشاء حساب بنك جديد بنجاح برقم معرف: {$newBankAccount->id} وكود: {$newAccountCode}\n";
    }

    echo "\n=== اكتمل إصلاح مشكلة قيد الفريد لحسابات البنك ===\n";
} catch (\Exception $e) {
    echo "حدث خطأ: " . $e->getMessage() . "\n";
    echo "ملف: " . $e->getFile() . "\n";
    echo "سطر: " . $e->getLine() . "\n";
}
