<?php
require_once __DIR__ . '/vendor/autoload.php';

use App\Models\Bank;
use App\Models\ChartOfAccount;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

// Initialize Laravel application
$app = require_once __DIR__ . '/bootstrap/app.php';
$app->make(\Illuminate\Contracts\Console\Kernel::class)->bootstrap();

echo "=== بدء تشخيص وإصلاح مشكلة حسابات البنك ===\n\n";

try {
    // تحديد الشركة التي تواجه المشكلة
    $companyId = 3;
    echo "فحص حسابات البنك للشركة رقم: {$companyId}\n";

    // البحث عن جميع حسابات البنك الرئيسية (BANK) للشركة المحددة، بما في ذلك المحذوفة
    $bankAccounts = ChartOfAccount::withTrashed()
        ->where('company_id', $companyId)
        ->where('account_code', 'BANK')
        ->get();

    echo "تم العثور على " . $bankAccounts->count() . " حساب بنك رئيسي (BANK) للشركة {$companyId}\n";

    // عرض تفاصيل الحسابات
    foreach ($bankAccounts as $index => $account) {
        $status = $account->deleted_at ? 'محذوف' : 'نشط';
        echo "\n[{$index}] معرف الحساب: {$account->id}, الحالة: {$status}, تاريخ الإنشاء: {$account->created_at}";
        if ($account->deleted_at) {
            echo ", تاريخ الحذف: {$account->deleted_at}";
        }
        echo "\n";
    }

    // إذا وجدنا أكثر من حساب، نقوم بالإصلاح
    if ($bankAccounts->count() > 1) {
        echo "\n=== بدء عملية الإصلاح ===\n";

        // تحديد الحساب النشط (إن وجد) أو أحدث حساب محذوف
        $activeAccount = $bankAccounts->firstWhere('deleted_at', null);

        if ($activeAccount) {
            echo "تم العثور على حساب نشط برقم معرف: {$activeAccount->id}\n";
            $accountToKeep = $activeAccount;
        } else {
            // إذا كانت جميع الحسابات محذوفة، نختار أحدثها
            $latestAccount = $bankAccounts->sortByDesc('created_at')->first();
            echo "لم يتم العثور على حساب نشط. سيتم استعادة أحدث حساب محذوف برقم معرف: {$latestAccount->id}\n";

            // استعادة الحساب المحذوف
            $latestAccount->restore();
            echo "تم استعادة الحساب رقم {$latestAccount->id} بنجاح\n";
            $accountToKeep = $latestAccount;
        }

        // حذف الحسابات الأخرى نهائياً
        foreach ($bankAccounts as $account) {
            if ($account->id != $accountToKeep->id) {
                echo "حذف نهائي للحساب رقم {$account->id}...\n";

                // تحديث الحسابات الفرعية التي تشير إلى هذا الحساب كأب
                $childAccounts = ChartOfAccount::withTrashed()
                    ->where('parent_id', $account->id)
                    ->get();

                if ($childAccounts->count() > 0) {
                    echo "  تحديث {$childAccounts->count()} حساب فرعي ليشير إلى الحساب الرئيسي الجديد {$accountToKeep->id}...\n";

                    foreach ($childAccounts as $child) {
                        $child->parent_id = $accountToKeep->id;
                        $child->save();
                    }
                }

                // حذف الحساب نهائياً من قاعدة البيانات
                try {
                    $account->forceDelete();
                    echo "  تم حذف الحساب نهائياً\n";
                } catch (\Exception $e) {
                    echo "  فشل في حذف الحساب: " . $e->getMessage() . "\n";

                    // محاولة تحديث الحساب بكود مختلف لتجنب تعارض المفتاح الفريد
                    try {
                        $account->account_code = 'BANK_DELETED_' . $account->id;
                        $account->save();
                        $account->forceDelete();
                        echo "  تم تغيير كود الحساب وحذفه نهائياً\n";
                    } catch (\Exception $e2) {
                        echo "  فشل في تحديث وحذف الحساب: " . $e2->getMessage() . "\n";
                    }
                }
            }
        }

        // التأكد من أن الحساب المحتفظ به هو حساب أب
        if (!$accountToKeep->is_parent) {
            $accountToKeep->is_parent = true;
            $accountToKeep->save();
            echo "تم تحديث الحساب {$accountToKeep->id} ليكون حساب أب\n";
        }

        // التأكد من أن الحساب المحتفظ به مرتبط بحساب الأصول المتداولة
        $currentAssetsAccount = ChartOfAccount::where('company_id', $companyId)
            ->where('account_code', '1110')
            ->first();

        if ($currentAssetsAccount && $accountToKeep->parent_id != $currentAssetsAccount->id) {
            $oldParentId = $accountToKeep->parent_id;
            $accountToKeep->parent_id = $currentAssetsAccount->id;
            $accountToKeep->level = $currentAssetsAccount->level + 1;
            $accountToKeep->save();
            echo "تم تحديث الحساب {$accountToKeep->id} ليكون تابعاً لحساب الأصول المتداولة (من {$oldParentId} إلى {$currentAssetsAccount->id})\n";
        }

        echo "\n=== تم الانتهاء من عملية الإصلاح ===\n";
    } else if ($bankAccounts->count() == 1 && $bankAccounts->first()->deleted_at) {
        // إذا كان هناك حساب واحد فقط وهو محذوف، نقوم باستعادته
        $account = $bankAccounts->first();
        echo "\nتم العثور على حساب واحد محذوف. جاري استعادته...\n";
        $account->restore();
        echo "تم استعادة الحساب رقم {$account->id} بنجاح\n";

        // التأكد من أن الحساب هو حساب أب
        if (!$account->is_parent) {
            $account->is_parent = true;
            $account->save();
            echo "تم تحديث الحساب {$account->id} ليكون حساب أب\n";
        }
    } else if ($bankAccounts->count() == 0) {
        echo "\nلم يتم العثور على أي حساب بنك رئيسي للشركة {$companyId}. سيتم إنشاء حساب جديد.\n";

        // إنشاء حساب بنك رئيسي جديد
        $currentAssetsAccount = ChartOfAccount::where('company_id', $companyId)
            ->where('account_code', '1110')
            ->first();

        if (!$currentAssetsAccount) {
            echo "لم يتم العثور على حساب الأصول المتداولة للشركة {$companyId}. لا يمكن إنشاء حساب بنك جديد.\n";
        } else {
            $newBankAccount = ChartOfAccount::create([
                'company_id' => $companyId,
                'account_code' => 'BANK',
                'name' => 'حسابات بنكية',
                'name_en' => 'Bank Accounts',
                'account_type' => 'asset',
                'sub_type' => 'bank',
                'parent_id' => $currentAssetsAccount->id,
                'description' => 'حساب الأصول البنكية الرئيسي',
                'is_active' => true,
                'is_system' => true,
                'opening_balance' => 0,
                'current_balance' => 0,
                'currency' => 'SAR',
                'balance_type' => 'debit',
                'is_cash_account' => false,
                'is_bank_account' => true,
                'level' => $currentAssetsAccount->level + 1,
                'is_parent' => true
            ]);

            echo "تم إنشاء حساب بنك رئيسي جديد برقم معرف: {$newBankAccount->id}\n";
        }
    }

    // التحقق من الحسابات بعد الإصلاح
    echo "\n=== التحقق من الحالة النهائية ===\n";
    $finalAccounts = ChartOfAccount::withTrashed()
        ->where('company_id', $companyId)
        ->where('account_code', 'BANK')
        ->get();

    echo "عدد حسابات البنك الرئيسية بعد الإصلاح: " . $finalAccounts->count() . "\n";
    foreach ($finalAccounts as $index => $account) {
        $status = $account->deleted_at ? 'محذوف' : 'نشط';
        echo "[{$index}] معرف الحساب: {$account->id}, الحالة: {$status}\n";
    }

    // اختبار إنشاء حساب بنك جديد
    echo "\n=== اختبار إنشاء حساب بنك جديد ===\n";
    $newBankAccount = createBankAccountFixed($companyId, 'بنك تجريبي جديد');

    if ($newBankAccount) {
        echo "تم إنشاء حساب بنك جديد بنجاح برقم معرف: {$newBankAccount->id}\n";
    } else {
        echo "فشل في إنشاء حساب بنك جديد\n";
    }

    echo "\n=== اكتمل تشخيص وإصلاح مشكلة حسابات البنك ===\n";
} catch (\Exception $e) {
    echo "حدث خطأ: " . $e->getMessage() . "\n";
    echo "ملف: " . $e->getFile() . "\n";
    echo "سطر: " . $e->getLine() . "\n";
}

// دالة إنشاء حساب بنك محسنة
function createBankAccountFixed($companyId, $bankName)
{
    // البحث عن حساب البنك الرئيسي، بما في ذلك المحذوفة
    $parentBankAccount = ChartOfAccount::withTrashed()
        ->where('company_id', $companyId)
        ->where('account_code', 'BANK')
        ->first();

    // إذا كان الحساب محذوفاً، نقوم باستعادته
    if ($parentBankAccount && $parentBankAccount->deleted_at) {
        $parentBankAccount->restore();
        echo "تم استعادة حساب البنك الرئيسي المحذوف برقم معرف: {$parentBankAccount->id}\n";
    }

    if (!$parentBankAccount) {
        // البحث عن حساب الأصول المتداولة
        $cashAndBanksAccount = ChartOfAccount::where('company_id', $companyId)
            ->where('account_code', '1110')
            ->first();

        if (!$cashAndBanksAccount) {
            echo "لم يتم العثور على حساب الأصول المتداولة للشركة {$companyId}\n";
            return null;
        }

        // إنشاء حساب البنك الرئيسي
        $parentBankAccount = ChartOfAccount::create([
            'company_id' => $companyId,
            'account_code' => 'BANK',
            'name' => 'حسابات بنكية',
            'name_en' => 'Bank Accounts',
            'account_type' => 'asset',
            'sub_type' => 'bank',
            'parent_id' => $cashAndBanksAccount->id,
            'description' => 'حساب الأصول البنكية الرئيسي',
            'is_active' => true,
            'is_system' => true,
            'opening_balance' => 0,
            'current_balance' => 0,
            'currency' => 'SAR',
            'balance_type' => 'debit',
            'is_cash_account' => false,
            'is_bank_account' => true,
            'level' => $cashAndBanksAccount->level + 1,
            'is_parent' => true
        ]);

        echo "تم إنشاء حساب البنك الرئيسي برقم معرف: {$parentBankAccount->id}\n";
    } else {
        echo "استخدام حساب البنك الرئيسي الموجود برقم معرف: {$parentBankAccount->id}\n";

        // التأكد من أن الحساب هو حساب أب
        if (!$parentBankAccount->is_parent) {
            $parentBankAccount->is_parent = true;
            $parentBankAccount->save();
            echo "تم تحديث الحساب {$parentBankAccount->id} ليكون حساب أب\n";
        }
    }

    // إنشاء حساب البنك الفرعي
    // توليد كود فريد لهذا البنك
    $lastBankAccount = ChartOfAccount::where('parent_id', $parentBankAccount->id)
        ->where('is_bank_account', true)
        ->where('company_id', $companyId)
        ->orderBy('account_code', 'desc')
        ->first();

    // تحديد كود الحساب الجديد
    if ($lastBankAccount) {
        // إذا كانت هناك حسابات بنك سابقة، نقوم بزيادة الكود
        $lastCode = intval(str_replace('BANK-', '', $lastBankAccount->account_code));
        $newAccountCode = 'BANK-' . ($lastCode + 1);
    } else {
        // إذا لم تكن هناك حسابات بنك سابقة، نقوم بإنشاء الأول
        $newAccountCode = 'BANK-1000001';
    }

    // التأكد من أن الكود فريد
    while (ChartOfAccount::withTrashed()
        ->where('company_id', $companyId)
        ->where('account_code', $newAccountCode)
        ->exists()
    ) {
        // إذا كان الكود موجوداً بالفعل، نقوم بزيادته
        $codeNumber = intval(str_replace('BANK-', '', $newAccountCode));
        $newAccountCode = 'BANK-' . ($codeNumber + 1);
    }

    // إنشاء حساب البنك
    $bankAccount = ChartOfAccount::create([
        'company_id' => $companyId,
        'account_code' => $newAccountCode,
        'name' => $bankName,
        'name_en' => $bankName,
        'account_type' => 'asset',
        'sub_type' => 'bank',
        'parent_id' => $parentBankAccount->id,
        'description' => 'حساب البنك: ' . $bankName,
        'is_active' => true,
        'is_system' => false,
        'opening_balance' => 0,
        'current_balance' => 0,
        'currency' => 'SAR',
        'balance_type' => 'debit',
        'is_cash_account' => false,
        'is_bank_account' => true,
        'level' => $parentBankAccount->level + 1,
        'is_parent' => false
    ]);

    echo "تم إنشاء حساب بنك جديد لـ '{$bankName}' برقم معرف: {$bankAccount->id} وكود: {$newAccountCode}\n";
    return $bankAccount;
}
