<?php
require_once __DIR__ . '/vendor/autoload.php';

use App\Models\Bank;
use App\Models\ChartOfAccount;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

// Initialize Laravel application
$app = require_once __DIR__ . '/bootstrap/app.php';
$app->make(\Illuminate\Contracts\Console\Kernel::class)->bootstrap();

try {
    // First, let's check if the BANK account with ID 61 is soft-deleted
    $bankAccount = DB::select("SELECT * FROM chart_of_accounts WHERE id = 61");

    if (!empty($bankAccount)) {
        $bankAccount = $bankAccount[0];
        echo "Found BANK account with ID 61:\n";
        echo "ID: {$bankAccount->id}, Code: {$bankAccount->account_code}, Name: {$bankAccount->name}, Deleted At: " . ($bankAccount->deleted_at ?? 'Not deleted') . "\n\n";

        // If it's soft-deleted, restore it
        if ($bankAccount->deleted_at) {
            DB::update("UPDATE chart_of_accounts SET deleted_at = NULL WHERE id = 61");
            echo "Restored soft-deleted BANK account with ID 61\n\n";
        }
    }

    // Now let's create a modified version of the BankAccountService::createBankAccount method
    // that properly handles the case where a BANK account already exists

    echo "Creating a modified version of BankAccountService::createBankAccount method...\n\n";

    // Here's the code that should be used to fix the BankAccountService.php file
    $fixedCode = <<<'EOD'
/**
 * إنشاء حساب للبنك في شجرة الحسابات وربطه به
 *
 * @param Bank $bank
 * @return ChartOfAccount|null
 */
public function createBankAccount(Bank $bank)
{
    try {
        // البحث عن حساب النقدية والبنوك (الحساب الأب للبنوك)
        $cashAndBanksAccount = $this->findOrCreateCashAndBanksAccount($bank->company_id);
        
        // البحث عن حساب البنوك الرئيسي (BANK) أو إنشائه إذا لم يكن موجوداً
        $parentBankAccount = $this->findOrCreateParentBankAccount($bank->company_id, $cashAndBanksAccount);
        
        // توليد رمز حساب فريد للبنك الجديد
        $lastBankAccount = ChartOfAccount::where('parent_id', $parentBankAccount->id)
            ->where('is_bank_account', true)
            ->where('company_id', $bank->company_id)
            ->orderBy('account_code', 'desc')
            ->first();
            
        // تحديد رمز الحساب الجديد
        if ($lastBankAccount) {
            // إذا كان هناك حسابات بنوك سابقة، نزيد الرمز بواحد
            // استخدام نمط BANK-XXXXXXX للرمز
            $lastCode = intval(str_replace('BANK-', '', $lastBankAccount->account_code));
            $newAccountCode = 'BANK-' . ($lastCode + 1);
        } else {
            // إذا لم يكن هناك حسابات بنوك سابقة، ننشئ أول حساب
            $newAccountCode = 'BANK-1000001';
        }
        
        // التحقق من عدم وجود حساب بنفس الرمز
        while (ChartOfAccount::where('company_id', $bank->company_id)
            ->where('account_code', $newAccountCode)
            ->exists()
        ) {
            // إذا كان الرمز موجوداً بالفعل، نزيد الرمز بواحد
            $codeNumber = intval(str_replace('BANK-', '', $newAccountCode));
            $newAccountCode = 'BANK-' . ($codeNumber + 1);
        }
        
        // إنشاء حساب للبنك (كحساب أب وليس فرعي)
        $account = ChartOfAccount::create([
            'company_id' => $bank->company_id,
            'account_code' => $newAccountCode,
            'name' => $bank->name,
            'name_en' => $bank->name,
            'account_type' => 'asset',
            'sub_type' => 'bank',
            'parent_id' => $parentBankAccount->id,
            'description' => 'حساب البنك: ' . $bank->name,
            'is_active' => $bank->is_active,
            'is_system' => false,
            'opening_balance' => 0,
            'current_balance' => 0,
            'currency' => 'SAR', // يمكن تعديله حسب عملة البنك
            'balance_type' => 'debit',
            'is_cash_account' => false,
            'is_bank_account' => true,
            'level' => $parentBankAccount->level + 1,
            'is_parent' => true // تم تغييره من false إلى true ليكون حساب أب
        ]);
        
        Log::info('تم إنشاء حساب للبنك في شجرة الحسابات', [
            'bank_id' => $bank->id,
            'bank_name' => $bank->name,
            'account_id' => $account->id,
            'account_code' => $newAccountCode,
            'user_id' => auth()->id()
        ]);
        
        return $account;
    } catch (\Exception $e) {
        Log::error('خطأ في إنشاء حساب البنك في شجرة الحسابات', [
            'bank_id' => $bank->id,
            'bank_name' => $bank->name,
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
        
        return null;
    }
}

/**
 * البحث عن حساب البنوك الرئيسي أو إنشائه إذا لم يكن موجوداً
 *
 * @param int $companyId
 * @param ChartOfAccount $cashAndBanksAccount
 * @return ChartOfAccount
 */
private function findOrCreateParentBankAccount($companyId, $cashAndBanksAccount)
{
    // البحث عن حساب البنوك الرئيسي بما في ذلك المحذوفة
    $parentBankAccount = DB::table('chart_of_accounts')
        ->where('company_id', $companyId)
        ->where('account_code', 'BANK')
        ->first();
        
    if ($parentBankAccount) {
        // إذا كان الحساب محذوفاً، نقوم باسترجاعه
        if ($parentBankAccount->deleted_at) {
            DB::table('chart_of_accounts')
                ->where('id', $parentBankAccount->id)
                ->update(['deleted_at' => null]);
                
            Log::info('تم استرجاع حساب البنوك الرئيسي المحذوف', [
                'account_id' => $parentBankAccount->id,
                'company_id' => $companyId
            ]);
        }
        
        // استرجاع الحساب كنموذج Eloquent
        return ChartOfAccount::find($parentBankAccount->id);
    }
    
    // إنشاء حساب البنوك الرئيسي إذا لم يكن موجوداً
    $parentBankAccount = ChartOfAccount::create([
        'company_id' => $companyId,
        'account_code' => 'BANK',
        'name' => 'حسابات بنكية',
        'name_en' => 'Bank Accounts',
        'account_type' => 'asset',
        'sub_type' => 'bank',
        'parent_id' => $cashAndBanksAccount->id,
        'description' => 'حساب الأصول البنكية الرئيسي',
        'is_active' => true,
        'is_system' => true,
        'opening_balance' => 0,
        'current_balance' => 0,
        'currency' => 'SAR',
        'balance_type' => 'debit',
        'is_cash_account' => false,
        'is_bank_account' => true,
        'level' => $cashAndBanksAccount->level + 1,
        'is_parent' => true
    ]);
    
    Log::info('تم إنشاء حساب البنوك الرئيسي', [
        'account_id' => $parentBankAccount->id,
        'company_id' => $companyId
    ]);
    
    return $parentBankAccount;
}
EOD;

    echo "Fixed code for BankAccountService.php:\n\n";
    echo $fixedCode;

    // Let's test if we can create a bank account now
    echo "\nTesting bank account creation with direct SQL approach...\n";

    // First, check if the BANK account exists
    $bankParentAccount = DB::select("SELECT * FROM chart_of_accounts WHERE company_id = 3 AND account_code = 'BANK'");

    if (!empty($bankParentAccount)) {
        $bankParentAccount = $bankParentAccount[0];
        echo "Using existing BANK parent account with ID: {$bankParentAccount->id}\n";

        // Generate a unique code for the new bank account
        $newBankCode = 'BANK-' . rand(1000000, 9999999);

        // Create a new bank account under the parent
        $result = DB::insert(
            "INSERT INTO chart_of_accounts (company_id, account_code, name, name_en, account_type, sub_type, parent_id, description, is_active, is_system, opening_balance, current_balance, currency, balance_type, is_cash_account, is_bank_account, level, is_parent, created_at, updated_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())",
            [
                3, // company_id
                $newBankCode, // account_code
                'بنك تجريبي جديد', // name
                'New Test Bank', // name_en
                'asset', // account_type
                'bank', // sub_type
                $bankParentAccount->id, // parent_id
                'حساب البنك: بنك تجريبي جديد', // description
                1, // is_active
                0, // is_system
                0, // opening_balance
                0, // current_balance
                'SAR', // currency
                'debit', // balance_type
                0, // is_cash_account
                1, // is_bank_account
                $bankParentAccount->level + 1, // level
                1, // is_parent
            ]
        );

        if ($result) {
            $newBankId = DB::getPdo()->lastInsertId();
            echo "Successfully created new bank account with ID: {$newBankId} and code: {$newBankCode}\n";
        } else {
            echo "Failed to create new bank account\n";
        }
    } else {
        echo "No BANK parent account found for company ID 3. This is unexpected based on previous results.\n";
    }

    echo "\nSummary of the issue and fix:\n";
    echo "1. The issue is that there's already a BANK account with ID 61 for company ID 3, but it might be soft-deleted or not visible to Eloquent.\n";
    echo "2. When trying to create a new bank account, the system tries to create a parent BANK account first, but fails due to the unique constraint on company_id and account_code.\n";
    echo "3. The fix is to modify the BankAccountService to properly check for existing BANK accounts, including soft-deleted ones, and restore them if needed.\n";
    echo "4. Additionally, we should use a different naming pattern for bank account codes to avoid conflicts.\n";
} catch (\Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    echo "File: " . $e->getFile() . "\n";
    echo "Line: " . $e->getLine() . "\n";
}
