<?php

require_once __DIR__ . '/vendor/autoload.php';

use App\Models\Bank;
use App\Models\ChartOfAccount;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

// تهيئة Laravel
$app = require_once __DIR__ . '/bootstrap/app.php';
$app->make(\Illuminate\Contracts\Console\Kernel::class)->bootstrap();

// تعديل دالة إنشاء الحساب البنكي
class BankAccountServiceFix
{
    /**
     * إنشاء حساب للبنك في شجرة الحسابات وربطه به
     * مع التحقق من وجود حساب للبنك الأب أولاً
     *
     * @param Bank $bank
     * @return ChartOfAccount|null
     */
    public function createBankAccount(Bank $bank)
    {
        try {
            // البحث عن حساب الأصول المتداولة (الحساب الأب للبنوك)
            $currentAssetsAccount = $this->findOrCreateCurrentAssetsAccount($bank->company_id);

            if (!$currentAssetsAccount) {
                Log::error('فشل في العثور على أو إنشاء حساب الأصول المتداولة', [
                    'bank_id' => $bank->id,
                    'company_id' => $bank->company_id
                ]);
                return null;
            }

            // 1. البحث عن حساب البنك الأب (بنفس اسم البنك) أو إنشائه إذا لم يكن موجوداً
            $parentBankAccount = $this->findOrCreateBankParentAccount($bank, $currentAssetsAccount);

            if (!$parentBankAccount) {
                Log::error('فشل في العثور على أو إنشاء حساب البنك الأب', [
                    'bank_id' => $bank->id,
                    'bank_name' => $bank->name,
                    'company_id' => $bank->company_id
                ]);
                return null;
            }

            // 2. إنشاء حساب فرعي للحساب البنكي الجديد تحت حساب البنك الأب
            // توليد رمز حساب فريد للبنك الجديد
            $lastBankAccount = ChartOfAccount::where('company_id', $bank->company_id)
                ->where('is_bank_account', true)
                ->where('account_code', 'like', 'BANK-%')
                ->orderBy('account_code', 'desc')
                ->first();

            // تحديد رمز الحساب الجديد
            if ($lastBankAccount) {
                // إذا كان هناك حسابات بنوك سابقة، نزيد الرمز بواحد
                $lastCode = intval(str_replace('BANK-', '', $lastBankAccount->account_code));
                $newAccountCode = 'BANK-' . ($lastCode + 1);
            } else {
                // إذا لم يكن هناك حسابات بنوك سابقة، ننشئ أول حساب
                $newAccountCode = 'BANK-' . rand(1000000, 9999999);
            }

            // التحقق من عدم وجود حساب بنفس الرمز
            while (ChartOfAccount::withTrashed()
                ->where('company_id', $bank->company_id)
                ->where('account_code', $newAccountCode)
                ->exists()
            ) {
                // إذا كان الرمز موجوداً بالفعل، نزيد الرمز بواحد
                $codeNumber = intval(str_replace('BANK-', '', $newAccountCode));
                $newAccountCode = 'BANK-' . ($codeNumber + 1);
            }

            // محاولة إنشاء حساب فرعي للبنك
            try {
                $account = ChartOfAccount::create([
                    'company_id' => $bank->company_id,
                    'account_code' => $newAccountCode,
                    'name' => 'حساب بنك ' . $bank->name . ' رقم ' . rand(1, 9),
                    'name_en' => $bank->name . ' Account #' . rand(1, 9),
                    'account_type' => 'asset',
                    'sub_type' => 'bank',
                    'parent_id' => $parentBankAccount->id,
                    'description' => 'حساب البنك: ' . $bank->name,
                    'is_active' => $bank->is_active,
                    'is_system' => false,
                    'opening_balance' => 0,
                    'current_balance' => 0,
                    'currency' => 'SAR',
                    'balance_type' => 'debit',
                    'is_cash_account' => false,
                    'is_bank_account' => true,
                    'level' => $parentBankAccount->level + 1,
                    'is_parent' => false // حساب فرعي وليس أب
                ]);

                Log::info('تم إنشاء حساب فرعي للبنك في شجرة الحسابات', [
                    'bank_id' => $bank->id,
                    'bank_name' => $bank->name,
                    'account_id' => $account->id,
                    'account_code' => $newAccountCode,
                    'parent_account_id' => $parentBankAccount->id,
                    'user_id' => auth()->id() ?? 'system'
                ]);

                return $account;
            } catch (\Exception $createError) {
                Log::error('خطأ في إنشاء حساب البنك الفرعي', [
                    'bank_id' => $bank->id,
                    'bank_name' => $bank->name,
                    'error' => $createError->getMessage()
                ]);
                return null;
            }
        } catch (\Exception $e) {
            Log::error('خطأ عام في إنشاء حساب البنك: ' . $e->getMessage(), [
                'bank_id' => $bank->id,
                'bank_name' => $bank->name,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return null;
        }
    }

    /**
     * البحث عن حساب البنك الأب أو إنشائه إذا لم يكن موجوداً
     *
     * @param Bank $bank
     * @param ChartOfAccount $currentAssetsAccount
     * @return ChartOfAccount|null
     */
    private function findOrCreateBankParentAccount(Bank $bank, $currentAssetsAccount)
    {
        try {
            // البحث عن حساب البنك الأب باستخدام اسم البنك
            $parentBankAccount = ChartOfAccount::withTrashed()
                ->where('company_id', $bank->company_id)
                ->where('name', $bank->name)
                ->where('is_parent', true)
                ->where('is_bank_account', true)
                ->first();

            if ($parentBankAccount) {
                // إذا كان الحساب محذوفاً، نقوم باسترجاعه
                if ($parentBankAccount->trashed()) {
                    $parentBankAccount->restore();
                    Log::info('تم استرجاع حساب البنك الأب المحذوف', [
                        'account_id' => $parentBankAccount->id,
                        'bank_id' => $bank->id
                    ]);
                }

                // تحديث parent_id إذا كان مختلفاً
                if ($parentBankAccount->parent_id != $currentAssetsAccount->id) {
                    $parentBankAccount->parent_id = $currentAssetsAccount->id;
                    $parentBankAccount->level = $currentAssetsAccount->level + 1;
                    $parentBankAccount->save();
                    Log::info('تم تحديث الحساب الأب لحساب البنك', [
                        'account_id' => $parentBankAccount->id,
                        'parent_id' => $currentAssetsAccount->id
                    ]);
                }

                return $parentBankAccount;
            }

            // إنشاء حساب البنك الأب
            $bankCode = 'BANK-' . $bank->id . '-PARENT';
            $parentBankAccount = ChartOfAccount::create([
                'company_id' => $bank->company_id,
                'account_code' => $bankCode,
                'name' => $bank->name,
                'name_en' => $bank->name,
                'account_type' => 'asset',
                'sub_type' => 'bank',
                'parent_id' => $currentAssetsAccount->id,
                'description' => 'حساب البنك الأب: ' . $bank->name,
                'is_active' => $bank->is_active,
                'is_system' => true,
                'opening_balance' => 0,
                'current_balance' => 0,
                'currency' => 'SAR',
                'balance_type' => 'debit',
                'is_cash_account' => false,
                'is_bank_account' => true,
                'level' => $currentAssetsAccount->level + 1,
                'is_parent' => true
            ]);

            Log::info('تم إنشاء حساب البنك الأب', [
                'account_id' => $parentBankAccount->id,
                'bank_id' => $bank->id,
                'bank_name' => $bank->name
            ]);

            return $parentBankAccount;
        } catch (\Exception $e) {
            Log::error('خطأ في البحث عن أو إنشاء حساب البنك الأب', [
                'bank_id' => $bank->id,
                'bank_name' => $bank->name,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * البحث عن حساب الأصول المتداولة أو إنشائه إذا لم يكن موجوداً
     *
     * @param int $companyId
     * @return ChartOfAccount
     */
    private function findOrCreateCurrentAssetsAccount($companyId)
    {
        // البحث عن حساب الأصول المتداولة
        $currentAssetsAccount = ChartOfAccount::where('company_id', $companyId)
            ->where('account_code', '1100')
            ->where('name', 'الأصول المتداولة')
            ->first();

        if (!$currentAssetsAccount) {
            // البحث عن حساب الأصول
            $assetsAccount = $this->findOrCreateAssetsAccount($companyId);

            // إنشاء حساب الأصول المتداولة تحت حساب الأصول
            $currentAssetsAccount = ChartOfAccount::create([
                'company_id' => $companyId,
                'parent_id' => $assetsAccount->id,
                'account_code' => '1100',
                'name' => 'الأصول المتداولة',
                'name_en' => 'Current Assets',
                'account_type' => 'asset',
                'sub_type' => 'current_asset',
                'description' => 'حساب الأصول المتداولة',
                'is_active' => true,
                'is_system' => true,
                'opening_balance' => 0,
                'current_balance' => 0,
                'currency' => 'SAR',
                'balance_type' => 'debit',
                'is_cash_account' => false,
                'is_bank_account' => false,
                'level' => $assetsAccount->level + 1,
                'is_parent' => true
            ]);

            Log::info('تم إنشاء حساب الأصول المتداولة في شجرة الحسابات', [
                'company_id' => $companyId,
                'account_id' => $currentAssetsAccount->id,
                'user_id' => auth()->id()
            ]);
        }

        return $currentAssetsAccount;
    }

    /**
     * البحث عن حساب الأصول أو إنشائه إذا لم يكن موجوداً
     *
     * @param int $companyId
     * @return ChartOfAccount
     */
    private function findOrCreateAssetsAccount($companyId)
    {
        // البحث عن حساب الأصول
        $assetsAccount = ChartOfAccount::where('company_id', $companyId)
            ->where('account_code', '1000')
            ->where('name', 'الأصول')
            ->first();

        if (!$assetsAccount) {
            // إنشاء حساب الأصول
            $assetsAccount = ChartOfAccount::create([
                'company_id' => $companyId,
                'parent_id' => null,
                'account_code' => '1000',
                'name' => 'الأصول',
                'name_en' => 'Assets',
                'account_type' => 'asset',
                'sub_type' => 'asset',
                'description' => 'حساب الأصول الرئيسي',
                'is_active' => true,
                'is_system' => true,
                'opening_balance' => 0,
                'current_balance' => 0,
                'currency' => 'SAR',
                'balance_type' => 'debit',
                'is_cash_account' => false,
                'is_bank_account' => false,
                'level' => 1,
                'is_parent' => true
            ]);

            Log::info('تم إنشاء حساب الأصول في شجرة الحسابات', [
                'company_id' => $companyId,
                'account_id' => $assetsAccount->id,
                'user_id' => auth()->id()
            ]);
        }

        return $assetsAccount;
    }
}

// حذف الحساب المكرر "حسابات بنكية" في نهاية الشجرة
function deleteDuplicateBankAccounts()
{
    echo "البحث عن حسابات 'حسابات بنكية' المكررة...\n";

    // البحث عن جميع حسابات "حسابات بنكية" في النظام
    $bankAccounts = ChartOfAccount::where('name', 'حسابات بنكية')
        ->where('is_bank_account', true)
        ->get();

    echo "تم العثور على " . $bankAccounts->count() . " حساب باسم 'حسابات بنكية'\n";

    if ($bankAccounts->count() <= 1) {
        echo "لا توجد حسابات مكررة للحذف.\n";
        return;
    }

    // تحديد الحسابات التي يمكن حذفها (الحسابات التي ليس لها أبناء)
    $accountsToDelete = [];
    foreach ($bankAccounts as $account) {
        $childrenCount = ChartOfAccount::where('parent_id', $account->id)->count();
        if ($childrenCount == 0) {
            $accountsToDelete[] = $account;
            echo "الحساب رقم " . $account->id . " ليس له حسابات فرعية ويمكن حذفه.\n";
        } else {
            echo "الحساب رقم " . $account->id . " له " . $childrenCount . " حساب فرعي ولا يمكن حذفه.\n";
        }
    }

    // حذف الحسابات المكررة
    foreach ($accountsToDelete as $account) {
        echo "جاري حذف الحساب رقم " . $account->id . "...\n";
        try {
            $account->delete();
            echo "تم حذف الحساب بنجاح.\n";
        } catch (\Exception $e) {
            echo "فشل في حذف الحساب: " . $e->getMessage() . "\n";
        }
    }
}

// تطبيق التعديلات على الكود الأصلي
function applyChangesToOriginalCode()
{
    echo "جاري تطبيق التعديلات على الكود الأصلي...\n";

    $originalFile = __DIR__ . '/app/Services/BankAccountService.php';
    $backupFile = __DIR__ . '/app/Services/BankAccountService.php.bak';

    // إنشاء نسخة احتياطية من الملف الأصلي
    if (!file_exists($backupFile)) {
        copy($originalFile, $backupFile);
        echo "تم إنشاء نسخة احتياطية من الملف الأصلي.\n";
    }

    // قراءة محتوى الملف الأصلي
    $originalCode = file_get_contents($originalFile);

    // استبدال دالة createBankAccount
    $pattern = '/public function createBankAccount\(Bank \$bank\)\s*\{.*?private function findOrCreateCurrentAssetsAccount/s';
    $replacement = 'public function createBankAccount(Bank $bank)
    {
        try {
            // البحث عن حساب الأصول المتداولة (الحساب الأب للبنوك)
            $currentAssetsAccount = $this->findOrCreateCurrentAssetsAccount($bank->company_id);

            if (!$currentAssetsAccount) {
                Log::error(\'فشل في العثور على أو إنشاء حساب الأصول المتداولة\', [
                    \'bank_id\' => $bank->id,
                    \'company_id\' => $bank->company_id
                ]);
                return null;
            }

            // 1. البحث عن حساب البنك الأب (بنفس اسم البنك) أو إنشائه إذا لم يكن موجوداً
            $parentBankAccount = $this->findOrCreateBankParentAccount($bank, $currentAssetsAccount);

            if (!$parentBankAccount) {
                Log::error(\'فشل في العثور على أو إنشاء حساب البنك الأب\', [
                    \'bank_id\' => $bank->id,
                    \'bank_name\' => $bank->name,
                    \'company_id\' => $bank->company_id
                ]);
                return null;
            }

            // 2. إنشاء حساب فرعي للحساب البنكي الجديد تحت حساب البنك الأب
            // توليد رمز حساب فريد للبنك الجديد
            $lastBankAccount = ChartOfAccount::where(\'company_id\', $bank->company_id)
                ->where(\'is_bank_account\', true)
                ->where(\'account_code\', \'like\', \'BANK-%\')
                ->orderBy(\'account_code\', \'desc\')
                ->first();

            // تحديد رمز الحساب الجديد
            if ($lastBankAccount) {
                // إذا كان هناك حسابات بنوك سابقة، نزيد الرمز بواحد
                $lastCode = intval(str_replace(\'BANK-\', \'\', $lastBankAccount->account_code));
                $newAccountCode = \'BANK-\' . ($lastCode + 1);
            } else {
                // إذا لم يكن هناك حسابات بنوك سابقة، ننشئ أول حساب
                $newAccountCode = \'BANK-\' . rand(1000000, 9999999);
            }

            // التحقق من عدم وجود حساب بنفس الرمز
            while (ChartOfAccount::withTrashed()
                ->where(\'company_id\', $bank->company_id)
                ->where(\'account_code\', $newAccountCode)
                ->exists()
            ) {
                // إذا كان الرمز موجوداً بالفعل، نزيد الرمز بواحد
                $codeNumber = intval(str_replace(\'BANK-\', \'\', $newAccountCode));
                $newAccountCode = \'BANK-\' . ($codeNumber + 1);
            }

            // محاولة إنشاء حساب فرعي للبنك
            try {
                $account = ChartOfAccount::create([
                    \'company_id\' => $bank->company_id,
                    \'account_code\' => $newAccountCode,
                    \'name\' => \'حساب بنك \' . $bank->name . \' رقم \' . rand(1, 9),
                    \'name_en\' => $bank->name . \' Account #\' . rand(1, 9),
                    \'account_type\' => \'asset\',
                    \'sub_type\' => \'bank\',
                    \'parent_id\' => $parentBankAccount->id,
                    \'description\' => \'حساب البنك: \' . $bank->name,
                    \'is_active\' => $bank->is_active,
                    \'is_system\' => false,
                    \'opening_balance\' => 0,
                    \'current_balance\' => 0,
                    \'currency\' => \'SAR\',
                    \'balance_type\' => \'debit\',
                    \'is_cash_account\' => false,
                    \'is_bank_account\' => true,
                    \'level\' => $parentBankAccount->level + 1,
                    \'is_parent\' => false // حساب فرعي وليس أب
                ]);

                Log::info(\'تم إنشاء حساب فرعي للبنك في شجرة الحسابات\', [
                    \'bank_id\' => $bank->id,
                    \'bank_name\' => $bank->name,
                    \'account_id\' => $account->id,
                    \'account_code\' => $newAccountCode,
                    \'parent_account_id\' => $parentBankAccount->id,
                    \'user_id\' => auth()->id() ?? \'system\'
                ]);

                return $account;
            } catch (\\Exception $createError) {
                Log::error(\'خطأ في إنشاء حساب البنك الفرعي\', [
                    \'bank_id\' => $bank->id,
                    \'bank_name\' => $bank->name,
                    \'error\' => $createError->getMessage()
                ]);
                return null;
            }
        } catch (\\Exception $e) {
            Log::error(\'خطأ عام في إنشاء حساب البنك: \' . $e->getMessage(), [
                \'bank_id\' => $bank->id,
                \'bank_name\' => $bank->name,
                \'error\' => $e->getMessage(),
                \'trace\' => $e->getTraceAsString()
            ]);
            return null;
        }
    }

    /**
     * البحث عن حساب البنك الأب أو إنشائه إذا لم يكن موجوداً
     *
     * @param Bank $bank
     * @param ChartOfAccount $currentAssetsAccount
     * @return ChartOfAccount|null
     */
    private function findOrCreateBankParentAccount(Bank $bank, $currentAssetsAccount)
    {
        try {
            // البحث عن حساب البنك الأب باستخدام اسم البنك
            $parentBankAccount = ChartOfAccount::withTrashed()
                ->where(\'company_id\', $bank->company_id)
                ->where(\'name\', $bank->name)
                ->where(\'is_parent\', true)
                ->where(\'is_bank_account\', true)
                ->first();

            if ($parentBankAccount) {
                // إذا كان الحساب محذوفاً، نقوم باسترجاعه
                if ($parentBankAccount->trashed()) {
                    $parentBankAccount->restore();
                    Log::info(\'تم استرجاع حساب البنك الأب المحذوف\', [
                        \'account_id\' => $parentBankAccount->id,
                        \'bank_id\' => $bank->id
                    ]);
                }

                // تحديث parent_id إذا كان مختلفاً
                if ($parentBankAccount->parent_id != $currentAssetsAccount->id) {
                    $parentBankAccount->parent_id = $currentAssetsAccount->id;
                    $parentBankAccount->level = $currentAssetsAccount->level + 1;
                    $parentBankAccount->save();
                    Log::info(\'تم تحديث الحساب الأب لحساب البنك\', [
                        \'account_id\' => $parentBankAccount->id,
                        \'parent_id\' => $currentAssetsAccount->id
                    ]);
                }

                return $parentBankAccount;
            }

            // إنشاء حساب البنك الأب
            $bankCode = \'BANK-\' . $bank->id . \'-PARENT\';
            $parentBankAccount = ChartOfAccount::create([
                \'company_id\' => $bank->company_id,
                \'account_code\' => $bankCode,
                \'name\' => $bank->name,
                \'name_en\' => $bank->name,
                \'account_type\' => \'asset\',
                \'sub_type\' => \'bank\',
                \'parent_id\' => $currentAssetsAccount->id,
                \'description\' => \'حساب البنك الأب: \' . $bank->name,
                \'is_active\' => $bank->is_active,
                \'is_system\' => true,
                \'opening_balance\' => 0,
                \'current_balance\' => 0,
                \'currency\' => \'SAR\',
                \'balance_type\' => \'debit\',
                \'is_cash_account\' => false,
                \'is_bank_account\' => true,
                \'level\' => $currentAssetsAccount->level + 1,
                \'is_parent\' => true
            ]);

            Log::info(\'تم إنشاء حساب البنك الأب\', [
                \'account_id\' => $parentBankAccount->id,
                \'bank_id\' => $bank->id,
                \'bank_name\' => $bank->name
            ]);

            return $parentBankAccount;
        } catch (\\Exception $e) {
            Log::error(\'خطأ في البحث عن أو إنشاء حساب البنك الأب\', [
                \'bank_id\' => $bank->id,
                \'bank_name\' => $bank->name,
                \'error\' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * البحث عن حساب الأصول المتداولة أو إنشائه إذا لم يكن موجوداً
     *
     * @param int $companyId
     * @return ChartOfAccount
     */
    private function findOrCreateCurrentAssetsAccount';

    // تطبيق التعديلات
    $newCode = preg_replace($pattern, $replacement, $originalCode);

    // حذف دالة findOrCreateParentBankAccount القديمة
    $pattern2 = '/private function findOrCreateParentBankAccount\(\$companyId, \$currentAssetsAccount\)\s*\{.*?}\s*}/s';
    $newCode = preg_replace($pattern2, '}', $newCode);

    // كتابة الكود المعدل إلى الملف الأصلي
    file_put_contents($originalFile, $newCode);

    echo "تم تطبيق التعديلات على الكود الأصلي بنجاح.\n";
}

// تنفيذ الوظائف
echo "بدء تنفيذ التعديلات على نظام الحسابات البنكية...\n";

// حذف الحسابات المكررة
deleteDuplicateBankAccounts();

// تطبيق التعديلات على الكود الأصلي
applyChangesToOriginalCode();

echo "تم الانتهاء من تنفيذ التعديلات بنجاح.\n";
