<?php

/**
 * سكريبت تشخيصي شامل لعملية إنشاء حساب البنك
 * يقوم بفحص جميع جوانب عملية إنشاء حساب البنك
 */

require_once __DIR__ . '/vendor/autoload.php';

use App\Models\Bank;
use App\Models\ChartOfAccount;
use App\Services\BankAccountService;
use Illuminate\Support\Facades\DB;

// تهيئة Laravel
$app = require_once __DIR__ . '/bootstrap/app.php';
$app->make(\Illuminate\Contracts\Console\Kernel::class)->bootstrap();

echo "تشخيص شامل لعملية إنشاء حساب البنك...\n\n";

// فحص هيكل الحسابات للشركة 3
echo "1. فحص هيكل الحسابات للشركة 3:\n";
$assetsAccount = ChartOfAccount::where('company_id', 3)
    ->where('account_code', '1000')
    ->first();

if ($assetsAccount) {
    echo "  - حساب الأصول موجود (ID: {$assetsAccount->id})\n";

    $currentAssetsAccount = ChartOfAccount::where('company_id', 3)
        ->where('parent_id', $assetsAccount->id)
        ->where('account_code', '1100')
        ->first();

    if ($currentAssetsAccount) {
        echo "  - حساب الأصول المتداولة موجود (ID: {$currentAssetsAccount->id})\n";

        $bankAccount = ChartOfAccount::where('company_id', 3)
            ->where('parent_id', $currentAssetsAccount->id)
            ->where('account_code', 'BANK')
            ->first();

        if ($bankAccount) {
            echo "  - حساب البنوك موجود (ID: {$bankAccount->id})\n";
            echo "    - المستوى: {$bankAccount->level}\n";

            $individualBankAccounts = ChartOfAccount::where('company_id', 3)
                ->where('parent_id', $bankAccount->id)
                ->where('is_bank_account', true)
                ->get();

            echo "  - عدد الحسابات البنكية الفردية: " . $individualBankAccounts->count() . "\n";

            foreach ($individualBankAccounts as $account) {
                echo "    - حساب {$account->name} (ID: {$account->id}), المستوى: {$account->level}\n";

                if ($account->level != $bankAccount->level + 1) {
                    echo "      ! تحذير: مستوى الحساب غير صحيح. يجب أن يكون " . ($bankAccount->level + 1) . "\n";
                }
            }
        } else {
            echo "  ! خطأ: حساب البنوك غير موجود تحت الأصول المتداولة\n";
        }
    } else {
        echo "  ! خطأ: حساب الأصول المتداولة غير موجود تحت الأصول\n";
    }
} else {
    echo "  ! خطأ: حساب الأصول غير موجود للشركة 3\n";
}

// فحص قيد الفريد (Unique Constraint) في جدول chart_of_accounts
echo "\n2. فحص قيد الفريد (Unique Constraint) في جدول chart_of_accounts:\n";

$uniqueConstraints = DB::select("SHOW CREATE TABLE chart_of_accounts");
echo "  - تعريف الجدول:\n";
echo "    " . $uniqueConstraints[0]->{"Create Table"} . "\n";

// فحص وجود حسابات بنفس الرمز للشركة 3
echo "\n3. فحص وجود حسابات بنفس الرمز للشركة 3:\n";

$duplicateAccounts = DB::select("
    SELECT account_code, COUNT(*) as count
    FROM chart_of_accounts
    WHERE company_id = 3
    GROUP BY account_code
    HAVING COUNT(*) > 1
");

if (count($duplicateAccounts) > 0) {
    echo "  ! تحذير: تم العثور على حسابات متكررة بنفس الرمز:\n";
    foreach ($duplicateAccounts as $duplicate) {
        echo "    - الرمز: {$duplicate->account_code}, العدد: {$duplicate->count}\n";

        $accounts = ChartOfAccount::where('company_id', 3)
            ->where('account_code', $duplicate->account_code)
            ->get();

        foreach ($accounts as $account) {
            echo "      * ID: {$account->id}, الاسم: {$account->name}, المستوى: {$account->level}, محذوف: " . ($account->deleted_at ? 'نعم' : 'لا') . "\n";
        }
    }
} else {
    echo "  - لا توجد حسابات متكررة بنفس الرمز للشركة 3\n";
}

// محاولة إنشاء بنك جديد واختبار إنشاء حساب له
echo "\n4. اختبار إنشاء بنك جديد وحساب له:\n";

try {
    // بدء المعاملة
    DB::beginTransaction();

    // إنشاء بنك جديد للاختبار
    $testBankName = 'بنك اختباري ' . date('YmdHis');
    $bank = new Bank();
    $bank->name = $testBankName;
    $bank->company_id = 3;
    $bank->is_active = true;
    $bank->save();

    echo "  - تم إنشاء بنك اختباري بنجاح (ID: {$bank->id})\n";

    // إنشاء حساب للبنك
    $bankAccountService = new BankAccountService();
    $account = $bankAccountService->createBankAccount($bank);

    if ($account) {
        echo "  - تم إنشاء حساب للبنك بنجاح (ID: {$account->id})\n";
        echo "    - الرمز: {$account->account_code}\n";
        echo "    - المستوى: {$account->level}\n";
        echo "    - الحساب الأب: {$account->parent_id}\n";
    } else {
        echo "  ! فشل في إنشاء حساب للبنك\n";

        // فحص سجل الأخطاء
        $latestError = DB::table('logs')
            ->where('level', 'error')
            ->orderBy('created_at', 'desc')
            ->first();

        if ($latestError) {
            echo "  - آخر خطأ مسجل: {$latestError->message}\n";
        }
    }

    // التراجع عن المعاملة لتجنب إنشاء بيانات اختبارية
    DB::rollBack();
    echo "  - تم التراجع عن المعاملة لتجنب إنشاء بيانات اختبارية\n";
} catch (\Exception $e) {
    // التراجع عن المعاملة
    DB::rollBack();

    echo "  ! خطأ أثناء الاختبار: " . $e->getMessage() . "\n";
    echo "    - الملف: " . $e->getFile() . "\n";
    echo "    - السطر: " . $e->getLine() . "\n";
}

echo "\nتم الانتهاء من التشخيص الشامل.\n";
