<?php
require_once __DIR__ . '/vendor/autoload.php';

use App\Models\Bank;
use App\Models\ChartOfAccount;
use App\Services\BankAccountService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Http\Request;
use App\Http\Controllers\BankController;

// Initialize Laravel application
$app = require_once __DIR__ . '/bootstrap/app.php';
$app->make(\Illuminate\Contracts\Console\Kernel::class)->bootstrap();

try {
    // Set up a mock request with the necessary data
    $requestData = [
        'name' => 'Debug Test Bank ' . date('YmdHis'),
        'branch' => 'Main Branch',
        'address' => 'Test Address',
        'contact_person' => 'Test Contact',
        'phone' => '123456789',
        'email' => 'test@example.com',
        'swift_code' => 'TESTSWIFT',
        'notes' => 'Test Notes',
        'status' => true,
    ];

    echo "Creating a test bank with name: {$requestData['name']}\n";

    // Create a request object
    $request = new Request($requestData);

    // Set up authentication
    $user = DB::table('users')->where('id', 10)->first(); // Assuming user ID 10 exists
    if (!$user) {
        echo "User not found. Please specify a valid user ID.\n";
        exit;
    }

    auth()->loginUsingId($user->id);
    echo "Authenticated as user ID: {$user->id}\n";

    // Create a new BankAccountService instance
    $bankAccountService = new BankAccountService();

    // Create a new BankController instance
    $bankController = new BankController($bankAccountService);

    // Enable query logging to see what SQL queries are being executed
    DB::enableQueryLog();

    // Begin transaction
    DB::beginTransaction();

    try {
        // Call the store method directly
        echo "Calling BankController store method...\n";
        $response = $bankController->store($request);

        // Check the response
        if ($response) {
            echo "Controller returned a response.\n";

            // If it's a redirect response, get the session data
            if (method_exists($response, 'getSession')) {
                $session = $response->getSession();
                if ($session->has('success')) {
                    echo "Success message: " . $session->get('success') . "\n";
                }
                if ($session->has('error')) {
                    echo "Error message: " . $session->get('error') . "\n";
                }
            }
        } else {
            echo "Controller did not return a response.\n";
        }

        // Get the query log
        $queries = DB::getQueryLog();
        echo "\nSQL Queries executed:\n";
        foreach ($queries as $index => $query) {
            echo ($index + 1) . ". " . $query['query'] . "\n";
            echo "   Bindings: " . json_encode($query['bindings']) . "\n";
            echo "   Time: " . $query['time'] . "ms\n\n";
        }

        // Check if a bank was created
        $bank = Bank::where('name', $requestData['name'])->first();
        if ($bank) {
            echo "Bank was created with ID: {$bank->id}\n";

            // Check if a chart of account was created for this bank
            $account = ChartOfAccount::where('name', $bank->name)
                ->where('is_bank_account', true)
                ->first();

            if ($account) {
                echo "Chart of account was created for the bank:\n";
                echo "ID: {$account->id}\n";
                echo "Account Code: {$account->account_code}\n";
                echo "Name: {$account->name}\n";
                echo "Is Parent: " . ($account->is_parent ? 'Yes' : 'No') . "\n";
                echo "Parent ID: " . ($account->parent_id ?? 'None') . "\n";
            } else {
                echo "No chart of account was created for the bank.\n";
            }
        } else {
            echo "No bank was created with the name: {$requestData['name']}\n";
        }

        // Rollback transaction to avoid creating test data
        DB::rollBack();
        echo "\nTransaction rolled back to avoid creating test data.\n";
    } catch (\Exception $e) {
        // Rollback transaction
        DB::rollBack();

        echo "Error in controller: " . $e->getMessage() . "\n";
        echo "File: " . $e->getFile() . "\n";
        echo "Line: " . $e->getLine() . "\n";
        echo "Trace:\n" . $e->getTraceAsString() . "\n";

        // Get the query log
        $queries = DB::getQueryLog();
        echo "\nSQL Queries executed before error:\n";
        foreach ($queries as $index => $query) {
            echo ($index + 1) . ". " . $query['query'] . "\n";
            echo "   Bindings: " . json_encode($query['bindings']) . "\n";
            echo "   Time: " . $query['time'] . "ms\n\n";
        }
    }
} catch (\Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    echo "File: " . $e->getFile() . "\n";
    echo "Line: " . $e->getLine() . "\n";
}
