<?php
require_once __DIR__ . '/vendor/autoload.php';

use App\Models\Bank;
use App\Models\ChartOfAccount;
use App\Services\BankAccountService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

// Initialize Laravel application
$app = require_once __DIR__ . '/bootstrap/app.php';
$app->make(\Illuminate\Contracts\Console\Kernel::class)->bootstrap();

try {
    // Get a bank to test with
    $bank = Bank::first();

    if (!$bank) {
        echo "No banks found in the database. Please create a bank first.\n";
        exit;
    }

    echo "Testing with bank: {$bank->name} (ID: {$bank->id})\n";

    // Create a new instance of BankAccountService
    $bankAccountService = new BankAccountService();

    // Begin transaction
    DB::beginTransaction();

    try {
        // Try to create a bank account
        echo "Attempting to create a bank account...\n";
        $account = $bankAccountService->createBankAccount($bank);

        if ($account) {
            echo "Successfully created bank account:\n";
            echo "ID: {$account->id}\n";
            echo "Account Code: {$account->account_code}\n";
            echo "Name: {$account->name}\n";
            echo "Is Parent: " . ($account->is_parent ? 'Yes' : 'No') . "\n";
            echo "Parent ID: " . ($account->parent_id ?? 'None') . "\n";
        } else {
            echo "Failed to create bank account. No exception was thrown, but no account was returned.\n";
        }

        // Check if there's already an account for this bank
        echo "\nChecking if bank already has an account...\n";
        $existingAccount = $bankAccountService->findBankAccount($bank);

        if ($existingAccount) {
            echo "Bank already has an account:\n";
            echo "ID: {$existingAccount->id}\n";
            echo "Account Code: {$existingAccount->account_code}\n";
            echo "Name: {$existingAccount->name}\n";
            echo "Is Parent: " . ($existingAccount->is_parent ? 'Yes' : 'No') . "\n";
            echo "Parent ID: " . ($existingAccount->parent_id ?? 'None') . "\n";
        } else {
            echo "Bank does not have an existing account.\n";
        }

        // Rollback transaction to avoid creating duplicate accounts
        DB::rollBack();
        echo "\nTransaction rolled back to avoid creating duplicate accounts.\n";
    } catch (\Exception $e) {
        // Rollback transaction
        DB::rollBack();

        echo "Error creating bank account: " . $e->getMessage() . "\n";
        echo "File: " . $e->getFile() . "\n";
        echo "Line: " . $e->getLine() . "\n";
        echo "Trace:\n" . $e->getTraceAsString() . "\n";
    }

    // Check for unique constraints in the database
    echo "\nChecking for unique constraints that might cause conflicts...\n";
    $uniqueConstraints = DB::select("
        SELECT TABLE_NAME, COLUMN_NAME, CONSTRAINT_NAME
        FROM information_schema.KEY_COLUMN_USAGE
        WHERE TABLE_SCHEMA = 'elevators_management'
        AND TABLE_NAME = 'chart_of_accounts'
        AND CONSTRAINT_NAME LIKE '%unique%'
    ");

    foreach ($uniqueConstraints as $constraint) {
        echo "Unique constraint: {$constraint->CONSTRAINT_NAME} on column {$constraint->COLUMN_NAME}\n";
    }
} catch (\Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    echo "File: " . $e->getFile() . "\n";
    echo "Line: " . $e->getLine() . "\n";
}
