<?php

namespace Database\Seeders;

use App\Models\Permission;
use App\Models\Role;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class PermissionsSeeder extends Seeder
{
    /**
     * تشغيل عملية البذر.
     *
     * @return void
     */
    public function run()
    {
        // تعطيل فحص المفاتيح الأجنبية مؤقتاً
        Schema::disableForeignKeyConstraints();

        // إنشاء الصلاحيات الأساسية
        $this->createBasicPermissions();

        // إنشاء الأدوار الأساسية
        $this->createBasicRoles();

        // تعيين الصلاحيات للأدوار
        $this->assignPermissionsToRoles();

        // التأكد من أن المستخدم سوبر أدمن لديه دور المدير
        $this->ensureSuperAdminHasAdminRole();

        // إعادة تفعيل فحص المفاتيح الأجنبية
        Schema::enableForeignKeyConstraints();
    }

    /**
     * إنشاء الصلاحيات الأساسية.
     *
     * @return void
     */
    private function createBasicPermissions()
    {
        // حذف الصلاحيات الموجودة إذا كانت موجودة
        Permission::truncate();

        // مجموعات الصلاحيات
        $permissionGroups = [
            'users' => 'إدارة المستخدمين',
            'roles' => 'إدارة الأدوار',
            'permissions' => 'إدارة الصلاحيات',
            'companies' => 'إدارة الشركات',
            'branches' => 'إدارة الفروع',
            'elevators' => 'إدارة المصاعد',
            'customers' => 'إدارة العملاء',
            'locations' => 'إدارة المواقع',
            'contracts' => 'إدارة العقود',
            'invoices' => 'إدارة الفواتير',
            'invoice_templates' => 'إدارة قوالب الفواتير',
            'invoice_installments' => 'إدارة أقساط الفواتير',
            'payments' => 'إدارة المدفوعات',
            'reports' => 'إدارة التقارير',
            'settings' => 'إدارة الإعدادات',
            'security' => 'إدارة الأمان',
        ];

        // الصلاحيات الأساسية لكل مجموعة
        $basicPermissions = [
            'view_%s' => 'عرض %s',
            'create_%s' => 'إنشاء %s',
            'edit_%s' => 'تعديل %s',
            'delete_%s' => 'حذف %s',
        ];

        // إنشاء الصلاحيات لكل مجموعة
        foreach ($permissionGroups as $group => $groupName) {
            // إنشاء صلاحية المجموعة الرئيسية
            $parentPermission = Permission::create([
                'name' => $groupName,
                'slug' => 'manage_' . $group,
                'group' => $group,
                'description' => 'إدارة كاملة لـ ' . $groupName,
                'level' => 0,
                'path' => null,
                'type' => 'group',
                'resource' => $group,
                'order' => 0,
                'is_active' => true,
            ]);

            // تحديث المسار بعد الإنشاء
            $parentPermission->path = $parentPermission->id;
            $parentPermission->save();

            // إنشاء الصلاحيات الفرعية
            $order = 1;
            foreach ($basicPermissions as $slug => $name) {
                $formattedSlug = sprintf($slug, rtrim($group, 's'));
                $formattedName = sprintf($name, $groupName);

                $permission = Permission::create([
                    'parent_id' => $parentPermission->id,
                    'name' => $formattedName,
                    'slug' => $formattedSlug,
                    'group' => $group,
                    'description' => $formattedName,
                    'level' => 1,
                    'path' => null,
                    'type' => 'action',
                    'resource' => $group,
                    'order' => $order++,
                    'is_active' => true,
                ]);

                // تحديث المسار بعد الإنشاء
                $permission->path = $parentPermission->path . '.' . $permission->id;
                $permission->save();
            }
        }

        // إضافة صلاحيات خاصة
        $specialPermissions = [
            [
                'name' => 'عرض سجل النشاطات',
                'slug' => 'view_activity_logs',
                'group' => 'security',
                'description' => 'عرض سجل النشاطات في النظام',
                'type' => 'action',
                'resource' => 'activity_logs',
            ],
            [
                'name' => 'عرض سجل تسجيل الدخول',
                'slug' => 'view_login_history',
                'group' => 'security',
                'description' => 'عرض سجل تسجيل الدخول في النظام',
                'type' => 'action',
                'resource' => 'login_history',
            ],
            [
                'name' => 'إدارة المصادقة الثنائية',
                'slug' => 'manage_two_factor',
                'group' => 'security',
                'description' => 'إدارة إعدادات المصادقة الثنائية',
                'type' => 'action',
                'resource' => 'two_factor',
            ],
            [
                'name' => 'إدارة النسخ الاحتياطية',
                'slug' => 'manage_backups',
                'group' => 'settings',
                'description' => 'إدارة النسخ الاحتياطية للنظام',
                'type' => 'action',
                'resource' => 'backups',
            ],
            [
                'name' => 'إدارة إعدادات النظام',
                'slug' => 'manage_system_settings',
                'group' => 'settings',
                'description' => 'إدارة إعدادات النظام العامة',
                'type' => 'action',
                'resource' => 'system_settings',
            ],
        ];

        // إنشاء الصلاحيات الخاصة
        foreach ($specialPermissions as $permissionData) {
            $parentGroup = Permission::where('slug', 'manage_' . $permissionData['group'])->first();

            if ($parentGroup) {
                $permission = Permission::create([
                    'parent_id' => $parentGroup->id,
                    'name' => $permissionData['name'],
                    'slug' => $permissionData['slug'],
                    'group' => $permissionData['group'],
                    'description' => $permissionData['description'],
                    'level' => 1,
                    'path' => null,
                    'type' => $permissionData['type'],
                    'resource' => $permissionData['resource'],
                    'order' => 10, // ترتيب عالي للصلاحيات الخاصة
                    'is_active' => true,
                ]);

                // تحديث المسار بعد الإنشاء
                $permission->path = $parentGroup->path . '.' . $permission->id;
                $permission->save();
            }
        }
    }

    /**
     * إنشاء الأدوار الأساسية.
     *
     * @return void
     */
    private function createBasicRoles()
    {
        // حذف الأدوار الموجودة إذا كانت موجودة
        Role::truncate();
        DB::table('role_user')->truncate();
        DB::table('permission_role')->truncate();

        // إنشاء دور المدير
        Role::create([
            'name' => 'مدير النظام',
            'slug' => 'admin',
            'description' => 'مدير النظام مع كامل الصلاحيات',
            'is_system_role' => true,
            'level' => 0,
            'path' => '1',
            'order' => 1,
            'is_active' => true,
            'is_default' => false,
        ]);

        // إنشاء دور المستخدم العادي
        Role::create([
            'name' => 'مستخدم',
            'slug' => 'user',
            'description' => 'مستخدم عادي مع صلاحيات محدودة',
            'is_system_role' => true,
            'level' => 0,
            'path' => '2',
            'order' => 2,
            'is_active' => true,
            'is_default' => true,
        ]);

        // إنشاء دور مدير الفرع
        Role::create([
            'name' => 'مدير فرع',
            'slug' => 'branch_manager',
            'description' => 'مدير فرع مع صلاحيات إدارة الفرع',
            'is_system_role' => true,
            'level' => 0,
            'path' => '3',
            'order' => 3,
            'is_active' => true,
            'is_default' => false,
        ]);

        // إنشاء دور فني
        Role::create([
            'name' => 'فني',
            'slug' => 'technician',
            'description' => 'فني مع صلاحيات محددة للصيانة',
            'is_system_role' => true,
            'level' => 0,
            'path' => '4',
            'order' => 4,
            'is_active' => true,
            'is_default' => false,
        ]);

        // إنشاء دور محاسب
        Role::create([
            'name' => 'محاسب',
            'slug' => 'accountant',
            'description' => 'محاسب مع صلاحيات مالية',
            'is_system_role' => true,
            'level' => 0,
            'path' => '5',
            'order' => 5,
            'is_active' => true,
            'is_default' => false,
        ]);
    }

    /**
     * تعيين الصلاحيات للأدوار.
     *
     * @return void
     */
    private function assignPermissionsToRoles()
    {
        // الحصول على الأدوار
        $adminRole = Role::where('slug', 'admin')->first();
        $userRole = Role::where('slug', 'user')->first();
        $branchManagerRole = Role::where('slug', 'branch_manager')->first();
        $technicianRole = Role::where('slug', 'technician')->first();
        $accountantRole = Role::where('slug', 'accountant')->first();

        // تعيين جميع الصلاحيات لدور المدير
        if ($adminRole) {
            $allPermissions = Permission::all();
            $adminRole->permissions()->sync($allPermissions->pluck('id')->toArray());
        }

        // تعيين صلاحيات محدودة لدور المستخدم العادي
        if ($userRole) {
            $userPermissions = Permission::whereIn('slug', [
                'view_elevator',
                'view_customer',
                'view_location',
                'view_contract',
            ])->get();
            $userRole->permissions()->sync($userPermissions->pluck('id')->toArray());
        }

        // تعيين صلاحيات لدور مدير الفرع
        if ($branchManagerRole) {
            $branchManagerPermissions = Permission::whereIn('slug', [
                'manage_elevators',
                'manage_customers',
                'manage_locations',
                'manage_contracts',
                'view_invoice',
                'view_payment',
                'view_report',
            ])->orWhere('type', 'action')->whereIn('group', [
                'elevators',
                'customers',
                'locations',
                'contracts',
            ])->get();
            $branchManagerRole->permissions()->sync($branchManagerPermissions->pluck('id')->toArray());
        }

        // تعيين صلاحيات لدور الفني
        if ($technicianRole) {
            $technicianPermissions = Permission::whereIn('slug', [
                'view_elevator',
                'edit_elevator',
                'view_customer',
                'view_location',
                'view_contract',
            ])->get();
            $technicianRole->permissions()->sync($technicianPermissions->pluck('id')->toArray());
        }

        // تعيين صلاحيات لدور المحاسب
        if ($accountantRole) {
            $accountantPermissions = Permission::whereIn('slug', [
                'view_customer',
                'manage_invoices',
                'manage_invoice_templates',
                'manage_invoice_installments',
                'manage_payments',
                'view_report',
            ])->orWhere('type', 'action')->whereIn('group', [
                'invoices',
                'invoice_templates',
                'invoice_installments',
                'payments',
            ])->get();
            $accountantRole->permissions()->sync($accountantPermissions->pluck('id')->toArray());
        }
    }

    /**
     * التأكد من أن المستخدم الأول لديه دور المدير.
     *
     * @return void
     */
    private function ensureSuperAdminHasAdminRole()
    {
        // الحصول على دور المدير
        $adminRole = Role::where('slug', 'admin')->first();

        if ($adminRole) {
            // الحصول على المستخدم الأول (غالباً ما يكون المدير)
            $firstUser = User::first();

            if ($firstUser) {
                // تعيين دور المدير للمستخدم إذا لم يكن لديه بالفعل
                if (!$firstUser->hasRole($adminRole)) {
                    $firstUser->roles()->syncWithoutDetaching([$adminRole->id]);
                }
            }
        }
    }
}
