<?php

namespace Database\Seeders;

use App\Models\Permission;
use App\Models\Role;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Schema;

class PaymentAndTaxPermissionsSeeder extends Seeder
{
    /**
     * تشغيل عملية البذر.
     *
     * @return void
     */
    public function run()
    {
        // تعطيل فحص المفاتيح الأجنبية مؤقتاً
        Schema::disableForeignKeyConstraints();

        // إضافة صلاحيات المدفوعات وإعدادات الضرائب
        $this->createPaymentAndTaxPermissions();

        // تعيين الصلاحيات للأدوار
        $this->assignPermissionsToRoles();

        // إعادة تفعيل فحص المفاتيح الأجنبية
        Schema::enableForeignKeyConstraints();
    }

    /**
     * إنشاء صلاحيات المدفوعات وإعدادات الضرائب.
     *
     * @return void
     */
    private function createPaymentAndTaxPermissions()
    {
        // صلاحيات المدفوعات الإضافية
        $paymentPermissions = [
            [
                'name' => 'تغيير حالة الدفعة',
                'slug' => 'change_payment_status',
                'group' => 'payments',
                'description' => 'تغيير حالة الدفعة (مكتمل، قيد الانتظار، فشل، مسترجع)',
                'type' => 'action',
                'resource' => 'payments',
            ],
            [
                'name' => 'طباعة إيصال الدفع',
                'slug' => 'print_payment_receipt',
                'group' => 'payments',
                'description' => 'طباعة إيصال الدفع',
                'type' => 'action',
                'resource' => 'payments',
            ],
        ];

        // صلاحيات إعدادات الضرائب
        $taxSettingPermissions = [
            [
                'name' => 'إدارة إعدادات الضرائب',
                'slug' => 'manage_tax_settings',
                'group' => 'settings',
                'description' => 'إدارة كاملة لإعدادات الضرائب',
                'type' => 'group',
                'resource' => 'tax_settings',
            ],
            [
                'name' => 'عرض إعدادات الضرائب',
                'slug' => 'tax_settings.view',
                'group' => 'settings',
                'description' => 'عرض إعدادات الضرائب',
                'type' => 'action',
                'resource' => 'tax_settings',
            ],
            [
                'name' => 'إنشاء إعدادات الضرائب',
                'slug' => 'tax_settings.create',
                'group' => 'settings',
                'description' => 'إنشاء إعدادات الضرائب',
                'type' => 'action',
                'resource' => 'tax_settings',
            ],
            [
                'name' => 'تعديل إعدادات الضرائب',
                'slug' => 'tax_settings.edit',
                'group' => 'settings',
                'description' => 'تعديل إعدادات الضرائب',
                'type' => 'action',
                'resource' => 'tax_settings',
            ],
            [
                'name' => 'حذف إعدادات الضرائب',
                'slug' => 'tax_settings.delete',
                'group' => 'settings',
                'description' => 'حذف إعدادات الضرائب',
                'type' => 'action',
                'resource' => 'tax_settings',
            ],
        ];

        // إنشاء صلاحيات المدفوعات الإضافية
        $this->createPermissions($paymentPermissions);

        // إنشاء صلاحيات إعدادات الضرائب
        $this->createPermissions($taxSettingPermissions);
    }

    /**
     * إنشاء الصلاحيات.
     *
     * @param array $permissions
     * @return void
     */
    private function createPermissions($permissions)
    {
        foreach ($permissions as $permissionData) {
            // تحديد المجموعة الأب
            $parentGroup = null;

            if ($permissionData['type'] == 'group') {
                $parentGroup = Permission::where('slug', 'manage_' . $permissionData['group'])->first();
            } else {
                if (strpos($permissionData['slug'], 'tax_settings') !== false) {
                    $parentGroup = Permission::where('slug', 'manage_tax_settings')->first();

                    // إذا لم يتم العثور على المجموعة، استخدم مجموعة الإعدادات
                    if (!$parentGroup) {
                        $parentGroup = Permission::where('slug', 'manage_settings')->first();
                    }
                } else {
                    $parentGroup = Permission::where('slug', 'manage_' . $permissionData['group'])->first();
                }
            }

            // إنشاء الصلاحية فقط إذا لم تكن موجودة بالفعل
            $existingPermission = Permission::where('slug', $permissionData['slug'])->first();

            if (!$existingPermission) {
                $permission = Permission::create([
                    'parent_id' => $parentGroup ? $parentGroup->id : null,
                    'name' => $permissionData['name'],
                    'slug' => $permissionData['slug'],
                    'group' => $permissionData['group'],
                    'description' => $permissionData['description'],
                    'level' => $parentGroup ? $parentGroup->level + 1 : 0,
                    'path' => null,
                    'type' => $permissionData['type'],
                    'resource' => $permissionData['resource'],
                    'order' => 10, // ترتيب عالي للصلاحيات الإضافية
                    'is_active' => true,
                ]);

                // تحديث المسار بعد الإنشاء
                if ($parentGroup) {
                    $permission->path = $parentGroup->path . '.' . $permission->id;
                } else {
                    $permission->path = $permission->id;
                }

                $permission->save();
            }
        }
    }

    /**
     * تعيين الصلاحيات للأدوار.
     *
     * @return void
     */
    private function assignPermissionsToRoles()
    {
        // الحصول على الأدوار
        $adminRole = Role::where('slug', 'admin')->first();
        $accountantRole = Role::where('slug', 'accountant')->first();

        // تعيين جميع الصلاحيات الجديدة لدور المدير
        if ($adminRole) {
            $allPermissions = Permission::all();
            $adminRole->permissions()->syncWithoutDetaching($allPermissions->pluck('id')->toArray());
        }

        // تعيين صلاحيات المدفوعات وإعدادات الضرائب للمحاسب
        if ($accountantRole) {
            $accountantPermissions = Permission::whereIn('slug', [
                'change_payment_status',
                'print_payment_receipt',
                'tax_settings.view',
            ])->get();

            $accountantRole->permissions()->syncWithoutDetaching($accountantPermissions->pluck('id')->toArray());
        }
    }
}
