<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use App\Models\Permission;
use App\Models\Role;

class InventoryCountPermissionsSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Create permissions for inventory counts
        $countPermissions = [
            [
                'name' => 'view_inventory_counts',
                'display_name' => 'View Inventory Counts',
                'description' => 'Can view inventory count sessions',
                'group' => 'inventory',
            ],
            [
                'name' => 'create_inventory_counts',
                'display_name' => 'Create Inventory Counts',
                'description' => 'Can create new inventory count sessions',
                'group' => 'inventory',
            ],
            [
                'name' => 'update_inventory_counts',
                'display_name' => 'Update Inventory Counts',
                'description' => 'Can update inventory count items',
                'group' => 'inventory',
            ],
            [
                'name' => 'complete_inventory_counts',
                'display_name' => 'Complete Inventory Counts',
                'description' => 'Can mark inventory counts as complete',
                'group' => 'inventory',
            ],
            [
                'name' => 'approve_inventory_counts',
                'display_name' => 'Approve Inventory Counts',
                'description' => 'Can approve inventory counts and adjust stock',
                'group' => 'inventory',
            ],
            [
                'name' => 'cancel_inventory_counts',
                'display_name' => 'Cancel Inventory Counts',
                'description' => 'Can cancel inventory count sessions',
                'group' => 'inventory',
            ],
        ];

        // Create permissions for inventory alerts
        $alertPermissions = [
            [
                'name' => 'view_inventory_alerts',
                'display_name' => 'View Inventory Alerts',
                'description' => 'Can view inventory alerts',
                'group' => 'inventory',
            ],
            [
                'name' => 'manage_inventory_alerts',
                'display_name' => 'Manage Inventory Alerts',
                'description' => 'Can manage inventory alert settings',
                'group' => 'inventory',
            ],
            [
                'name' => 'send_inventory_alerts',
                'display_name' => 'Send Inventory Alerts',
                'description' => 'Can send inventory alerts manually',
                'group' => 'inventory',
            ],
        ];

        // Combine all permissions
        $permissions = array_merge($countPermissions, $alertPermissions);

        // Create permissions
        foreach ($permissions as $permission) {
            Permission::firstOrCreate(
                ['name' => $permission['name']],
                [
                    'slug' => $permission['name'],
                    'description' => $permission['description'],
                    'group' => $permission['group'],
                    'is_active' => true,
                ]
            );
        }

        // Assign permissions to admin role
        $adminRole = Role::where('name', 'admin')->first();
        if ($adminRole) {
            foreach ($permissions as $permission) {
                $perm = Permission::where('name', $permission['name'])->first();
                if ($perm && !$adminRole->hasPermission($perm->name)) {
                    $adminRole->attachPermission($perm);
                }
            }
        }

        // Assign view permissions to inventory_manager role
        $inventoryManagerRole = Role::where('name', 'inventory_manager')->first();
        if ($inventoryManagerRole) {
            $viewPermissions = [
                'view_inventory_counts',
                'create_inventory_counts',
                'update_inventory_counts',
                'complete_inventory_counts',
                'view_inventory_alerts',
                'send_inventory_alerts',
            ];

            foreach ($viewPermissions as $permName) {
                $perm = Permission::where('name', $permName)->first();
                if ($perm && !$inventoryManagerRole->hasPermission($perm->name)) {
                    $inventoryManagerRole->attachPermission($perm);
                }
            }
        }

        // Assign view permissions to warehouse_staff role
        $warehouseStaffRole = Role::where('name', 'warehouse_staff')->first();
        if ($warehouseStaffRole) {
            $staffPermissions = [
                'view_inventory_counts',
                'update_inventory_counts',
                'view_inventory_alerts',
            ];

            foreach ($staffPermissions as $permName) {
                $perm = Permission::where('name', $permName)->first();
                if ($perm && !$warehouseStaffRole->hasPermission($perm->name)) {
                    $warehouseStaffRole->attachPermission($perm);
                }
            }
        }
    }
}
