<?php

namespace Database\Seeders;

use App\Models\Permission;
use App\Models\Role;
use Illuminate\Database\Seeder;

class InstallmentPermissionsSeeder extends Seeder
{
    /**
     * تنفيذ عملية البذر.
     *
     * @return void
     */
    public function run()
    {
        // مجموعة صلاحيات الأقساط
        $permissionGroup = 'installments';

        // الصلاحيات الخاصة بالأقساط
        $permissions = [
            [
                'name' => 'إدارة الأقساط',
                'slug' => 'manage_installments',
                'description' => 'إدارة أقساط الفواتير',
                'group' => $permissionGroup,
                'type' => 'crud',
            ],
            [
                'name' => 'عرض الأقساط',
                'slug' => 'view_installments',
                'description' => 'عرض قائمة الأقساط',
                'group' => $permissionGroup,
                'type' => 'view',
            ],
            [
                'name' => 'إنشاء أقساط',
                'slug' => 'create_installments',
                'description' => 'إنشاء أقساط جديدة للفواتير',
                'group' => $permissionGroup,
                'type' => 'action',
            ],
            [
                'name' => 'تعديل قسط',
                'slug' => 'edit_installment',
                'description' => 'تعديل بيانات القسط',
                'group' => $permissionGroup,
                'type' => 'action',
            ],
            [
                'name' => 'حذف قسط',
                'slug' => 'delete_installment',
                'description' => 'حذف قسط',
                'group' => $permissionGroup,
                'type' => 'action',
            ],
            [
                'name' => 'تسجيل دفعة قسط',
                'slug' => 'record_installment_payment',
                'description' => 'تسجيل دفعة لقسط',
                'group' => $permissionGroup,
                'type' => 'action',
            ],
            [
                'name' => 'إرسال تذكير بالقسط',
                'slug' => 'send_installment_reminder',
                'description' => 'إرسال تذكير للعميل بالقسط المستحق',
                'group' => $permissionGroup,
                'type' => 'action',
            ],
            [
                'name' => 'تحديث حالة الأقساط المتأخرة',
                'slug' => 'update_overdue_installments',
                'description' => 'تحديث حالة الأقساط المتأخرة',
                'group' => $permissionGroup,
                'type' => 'action',
            ],
            [
                'name' => 'عرض لوحة معلومات الأقساط',
                'slug' => 'view_installments_dashboard',
                'description' => 'عرض لوحة معلومات الأقساط والإحصائيات',
                'group' => $permissionGroup,
                'type' => 'view',
            ],
            [
                'name' => 'عرض تقارير الأقساط',
                'slug' => 'view_installments_reports',
                'description' => 'عرض تقارير الأقساط المختلفة',
                'group' => $permissionGroup,
                'type' => 'view',
            ],
            [
                'name' => 'تصدير تقارير الأقساط',
                'slug' => 'export_installments_reports',
                'description' => 'تصدير تقارير الأقساط',
                'group' => $permissionGroup,
                'type' => 'action',
            ],
        ];

        // إنشاء الصلاحيات
        foreach ($permissions as $permission) {
            Permission::firstOrCreate(
                ['slug' => $permission['slug']],
                [
                    'name' => $permission['name'],
                    'description' => $permission['description'],
                    'group' => $permission['group'],
                    'type' => $permission['type'],
                    'is_active' => true,
                ]
            );
        }

        // تحديث صلاحيات الأدوار
        $this->updateRolePermissions();
    }

    /**
     * تحديث صلاحيات الأدوار.
     *
     * @return void
     */
    private function updateRolePermissions()
    {
        // منح صلاحيات الأقساط للسوبر أدمن
        $superAdminRole = Role::where('slug', 'super_admin')->first();
        if ($superAdminRole) {
            $allPermissions = Permission::all();
            $permissionIds = $allPermissions->pluck('id')->filter()->toArray();
            $superAdminRole->syncPermissions($permissionIds);
        }

        // منح صلاحيات الأقساط للمدير
        $adminRole = Role::where('slug', 'admin')->first();
        if ($adminRole) {
            $adminPermissions = Permission::whereNotIn('slug', [
                'manage_system_settings',
                'view_system_logs',
                'manage_backups',
                'manage_server',
            ])->get();
            $adminPermissionIds = $adminPermissions->pluck('id')->filter()->toArray();
            $adminRole->syncPermissions($adminPermissionIds);
        }

        // منح صلاحيات الأقساط للمحاسب
        $accountantRole = Role::where('slug', 'accountant')->first();
        if ($accountantRole) {
            $accountantPermissions = Permission::where('group', 'installments')
                ->orWhere('group', 'profile')
                ->orWhereIn('slug', [
                    'dashboard_access',
                    'view_reports',
                    'view_customers',
                    'view_invoices',
                    'create_invoice',
                    'edit_invoice',
                    'view_payments',
                    'create_payment',
                ])
                ->get();
            $accountantPermissionIds = $accountantPermissions->pluck('id')->filter()->toArray();
            $accountantRole->syncPermissions($accountantPermissionIds);
        }
    }
}
