<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Permission;
use App\Models\Role;

class EmployeeTasksPermissionsSeeder extends Seeder
{
    /**
     * Run the database seeds.
     *
     * @return void
     */
    public function run()
    {
        // Log the start of the seeder
        \Log::info('Starting EmployeeTasksPermissionsSeeder');

        // Create Employee Tasks permissions
        $permissions = [
            // Tasks
            'view_employee_tasks',
            'create_employee_task',
            'edit_employee_task',
            'delete_employee_task',
            'manage_employee_tasks',

            // Task Status
            'update_task_status',
            'update_task_progress',

            // Task Assignments
            'view_task_assignments',
            'create_task_assignment',
            'delete_task_assignment',
            'manage_task_assignments',

            // Task Comments
            'view_task_comments',
            'create_task_comment',
            'edit_task_comment',
            'delete_task_comment',
            'manage_task_comments',

            // Task Attachments
            'view_task_attachments',
            'upload_task_attachment',
            'delete_task_attachment',
            'manage_task_attachments',

            // Task Dashboard
            'view_task_dashboard',
            'view_task_reports',
            'export_task_reports',
        ];

        // Create permissions if they don't exist
        foreach ($permissions as $permission) {
            // Check if permission already exists
            $existingPermission = Permission::where('slug', $permission)->first();

            if (!$existingPermission) {
                // Set all permissions to HRM group
                $group = 'hrm';

                // Check if HRM parent group exists
                $parentPermission = Permission::where('slug', 'manage_' . $group)->first();

                // Log parent permission check
                \Log::info('Checking for parent permission: manage_' . $group);

                // Create HRM parent group if it doesn't exist
                if (!$parentPermission) {
                    \Log::info('Creating parent permission for group: ' . $group);
                    $parentPermission = Permission::create([
                        'name' => 'إدارة الموارد البشرية',
                        'slug' => 'manage_' . $group,
                        'group' => $group,
                        'description' => 'إدارة كاملة لنظام الموارد البشرية',
                        'level' => 0,
                        'path' => null,
                        'type' => 'group',
                        'resource' => $group,
                        'order' => 0,
                        'is_active' => true,
                    ]);

                    // Update path after creation
                    $parentPermission->path = $parentPermission->id;
                    $parentPermission->save();
                }

                if ($parentPermission) {
                    \Log::info('Creating permission: ' . $permission);
                    $newPermission = Permission::create([
                        'parent_id' => $parentPermission->id,
                        'name' => ucfirst(str_replace('_', ' ', $permission)),
                        'slug' => $permission,
                        'group' => $group,
                        'description' => 'صلاحية ' . ucfirst(str_replace('_', ' ', $permission)),
                        'level' => 1,
                        'path' => null,
                        'type' => 'action',
                        'resource' => explode('_', $permission)[1] ?? $group,
                        'order' => 10,
                        'is_active' => true,
                    ]);

                    // Update path after creation
                    $newPermission->path = $parentPermission->path . '.' . $newPermission->id;
                    $newPermission->save();
                }
            }
        }

        // Assign permissions to admin role
        $adminRole = Role::where('slug', 'admin')->first();
        if ($adminRole) {
            $permissionIds = Permission::whereIn('slug', $permissions)->pluck('id')->toArray();
            \Log::info('Found ' . count($permissionIds) . ' permissions to assign to admin role');
            $adminRole->permissions()->syncWithoutDetaching($permissionIds);
        } else {
            \Log::error('Admin role not found');
        }

        // Assign view permissions to user role
        $userRole = Role::where('slug', 'user')->first();
        if ($userRole) {
            $viewPermissionIds = Permission::whereIn('slug', array_filter($permissions, function ($permission) {
                return strpos($permission, 'view_') === 0;
            }))->pluck('id')->toArray();
            $userRole->permissions()->syncWithoutDetaching($viewPermissionIds);
        }

        // Assign HR permissions to HR role
        $hrRole = Role::where('slug', 'hr')->first();
        if ($hrRole) {
            $permissionIds = Permission::whereIn('slug', $permissions)->pluck('id')->toArray();
            $hrRole->permissions()->syncWithoutDetaching($permissionIds);
        }

        $this->command->info('Employee Tasks permissions created and assigned successfully!');
    }
}
