<?php

namespace Database\Seeders;

use App\Models\Permission;
use App\Models\Role;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Schema;

class BankingPermissionsSeeder extends Seeder
{
    /**
     * تشغيل عملية البذر.
     *
     * @return void
     */
    public function run()
    {
        // تعطيل فحص المفاتيح الأجنبية مؤقتاً
        Schema::disableForeignKeyConstraints();

        // إنشاء صلاحيات البنوك
        $this->createBankingPermissions();

        // تعيين صلاحيات البنوك للأدوار
        $this->assignBankingPermissionsToRoles();

        // إعادة تفعيل فحص المفاتيح الأجنبية
        Schema::enableForeignKeyConstraints();

        $this->command->info('تم إنشاء وتعيين صلاحيات نظام البنوك بنجاح.');
    }

    /**
     * إنشاء صلاحيات البنوك.
     *
     * @return void
     */
    private function createBankingPermissions()
    {
        // مجموعات صلاحيات البنوك
        $bankingPermissionGroups = [
            'banks' => 'إدارة البنوك',
            'bank_accounts' => 'إدارة الحسابات البنكية',
            'bank_transactions' => 'إدارة المعاملات البنكية',
            'cheques' => 'إدارة الشيكات',
            'bank_reconciliations' => 'إدارة التسويات البنكية',
        ];

        // الصلاحيات الأساسية لكل مجموعة
        $basicPermissions = [
            'view_%s' => 'عرض %s',
            'create_%s' => 'إنشاء %s',
            'edit_%s' => 'تعديل %s',
            'delete_%s' => 'حذف %s',
        ];

        // إنشاء صلاحيات البنوك لكل مجموعة
        foreach ($bankingPermissionGroups as $group => $groupName) {
            // التحقق من وجود صلاحية المجموعة الرئيسية
            $parentPermission = Permission::where('slug', 'manage_' . $group)->first();

            // إنشاء صلاحية المجموعة الرئيسية إذا لم تكن موجودة
            if (!$parentPermission) {
                $parentPermission = Permission::create([
                    'name' => $groupName,
                    'slug' => 'manage_' . $group,
                    'group' => $group,
                    'description' => 'إدارة كاملة لـ ' . $groupName,
                    'level' => 0,
                    'path' => null,
                    'type' => 'group',
                    'resource' => $group,
                    'order' => 0,
                    'is_active' => true,
                ]);

                // تحديث المسار بعد الإنشاء
                $parentPermission->path = $parentPermission->id;
                $parentPermission->save();
            }

            // إنشاء الصلاحيات الفرعية
            $order = 1;
            foreach ($basicPermissions as $slug => $name) {
                $formattedSlug = sprintf($slug, rtrim($group, 's'));
                $formattedName = sprintf($name, $groupName);

                // التحقق من وجود الصلاحية الفرعية
                $permission = Permission::where('slug', $formattedSlug)->first();

                // إنشاء الصلاحية الفرعية إذا لم تكن موجودة
                if (!$permission) {
                    $permission = Permission::create([
                        'parent_id' => $parentPermission->id,
                        'name' => $formattedName,
                        'slug' => $formattedSlug,
                        'group' => $group,
                        'description' => $formattedName,
                        'level' => 1,
                        'path' => null,
                        'type' => 'action',
                        'resource' => $group,
                        'order' => $order++,
                        'is_active' => true,
                    ]);

                    // تحديث المسار بعد الإنشاء
                    $permission->path = $parentPermission->path . '.' . $permission->id;
                    $permission->save();
                }
            }
        }

        // إضافة صلاحيات خاصة للبنوك
        $specialBankingPermissions = [
            [
                'name' => 'إيداع شيك',
                'slug' => 'deposit_cheque',
                'group' => 'cheques',
                'description' => 'إيداع شيك في الحساب البنكي',
                'type' => 'action',
                'resource' => 'cheques',
            ],
            [
                'name' => 'تحصيل شيك',
                'slug' => 'collect_cheque',
                'group' => 'cheques',
                'description' => 'تحصيل شيك من الحساب البنكي',
                'type' => 'action',
                'resource' => 'cheques',
            ],
            [
                'name' => 'إرجاع شيك',
                'slug' => 'return_cheque',
                'group' => 'cheques',
                'description' => 'إرجاع شيك غير محصل',
                'type' => 'action',
                'resource' => 'cheques',
            ],
            [
                'name' => 'إلغاء شيك',
                'slug' => 'cancel_cheque',
                'group' => 'cheques',
                'description' => 'إلغاء شيك',
                'type' => 'action',
                'resource' => 'cheques',
            ],
            [
                'name' => 'عرض كشف حساب',
                'slug' => 'view_bank_statement',
                'group' => 'bank_accounts',
                'description' => 'عرض كشف حساب بنكي',
                'type' => 'action',
                'resource' => 'bank_accounts',
            ],
            [
                'name' => 'إجراء تسوية بنكية',
                'slug' => 'perform_bank_reconciliation',
                'group' => 'bank_reconciliations',
                'description' => 'إجراء تسوية بنكية للحساب',
                'type' => 'action',
                'resource' => 'bank_reconciliations',
            ],
        ];

        // إنشاء الصلاحيات الخاصة للبنوك
        foreach ($specialBankingPermissions as $permissionData) {
            $parentGroup = Permission::where('slug', 'manage_' . $permissionData['group'])->first();

            if ($parentGroup) {
                // التحقق من وجود الصلاحية الخاصة
                $permission = Permission::where('slug', $permissionData['slug'])->first();

                // إنشاء الصلاحية الخاصة إذا لم تكن موجودة
                if (!$permission) {
                    $permission = Permission::create([
                        'parent_id' => $parentGroup->id,
                        'name' => $permissionData['name'],
                        'slug' => $permissionData['slug'],
                        'group' => $permissionData['group'],
                        'description' => $permissionData['description'],
                        'level' => 1,
                        'path' => null,
                        'type' => $permissionData['type'],
                        'resource' => $permissionData['resource'],
                        'order' => 10, // ترتيب عالي للصلاحيات الخاصة
                        'is_active' => true,
                    ]);

                    // تحديث المسار بعد الإنشاء
                    $permission->path = $parentGroup->path . '.' . $permission->id;
                    $permission->save();
                }
            }
        }
    }

    /**
     * تعيين صلاحيات البنوك للأدوار.
     *
     * @return void
     */
    private function assignBankingPermissionsToRoles()
    {
        // الحصول على الأدوار
        $adminRole = Role::where('slug', 'admin')->first();
        $accountantRole = Role::where('slug', 'accountant')->first();
        $branchManagerRole = Role::where('slug', 'branch_manager')->first();
        $userRole = Role::where('slug', 'user')->first();

        // تعيين جميع صلاحيات البنوك لدور المدير
        if ($adminRole) {
            $bankingPermissions = Permission::where(function ($query) {
                $query->where('group', 'banks')
                    ->orWhere('group', 'bank_accounts')
                    ->orWhere('group', 'bank_transactions')
                    ->orWhere('group', 'cheques')
                    ->orWhere('group', 'bank_reconciliations');
            })->get();

            // الحصول على الصلاحيات الحالية للمدير
            $currentAdminPermissions = $adminRole->permissions()->pluck('permissions.id')->toArray();

            // إضافة صلاحيات البنوك إلى صلاحيات المدير الحالية
            $newAdminPermissions = array_merge(
                $currentAdminPermissions,
                $bankingPermissions->pluck('id')->filter()->toArray()
            );

            // تحديث صلاحيات المدير
            $adminRole->permissions()->sync($newAdminPermissions);
        }

        // تعيين صلاحيات البنوك لدور المحاسب
        if ($accountantRole) {
            $accountantBankingPermissions = Permission::where(function ($query) {
                $query->where('group', 'banks')
                    ->orWhere('group', 'bank_accounts')
                    ->orWhere('group', 'bank_transactions')
                    ->orWhere('group', 'cheques')
                    ->orWhere('group', 'bank_reconciliations');
            })->get();

            // الحصول على الصلاحيات الحالية للمحاسب
            $currentAccountantPermissions = $accountantRole->permissions()->pluck('permissions.id')->toArray();

            // إضافة صلاحيات البنوك إلى صلاحيات المحاسب الحالية
            $newAccountantPermissions = array_merge(
                $currentAccountantPermissions,
                $accountantBankingPermissions->pluck('id')->filter()->toArray()
            );

            // تحديث صلاحيات المحاسب
            $accountantRole->permissions()->sync($newAccountantPermissions);
        }

        // تعيين صلاحيات عرض البنوك لدور مدير الفرع
        if ($branchManagerRole) {
            $branchManagerBankingPermissions = Permission::where(function ($query) {
                $query->where('slug', 'view_bank')
                    ->orWhere('slug', 'view_bank_account')
                    ->orWhere('slug', 'view_bank_transaction')
                    ->orWhere('slug', 'view_cheque')
                    ->orWhere('slug', 'view_bank_reconciliation')
                    ->orWhere('slug', 'view_bank_statement');
            })->get();

            // الحصول على الصلاحيات الحالية لمدير الفرع
            $currentBranchManagerPermissions = $branchManagerRole->permissions()->pluck('permissions.id')->toArray();

            // إضافة صلاحيات عرض البنوك إلى صلاحيات مدير الفرع الحالية
            $newBranchManagerPermissions = array_merge(
                $currentBranchManagerPermissions,
                $branchManagerBankingPermissions->pluck('id')->filter()->toArray()
            );

            // تحديث صلاحيات مدير الفرع
            $branchManagerRole->permissions()->sync($newBranchManagerPermissions);
        }

        // تعيين صلاحيات عرض محدودة للمستخدم العادي
        if ($userRole) {
            $userBankingPermissions = Permission::where(function ($query) {
                $query->where('slug', 'view_bank')
                    ->orWhere('slug', 'view_bank_account');
            })->get();

            // الحصول على الصلاحيات الحالية للمستخدم
            $currentUserPermissions = $userRole->permissions()->pluck('permissions.id')->toArray();

            // إضافة صلاحيات عرض البنوك المحدودة إلى صلاحيات المستخدم الحالية
            $newUserPermissions = array_merge(
                $currentUserPermissions,
                $userBankingPermissions->pluck('id')->filter()->toArray()
            );

            // تحديث صلاحيات المستخدم
            $userRole->permissions()->sync($newUserPermissions);
        }
    }
}
