<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;
use App\Models\PurchaseInvoice;
use App\Models\User;

return new class extends Migration
{
    /**
     * Run the migrations.
     * حل مشكلة عزل الشركات في فواتير المشتريات
     * 1. تحديث قيم company_id للفواتير القديمة
     * 2. جعل عمود company_id إلزامياً
     *
     * @return void
     */
    public function up()
    {
        // الخطوة 1: تحديث الفواتير التي ليس لها company_id
        // نستخدم استعلام SQL مباشر لتجنب مشاكل التوافق مع Doctrine

        // أولاً: تحديث الفواتير بناءً على الشركة التي ينتمي إليها المستخدم الذي أنشأها
        DB::statement('
            UPDATE purchase_invoices pi
            JOIN users u ON pi.created_by = u.id
            SET pi.company_id = u.company_id
            WHERE pi.company_id IS NULL AND u.company_id IS NOT NULL
        ');

        // ثانياً: تحديث أي فواتير متبقية بدون company_id لتكون تابعة للشركة الأولى (الافتراضية)
        DB::statement('
            UPDATE purchase_invoices
            SET company_id = 1
            WHERE company_id IS NULL
        ');

        // الخطوة 2: جعل عمود company_id إلزامياً
        // نستخدم طريقة بديلة لتجنب مشاكل التوافق مع Doctrine

        // إضافة قيد NOT NULL باستخدام استعلام SQL مباشر
        DB::statement('
            ALTER TABLE purchase_invoices
            MODIFY company_id BIGINT UNSIGNED NOT NULL
        ');

        // إعادة إنشاء مفتاح الجدول الأجنبي
        Schema::table('purchase_invoices', function (Blueprint $table) {
            // إزالة المفتاح الأجنبي الحالي إذا كان موجوداً
            $foreignKeys = $this->listTableForeignKeys('purchase_invoices');
            if (in_array('purchase_invoices_company_id_foreign', $foreignKeys)) {
                $table->dropForeign('purchase_invoices_company_id_foreign');
            }

            // إعادة إنشاء المفتاح الأجنبي
            $table->foreign('company_id')->references('id')->on('companies')->onDelete('cascade');
        });
    }

    /**
     * Reverse the migrations.
     * إعادة عمود company_id ليكون اختيارياً
     *
     * @return void
     */
    public function down()
    {
        // إزالة قيد NOT NULL باستخدام استعلام SQL مباشر
        DB::statement('
            ALTER TABLE purchase_invoices
            MODIFY company_id BIGINT UNSIGNED NULL
        ');
    }

    /**
     * الحصول على قائمة المفاتيح الأجنبية للجدول
     *
     * @param string $table
     * @return array
     */
    private function listTableForeignKeys($table)
    {
        $conn = Schema::getConnection()->getDoctrineSchemaManager();

        $foreignKeys = [];

        if (method_exists($conn, 'listTableForeignKeys')) {
            $tableForeignKeys = $conn->listTableForeignKeys($table);

            foreach ($tableForeignKeys as $key) {
                $foreignKeys[] = $key->getName();
            }
        }

        return $foreignKeys;
    }
};
