<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * تشغيل الترحيلات.
     *
     * @return void
     */
    public function up()
    {
        // جدول البنوك
        Schema::create('banks', function (Blueprint $table) {
            $table->id();
            $table->foreignId('company_id')->constrained();
            $table->string('name');
            $table->string('code', 50)->unique();
            $table->string('swift_code')->nullable();
            $table->text('address')->nullable();
            $table->string('contact_person')->nullable();
            $table->string('contact_phone')->nullable();
            $table->string('contact_email')->nullable();
            $table->text('notes')->nullable();
            $table->boolean('is_active')->default(true);
            $table->timestamps();
            $table->softDeletes();
        });

        // جدول الحسابات البنكية
        Schema::create('bank_accounts', function (Blueprint $table) {
            $table->id();
            $table->foreignId('company_id')->constrained();
            $table->foreignId('bank_id')->constrained();
            $table->foreignId('account_id')->constrained('chart_of_accounts');
            $table->string('account_number');
            $table->string('iban')->nullable();
            $table->string('account_name');
            $table->enum('account_type', ['current', 'savings', 'deposit', 'credit', 'other']);
            $table->string('currency', 10)->default('SAR');
            $table->decimal('opening_balance', 15, 2)->default(0);
            $table->decimal('current_balance', 15, 2)->default(0);
            $table->date('opening_date');
            $table->boolean('is_active')->default(true);
            $table->text('notes')->nullable();
            $table->timestamps();
            $table->softDeletes();
        });

        // جدول المعاملات البنكية
        Schema::create('bank_transactions', function (Blueprint $table) {
            $table->id();
            $table->foreignId('company_id')->constrained();
            $table->foreignId('bank_account_id')->constrained();
            $table->string('transaction_number')->unique();
            $table->enum('transaction_type', ['deposit', 'withdrawal', 'transfer', 'cheque_deposit', 'cheque_withdrawal', 'bank_charges', 'interest', 'other']);
            $table->date('transaction_date');
            $table->decimal('amount', 15, 2);
            $table->foreignId('related_bank_account_id')->nullable()->constrained('bank_accounts')->onDelete('set null');
            $table->foreignId('related_cash_register_id')->nullable()->constrained('cash_registers')->onDelete('set null');
            $table->string('reference_number')->nullable();
            $table->string('reference_type')->nullable();
            $table->unsignedBigInteger('reference_id')->nullable();
            $table->text('description')->nullable();
            $table->string('attachment')->nullable();
            $table->foreignId('created_by')->constrained('users');
            $table->foreignId('approved_by')->nullable()->constrained('users');
            $table->timestamp('approved_at')->nullable();
            $table->boolean('is_reconciled')->default(false);
            $table->timestamps();
            $table->softDeletes();
        });

        // جدول الشيكات
        Schema::create('cheques', function (Blueprint $table) {
            $table->id();
            $table->foreignId('company_id')->constrained();
            $table->enum('type', ['incoming', 'outgoing']);
            $table->string('cheque_number');
            $table->date('cheque_date');
            $table->decimal('amount', 15, 2);
            $table->string('currency', 10)->default('SAR');
            $table->foreignId('bank_id')->constrained();
            $table->foreignId('bank_account_id')->nullable()->constrained()->onDelete('set null');
            $table->string('drawer_name')->nullable(); // اسم الساحب للشيكات الواردة
            $table->string('beneficiary_name')->nullable(); // اسم المستفيد للشيكات الصادرة
            $table->enum('status', ['issued', 'received', 'deposited', 'cleared', 'bounced', 'cancelled', 'on_hold']);
            $table->date('due_date');
            $table->date('deposit_date')->nullable();
            $table->date('clearing_date')->nullable();
            $table->string('reference_type')->nullable();
            $table->unsignedBigInteger('reference_id')->nullable();
            $table->foreignId('bank_transaction_id')->nullable()->constrained()->onDelete('set null');
            $table->text('notes')->nullable();
            $table->string('attachment')->nullable();
            $table->foreignId('created_by')->constrained('users');
            $table->timestamps();
            $table->softDeletes();
        });

        // جدول التسويات البنكية
        Schema::create('bank_reconciliations', function (Blueprint $table) {
            $table->id();
            $table->foreignId('company_id')->constrained();
            $table->foreignId('bank_account_id')->constrained();
            $table->string('reconciliation_number')->unique();
            $table->date('statement_date');
            $table->date('reconciliation_date');
            $table->decimal('statement_balance', 15, 2);
            $table->decimal('system_balance', 15, 2);
            $table->decimal('difference', 15, 2)->default(0);
            $table->text('notes')->nullable();
            $table->enum('status', ['draft', 'in_progress', 'completed', 'cancelled']);
            $table->foreignId('created_by')->constrained('users');
            $table->foreignId('approved_by')->nullable()->constrained('users');
            $table->timestamp('approved_at')->nullable();
            $table->timestamps();
            $table->softDeletes();
        });

        // جدول تفاصيل التسويات البنكية
        Schema::create('bank_reconciliation_items', function (Blueprint $table) {
            $table->id();
            $table->foreignId('bank_reconciliation_id')->constrained()->onDelete('cascade');
            $table->foreignId('bank_transaction_id')->nullable()->constrained()->onDelete('set null');
            $table->foreignId('cheque_id')->nullable()->constrained()->onDelete('set null');
            $table->enum('item_type', ['transaction', 'cheque', 'adjustment']);
            $table->enum('status', ['matched', 'unmatched', 'adjusted']);
            $table->decimal('amount', 15, 2);
            $table->date('transaction_date');
            $table->text('description')->nullable();
            $table->boolean('is_system_record')->default(true);
            $table->boolean('is_statement_record')->default(false);
            $table->timestamps();
        });

        // جدول ربط المعاملات البنكية بالقيود المحاسبية
        Schema::create('bank_transaction_entries', function (Blueprint $table) {
            $table->id();
            $table->foreignId('bank_transaction_id')->constrained()->onDelete('cascade');
            $table->foreignId('journal_entry_id')->constrained()->onDelete('cascade');
            $table->timestamps();

            $table->unique(['bank_transaction_id', 'journal_entry_id'], 'bank_trans_journal_unique');
        });
    }

    /**
     * عكس الترحيلات.
     *
     * @return void
     */
    public function down()
    {
        Schema::dropIfExists('bank_transaction_entries');
        Schema::dropIfExists('bank_reconciliation_items');
        Schema::dropIfExists('bank_reconciliations');
        Schema::dropIfExists('cheques');
        Schema::dropIfExists('bank_transactions');
        Schema::dropIfExists('bank_accounts');
        Schema::dropIfExists('banks');
    }
};
