<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * تشغيل الترحيلات.
     *
     * @return void
     */
    public function up()
    {
        // جدول دليل الحسابات
        Schema::create('chart_of_accounts', function (Blueprint $table) {
            $table->id();
            $table->foreignId('company_id')->constrained();
            $table->string('account_code', 50)->unique();
            $table->string('name');
            $table->string('account_type'); // أصول، خصوم، إيرادات، مصروفات، حقوق ملكية
            $table->string('sub_type')->nullable(); // تصنيف فرعي للحساب
            $table->unsignedBigInteger('parent_id')->nullable();
            $table->foreign('parent_id')->references('id')->on('chart_of_accounts')->onDelete('cascade');
            $table->text('description')->nullable();
            $table->boolean('is_active')->default(true);
            $table->decimal('opening_balance', 15, 2)->default(0);
            $table->decimal('current_balance', 15, 2)->default(0);
            $table->string('currency', 10)->default('SAR');
            $table->boolean('is_cash_account')->default(false);
            $table->boolean('is_bank_account')->default(false);
            $table->timestamps();
            $table->softDeletes();
        });

        // جدول السنوات المالية
        Schema::create('fiscal_years', function (Blueprint $table) {
            $table->id();
            $table->foreignId('company_id')->constrained();
            $table->string('name');
            $table->date('start_date');
            $table->date('end_date');
            $table->boolean('is_active')->default(false);
            $table->boolean('is_closed')->default(false);
            $table->timestamps();
            $table->softDeletes();
        });

        // جدول الفترات المحاسبية
        Schema::create('accounting_periods', function (Blueprint $table) {
            $table->id();
            $table->foreignId('fiscal_year_id')->constrained();
            $table->foreignId('company_id')->constrained();
            $table->string('name');
            $table->date('start_date');
            $table->date('end_date');
            $table->boolean('is_closed')->default(false);
            $table->timestamps();
            $table->softDeletes();
        });

        // جدول القيود المحاسبية
        Schema::create('journal_entries', function (Blueprint $table) {
            $table->id();
            $table->foreignId('company_id')->constrained();
            $table->foreignId('fiscal_year_id')->constrained();
            $table->foreignId('accounting_period_id')->constrained();
            $table->string('entry_number')->unique();
            $table->date('entry_date');
            $table->string('reference_type')->nullable();
            $table->unsignedBigInteger('reference_id')->nullable();
            $table->text('description')->nullable();
            $table->boolean('is_posted')->default(false);
            $table->boolean('is_approved')->default(false);
            $table->foreignId('created_by')->constrained('users');
            $table->foreignId('approved_by')->nullable()->constrained('users');
            $table->timestamp('approved_at')->nullable();
            $table->timestamps();
            $table->softDeletes();
        });

        // جدول تفاصيل القيود المحاسبية
        Schema::create('journal_entry_items', function (Blueprint $table) {
            $table->id();
            $table->foreignId('journal_entry_id')->constrained()->onDelete('cascade');
            $table->foreignId('account_id')->constrained('chart_of_accounts');
            $table->decimal('debit', 15, 2)->default(0);
            $table->decimal('credit', 15, 2)->default(0);
            $table->text('description')->nullable();
            $table->string('reference_type')->nullable();
            $table->unsignedBigInteger('reference_id')->nullable();
            $table->timestamps();
        });

        // جدول قوالب القيود المحاسبية
        Schema::create('journal_templates', function (Blueprint $table) {
            $table->id();
            $table->foreignId('company_id')->constrained();
            $table->string('name');
            $table->string('code', 50)->unique();
            $table->text('description')->nullable();
            $table->boolean('is_active')->default(true);
            $table->foreignId('created_by')->constrained('users');
            $table->timestamps();
            $table->softDeletes();
        });

        // جدول تفاصيل قوالب القيود المحاسبية
        Schema::create('journal_template_items', function (Blueprint $table) {
            $table->id();
            $table->foreignId('journal_template_id')->constrained()->onDelete('cascade');
            $table->foreignId('account_id')->constrained('chart_of_accounts');
            $table->enum('type', ['debit', 'credit']);
            $table->decimal('percentage', 5, 2)->nullable();
            $table->decimal('fixed_amount', 15, 2)->nullable();
            $table->text('description')->nullable();
            $table->timestamps();
        });

        // جدول ربط الخزائن بالحسابات
        Schema::create('cash_register_accounts', function (Blueprint $table) {
            $table->id();
            $table->foreignId('cash_register_id')->constrained();
            $table->foreignId('account_id')->constrained('chart_of_accounts');
            $table->timestamps();

            $table->unique(['cash_register_id', 'account_id']);
        });

        // جدول ربط معاملات الخزينة بالقيود المحاسبية
        Schema::create('cash_transaction_entries', function (Blueprint $table) {
            $table->id();
            $table->foreignId('cash_transaction_id')->constrained();
            $table->foreignId('journal_entry_id')->constrained();
            $table->timestamps();

            $table->unique(['cash_transaction_id', 'journal_entry_id'], 'cash_trans_journal_unique');
        });
    }

    /**
     * عكس الترحيلات.
     *
     * @return void
     */
    public function down()
    {
        Schema::dropIfExists('cash_transaction_entries');
        Schema::dropIfExists('cash_register_accounts');
        Schema::dropIfExists('journal_template_items');
        Schema::dropIfExists('journal_templates');
        Schema::dropIfExists('journal_entry_items');
        Schema::dropIfExists('journal_entries');
        Schema::dropIfExists('accounting_periods');
        Schema::dropIfExists('fiscal_years');
        Schema::dropIfExists('chart_of_accounts');
    }
};
