<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     *
     * @return void
     */
    public function up()
    {
        // إضافة حقول قوالب الفواتير
        Schema::table('invoices', function (Blueprint $table) {
            if (!Schema::hasColumn('invoices', 'template_id')) {
                $table->unsignedBigInteger('template_id')->nullable()->after('terms_conditions');
            }
            if (!Schema::hasColumn('invoices', 'language')) {
                $table->string('language', 10)->default('ar')->after('template_id');
            }
            if (!Schema::hasColumn('invoices', 'is_recurring')) {
                $table->boolean('is_recurring')->default(false)->after('language');
            }
            if (!Schema::hasColumn('invoices', 'recurring_cycle')) {
                $table->enum('recurring_cycle', ['monthly', 'quarterly', 'semi_annually', 'annually'])->nullable()->after('is_recurring');
            }
            if (!Schema::hasColumn('invoices', 'next_recurring_date')) {
                $table->date('next_recurring_date')->nullable()->after('recurring_cycle');
            }
            if (!Schema::hasColumn('invoices', 'parent_invoice_id')) {
                $table->unsignedBigInteger('parent_invoice_id')->nullable()->after('next_recurring_date');
                $table->foreign('parent_invoice_id')->references('id')->on('invoices')->onDelete('set null');
            }
        });

        // إنشاء جدول قوالب الفواتير
        if (!Schema::hasTable('invoice_templates')) {
            Schema::create('invoice_templates', function (Blueprint $table) {
                $table->id();
                $table->foreignId('company_id')->constrained()->onDelete('cascade');
                $table->string('name');
                $table->string('language', 10)->default('ar');
                $table->text('header_html')->nullable();
                $table->text('footer_html')->nullable();
                $table->text('css_styles')->nullable();
                $table->boolean('is_default')->default(false);
                $table->boolean('is_active')->default(true);
                $table->timestamps();
            });
        }

        // إنشاء جدول أقساط الفواتير
        if (!Schema::hasTable('invoice_installments')) {
            Schema::create('invoice_installments', function (Blueprint $table) {
                $table->id();
                $table->foreignId('invoice_id')->constrained()->onDelete('cascade');
                $table->decimal('amount', 10, 2);
                $table->date('due_date');
                $table->enum('status', ['pending', 'paid', 'overdue', 'cancelled'])->default('pending');
                $table->text('notes')->nullable();
                $table->timestamps();
            });
        }

        // إنشاء جدول تقارير الضرائب
        if (!Schema::hasTable('tax_reports')) {
            Schema::create('tax_reports', function (Blueprint $table) {
                $table->id();
                $table->foreignId('company_id')->constrained()->onDelete('cascade');
                $table->string('report_number')->unique();
                $table->date('start_date');
                $table->date('end_date');
                $table->decimal('total_tax_collected', 12, 2)->default(0);
                $table->decimal('total_tax_paid', 12, 2)->default(0);
                $table->decimal('net_tax', 12, 2)->default(0);
                $table->enum('status', ['draft', 'submitted', 'approved', 'rejected'])->default('draft');
                $table->text('notes')->nullable();
                $table->foreignId('created_by')->constrained('users')->onDelete('restrict');
                $table->timestamps();
            });
        }
    }

    /**
     * Reverse the migrations.
     *
     * @return void
     */
    public function down()
    {
        // حذف الجداول بترتيب عكسي لتجنب مشاكل المفاتيح الأجنبية
        Schema::dropIfExists('tax_reports');
        Schema::dropIfExists('invoice_installments');
        Schema::dropIfExists('invoice_templates');

        // حذف الحقول المضافة
        Schema::table('invoices', function (Blueprint $table) {
            if (Schema::hasColumn('invoices', 'parent_invoice_id')) {
                $table->dropForeign(['parent_invoice_id']);
                $table->dropColumn('parent_invoice_id');
            }
            if (Schema::hasColumn('invoices', 'next_recurring_date')) {
                $table->dropColumn('next_recurring_date');
            }
            if (Schema::hasColumn('invoices', 'recurring_cycle')) {
                $table->dropColumn('recurring_cycle');
            }
            if (Schema::hasColumn('invoices', 'is_recurring')) {
                $table->dropColumn('is_recurring');
            }
            if (Schema::hasColumn('invoices', 'language')) {
                $table->dropColumn('language');
            }
            if (Schema::hasColumn('invoices', 'template_id')) {
                $table->dropColumn('template_id');
            }
        });
    }
};
