<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     *
     * @return void
     */
    public function up()
    {
        // 0. إنشاء جدول الفروع إذا لم يكن موجوداً
        if (!Schema::hasTable('branches')) {
            Schema::create('branches', function (Blueprint $table) {
                $table->id();
                $table->foreignId('company_id')->constrained()->onDelete('cascade');
                $table->string('name');
                $table->string('name_en')->nullable();
                $table->string('phone')->nullable();
                $table->string('mobile')->nullable();
                $table->string('email')->nullable();
                $table->string('address')->nullable();
                $table->string('address_en')->nullable();
                $table->string('city')->nullable();
                $table->string('state')->nullable();
                $table->string('country')->nullable();
                $table->string('postal_code')->nullable();
                $table->decimal('location_lat', 10, 7)->nullable();
                $table->decimal('location_lng', 10, 7)->nullable();
                $table->boolean('is_main')->default(false);
                $table->boolean('is_active')->default(true);
                $table->timestamps();
                $table->softDeletes();
            });
        }

        // 1. إنشاء جدول العملاء إذا لم يكن موجوداً
        if (!Schema::hasTable('customers')) {
            Schema::create('customers', function (Blueprint $table) {
                $table->id();
                $table->foreignId('company_id')->constrained()->onDelete('cascade');
                $table->foreignId('branch_id')->nullable()->constrained()->onDelete('cascade');
                $table->string('name');
                $table->string('contact_person')->nullable();
                $table->string('email')->nullable();
                $table->string('phone')->nullable();
                $table->string('mobile')->nullable();
                $table->string('address')->nullable();
                $table->string('city')->nullable();
                $table->string('state')->nullable();
                $table->string('country')->nullable();
                $table->string('postal_code')->nullable();
                $table->string('tax_number')->nullable();
                $table->text('notes')->nullable();
                $table->enum('type', ['individual', 'company'])->default('individual');
                $table->boolean('is_active')->default(true);
                $table->timestamps();
                $table->softDeletes();
            });
        }

        // 2. إنشاء جدول المواقع إذا لم يكن موجوداً
        if (!Schema::hasTable('locations')) {
            Schema::create('locations', function (Blueprint $table) {
                $table->id();
                $table->foreignId('company_id')->constrained()->onDelete('cascade');
                $table->foreignId('customer_id')->constrained()->onDelete('cascade');
                $table->foreignId('branch_id')->nullable()->constrained()->onDelete('cascade');
                $table->string('name');
                $table->string('address')->nullable();
                $table->string('city')->nullable();
                $table->string('state')->nullable();
                $table->string('country')->nullable();
                $table->string('postal_code')->nullable();
                $table->string('contact_person')->nullable();
                $table->string('contact_phone')->nullable();
                $table->string('contact_email')->nullable();
                $table->decimal('latitude', 10, 7)->nullable();
                $table->decimal('longitude', 10, 7)->nullable();
                $table->text('notes')->nullable();
                $table->timestamps();
                $table->softDeletes();
            });
        }

        // 3. إنشاء جدول المصاعد إذا لم يكن موجوداً
        if (!Schema::hasTable('elevators')) {
            Schema::create('elevators', function (Blueprint $table) {
                $table->id();
                $table->foreignId('company_id')->constrained()->onDelete('cascade');
                $table->foreignId('customer_id')->constrained()->onDelete('cascade');
                $table->foreignId('location_id')->constrained()->onDelete('cascade');
                $table->foreignId('branch_id')->nullable()->constrained()->onDelete('cascade');
                $table->string('serial_number')->unique();
                $table->string('model');
                $table->string('manufacturer');
                $table->date('installation_date');
                $table->date('warranty_end_date')->nullable();
                $table->integer('capacity')->comment('in kg');
                $table->integer('floors_count');
                $table->decimal('speed', 5, 2)->comment('in m/s');
                $table->enum('type', ['passenger', 'freight', 'hospital', 'panoramic', 'other'])->default('passenger');
                $table->enum('status', ['operational', 'under_maintenance', 'out_of_service', 'installation'])->default('operational');
                $table->string('qr_code')->nullable();
                $table->text('specifications')->nullable();
                $table->text('notes')->nullable();
                $table->timestamps();
                $table->softDeletes();
            });
        }

        // 4. إنشاء جدول عقود الصيانة إذا لم يكن موجوداً
        if (!Schema::hasTable('maintenance_contracts')) {
            Schema::create('maintenance_contracts', function (Blueprint $table) {
                $table->id();
                $table->foreignId('company_id')->constrained()->onDelete('cascade');
                $table->foreignId('customer_id')->constrained()->onDelete('cascade');
                $table->foreignId('branch_id')->nullable()->constrained()->onDelete('cascade');
                $table->string('contract_number')->unique();
                $table->string('title');
                $table->date('start_date');
                $table->date('end_date');
                $table->decimal('total_amount', 10, 2);
                $table->enum('billing_cycle', ['monthly', 'quarterly', 'semi_annually', 'annually'])->default('monthly');
                $table->enum('status', ['draft', 'active', 'expired', 'cancelled'])->default('draft');
                $table->integer('visits_per_year');
                $table->text('terms_conditions')->nullable();
                $table->text('notes')->nullable();
                $table->timestamps();
                $table->softDeletes();
            });
        }

        // 5. إنشاء جدول العلاقة بين العقود والمصاعد إذا لم يكن موجوداً
        if (!Schema::hasTable('contract_elevator')) {
            Schema::create('contract_elevator', function (Blueprint $table) {
                $table->id();
                $table->foreignId('maintenance_contract_id')->constrained()->onDelete('cascade');
                $table->foreignId('elevator_id')->constrained()->onDelete('cascade');
                $table->timestamps();
            });
        }

        // 6. إنشاء جدول طرق الدفع إذا لم يكن موجوداً
        if (!Schema::hasTable('payment_methods')) {
            Schema::create('payment_methods', function (Blueprint $table) {
                $table->id();
                $table->foreignId('company_id')->constrained()->onDelete('cascade');
                $table->string('name');
                $table->text('description')->nullable();
                $table->boolean('is_active')->default(true);
                $table->timestamps();
            });
        }

        // 7. إنشاء جدول الفواتير إذا لم يكن موجوداً
        if (!Schema::hasTable('invoices')) {
            Schema::create('invoices', function (Blueprint $table) {
                $table->id();
                $table->foreignId('company_id')->constrained()->onDelete('cascade');
                $table->foreignId('customer_id')->constrained()->onDelete('cascade');
                $table->foreignId('maintenance_contract_id')->nullable()->constrained()->onDelete('set null');
                $table->foreignId('branch_id')->constrained()->onDelete('cascade');
                $table->foreignId('created_by')->constrained('users')->onDelete('restrict');
                $table->string('invoice_number')->unique();
                $table->date('invoice_date');
                $table->date('due_date');
                $table->decimal('subtotal', 10, 2);
                $table->decimal('tax_rate', 5, 2)->default(15.00);
                $table->decimal('tax_amount', 10, 2);
                $table->decimal('discount_percentage', 5, 2)->default(0);
                $table->decimal('discount_amount', 10, 2)->default(0);
                $table->decimal('total_amount', 10, 2);
                $table->decimal('paid_amount', 10, 2)->default(0);
                $table->decimal('due_amount', 10, 2);
                $table->enum('status', ['draft', 'sent', 'paid', 'partially_paid', 'overdue', 'cancelled'])->default('draft');
                $table->text('notes')->nullable();
                $table->text('terms_conditions')->nullable();
                $table->timestamps();
                $table->softDeletes();
            });
        }

        // 8. إنشاء جدول بنود الفواتير إذا لم يكن موجوداً
        if (!Schema::hasTable('invoice_items')) {
            Schema::create('invoice_items', function (Blueprint $table) {
                $table->id();
                $table->foreignId('invoice_id')->constrained()->onDelete('cascade');
                $table->string('description');
                $table->decimal('quantity', 8, 2);
                $table->decimal('unit_price', 10, 2);
                $table->decimal('discount_percentage', 5, 2)->default(0);
                $table->decimal('discount_amount', 10, 2)->default(0);
                $table->decimal('tax_percentage', 5, 2)->default(15.00);
                $table->decimal('tax_amount', 10, 2);
                $table->decimal('subtotal', 10, 2);
                $table->decimal('total', 10, 2);
                $table->timestamps();
            });
        }

        // 9. إنشاء جدول المدفوعات إذا لم يكن موجوداً
        if (!Schema::hasTable('payments')) {
            Schema::create('payments', function (Blueprint $table) {
                $table->id();
                $table->foreignId('company_id')->constrained()->onDelete('cascade');
                $table->foreignId('invoice_id')->constrained()->onDelete('cascade');
                $table->foreignId('payment_method_id')->constrained()->onDelete('restrict');
                $table->foreignId('created_by')->constrained('users')->onDelete('restrict');
                $table->decimal('amount', 10, 2);
                $table->date('payment_date');
                $table->string('reference_number')->nullable();
                $table->enum('status', ['completed', 'pending', 'failed', 'refunded'])->default('completed');
                $table->text('notes')->nullable();
                $table->timestamps();
                $table->softDeletes();
            });
        }
    }

    /**
     * Reverse the migrations.
     *
     * @return void
     */
    public function down()
    {
        // حذف الجداول بترتيب عكسي لتجنب مشاكل المفاتيح الأجنبية
        Schema::dropIfExists('payments');
        Schema::dropIfExists('invoice_items');
        Schema::dropIfExists('invoices');
        Schema::dropIfExists('payment_methods');
        Schema::dropIfExists('contract_elevator');
        Schema::dropIfExists('maintenance_contracts');
        Schema::dropIfExists('elevators');
        Schema::dropIfExists('locations');
        Schema::dropIfExists('customers');
        Schema::dropIfExists('branches');
    }
};
