<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * تنفيذ الهجرة لتحديث جدول الأدوار لدعم المستويات المتعددة
     *
     * @return void
     */
    public function up()
    {
        Schema::table('roles', function (Blueprint $table) {
            // إضافة عمود للإشارة إلى الدور الأب (للتسلسل الهرمي)
            if (!Schema::hasColumn('roles', 'parent_id')) {
                $table->foreignId('parent_id')->nullable()->after('id')
                    ->references('id')->on('roles')->onDelete('set null');
            }

            // إضافة عمود لمستوى الدور (للتصنيف الهرمي)
            if (!Schema::hasColumn('roles', 'level')) {
                $table->integer('level')->default(0)->after('description');
            }

            // إضافة عمود للمسار الكامل للدور (للبحث السريع)
            if (!Schema::hasColumn('roles', 'path')) {
                $table->string('path')->nullable()->after('level');
            }

            // إضافة عمود للترتيب
            if (!Schema::hasColumn('roles', 'order')) {
                $table->integer('order')->default(0)->after('path');
            }

            // إضافة عمود لتحديد ما إذا كان الدور افتراضيًا للمستخدمين الجدد
            if (!Schema::hasColumn('roles', 'is_default')) {
                $table->boolean('is_default')->default(false)->after('is_system_role');
            }

            // إضافة فهرس للبحث السريع إذا لم تكن الأعمدة موجودة
            if (
                Schema::hasColumn('roles', 'parent_id') &&
                Schema::hasColumn('roles', 'level') &&
                Schema::hasColumn('roles', 'is_active')
            ) {
                try {
                    $table->index(['parent_id', 'level', 'is_active']);
                } catch (\Exception $e) {
                    // الفهرس قد يكون موجودًا بالفعل
                }
            }
        });
    }

    /**
     * عكس الهجرة
     *
     * @return void
     */
    public function down()
    {
        Schema::table('roles', function (Blueprint $table) {
            // حذف المفتاح الأجنبي إذا كان العمود موجودًا
            if (Schema::hasColumn('roles', 'parent_id')) {
                try {
                    $table->dropForeign(['parent_id']);
                } catch (\Exception $e) {
                    // قد لا يكون المفتاح الأجنبي موجودًا
                }
            }

            // حذف الفهرس إذا كانت الأعمدة موجودة
            if (
                Schema::hasColumn('roles', 'parent_id') &&
                Schema::hasColumn('roles', 'level') &&
                Schema::hasColumn('roles', 'is_active')
            ) {
                try {
                    $table->dropIndex(['parent_id', 'level', 'is_active']);
                } catch (\Exception $e) {
                    // قد لا يكون الفهرس موجودًا
                }
            }

            // حذف الأعمدة إذا كانت موجودة
            $columns = [];
            foreach (['parent_id', 'level', 'path', 'order', 'is_default'] as $column) {
                if (Schema::hasColumn('roles', $column)) {
                    $columns[] = $column;
                }
            }

            if (!empty($columns)) {
                $table->dropColumn($columns);
            }
        });
    }
};
