<?php

namespace App\Services;

use App\Models\Invoice;
use App\Models\Company;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class ZatcaService
{
    /**
     * إنشاء رمز الاستجابة السريعة للفاتورة
     *
     * @param Invoice $invoice
     * @return string
     */
    public function generateQrCode(Invoice $invoice)
    {
        // الحصول على بيانات الشركة
        $company = $invoice->company;

        // تجميع البيانات المطلوبة للرمز
        $data = [
            'seller_name' => $company->name,
            'vat_number' => $company->vat_number,
            'timestamp' => $invoice->invoice_date->format('Y-m-d\TH:i:s\Z'),
            'invoice_total' => number_format($invoice->total_amount, 2, '.', ''),
            'vat_total' => number_format($invoice->tax_amount, 2, '.', '')
        ];

        // تحويل البيانات إلى سلسلة TLV (Tag-Length-Value)
        $tlvString = $this->convertToTlv($data);

        // تشفير البيانات بـ Base64
        $qrCode = base64_encode($tlvString);

        return $qrCode;
    }

    /**
     * تحويل البيانات إلى تنسيق TLV
     *
     * @param array $data
     * @return string
     */
    private function convertToTlv(array $data)
    {
        $tlvString = '';

        // تعيين العلامات (Tags) وفقًا لمواصفات هيئة الزكاة والدخل
        $tags = [
            'seller_name' => 1,
            'vat_number' => 2,
            'timestamp' => 3,
            'invoice_total' => 4,
            'vat_total' => 5
        ];

        foreach ($data as $key => $value) {
            if (isset($tags[$key])) {
                $tag = $tags[$key];
                $length = strlen($value);
                $tlvString .= chr($tag) . chr($length) . $value;
            }
        }

        return $tlvString;
    }

    /**
     * إنشاء ملف XML للفاتورة
     *
     * @param Invoice $invoice
     * @return string
     */
    public function generateXml(Invoice $invoice)
    {
        // الحصول على بيانات الشركة والعميل
        $company = $invoice->company;
        $customer = $invoice->customer;
        $items = $invoice->invoice_items;

        // إنشاء UUID للفاتورة إذا لم يكن موجودًا
        if (!$invoice->zatca_uuid) {
            $invoice->zatca_uuid = $this->generateUuid();
            $invoice->save();
        }

        // بناء محتوى XML
        $xml = new \SimpleXMLElement('<?xml version="1.0" encoding="UTF-8"?><Invoice xmlns="urn:oasis:names:specification:ubl:schema:xsd:Invoice-2" xmlns:cac="urn:oasis:names:specification:ubl:schema:xsd:CommonAggregateComponents-2" xmlns:cbc="urn:oasis:names:specification:ubl:schema:xsd:CommonBasicComponents-2" xmlns:ext="urn:oasis:names:specification:ubl:schema:xsd:CommonExtensionComponents-2"></Invoice>');

        // إضافة معلومات الفاتورة الأساسية
        $xml->addChild('cbc:UBLVersionID', '2.1');
        $xml->addChild('cbc:ProfileID', 'reporting:1.0');
        $xml->addChild('cbc:ID', $invoice->invoice_number);
        $xml->addChild('cbc:UUID', $invoice->zatca_uuid);
        $xml->addChild('cbc:IssueDate', $invoice->invoice_date->format('Y-m-d'));
        $xml->addChild('cbc:IssueTime', $invoice->invoice_date->format('H:i:s'));
        $xml->addChild('cbc:InvoiceTypeCode', $this->mapInvoiceType($invoice->zatca_invoice_type));

        // إضافة معلومات العملة
        $xml->addChild('cbc:DocumentCurrencyCode', 'SAR');
        $xml->addChild('cbc:TaxCurrencyCode', 'SAR');

        // إضافة معلومات البائع (الشركة)
        $accountingSupplierParty = $xml->addChild('cac:AccountingSupplierParty');
        $party = $accountingSupplierParty->addChild('cac:Party');
        $partyIdentification = $party->addChild('cac:PartyIdentification');
        $partyIdentification->addChild('cbc:ID', $company->vat_number)->addAttribute('schemeID', 'CRN');

        $partyName = $party->addChild('cac:PartyName');
        $partyName->addChild('cbc:Name', $company->name);

        $postalAddress = $party->addChild('cac:PostalAddress');
        $postalAddress->addChild('cbc:StreetName', $company->address);
        $postalAddress->addChild('cbc:CityName', $company->city);
        $postalAddress->addChild('cbc:PostalZone', $company->postal_code ?? '00000');
        $postalAddress->addChild('cbc:CountrySubentity', $company->region ?? 'الرياض');

        $country = $postalAddress->addChild('cac:Country');
        $country->addChild('cbc:IdentificationCode', 'SA');

        $taxScheme = $party->addChild('cac:PartyTaxScheme');
        $taxScheme->addChild('cbc:CompanyID', $company->vat_number);
        $taxSchemeElement = $taxScheme->addChild('cac:TaxScheme');
        $taxSchemeElement->addChild('cbc:ID', 'VAT');

        // إضافة معلومات المشتري (العميل)
        $accountingCustomerParty = $xml->addChild('cac:AccountingCustomerParty');
        $customerParty = $accountingCustomerParty->addChild('cac:Party');
        $customerPartyIdentification = $customerParty->addChild('cac:PartyIdentification');
        $customerPartyIdentification->addChild('cbc:ID', $customer->vat_number ?? '000000000')->addAttribute('schemeID', 'CRN');

        $customerPartyName = $customerParty->addChild('cac:PartyName');
        $customerPartyName->addChild('cbc:Name', $customer->name);

        $customerPostalAddress = $customerParty->addChild('cac:PostalAddress');
        $customerPostalAddress->addChild('cbc:StreetName', $customer->address ?? 'غير محدد');
        $customerPostalAddress->addChild('cbc:CityName', $customer->city ?? 'غير محدد');
        $customerPostalAddress->addChild('cbc:PostalZone', $customer->postal_code ?? '00000');
        $customerPostalAddress->addChild('cbc:CountrySubentity', $customer->region ?? 'الرياض');

        $customerCountry = $customerPostalAddress->addChild('cac:Country');
        $customerCountry->addChild('cbc:IdentificationCode', 'SA');

        if ($customer->vat_number) {
            $customerTaxScheme = $customerParty->addChild('cac:PartyTaxScheme');
            $customerTaxScheme->addChild('cbc:CompanyID', $customer->vat_number);
            $customerTaxSchemeElement = $customerTaxScheme->addChild('cac:TaxScheme');
            $customerTaxSchemeElement->addChild('cbc:ID', 'VAT');
        }

        // إضافة معلومات الدفع
        $paymentMeans = $xml->addChild('cac:PaymentMeans');
        $paymentMeans->addChild('cbc:PaymentMeansCode', '10');
        $paymentMeans->addChild('cbc:InstructionNote', 'دفع نقدي');

        // إضافة معلومات الضريبة
        $taxTotal = $xml->addChild('cac:TaxTotal');
        $taxTotal->addChild('cbc:TaxAmount', number_format($invoice->tax_amount, 2, '.', ''))->addAttribute('currencyID', 'SAR');

        $taxSubtotal = $taxTotal->addChild('cac:TaxSubtotal');
        $taxSubtotal->addChild('cbc:TaxableAmount', number_format($invoice->subtotal - $invoice->discount_amount, 2, '.', ''))->addAttribute('currencyID', 'SAR');
        $taxSubtotal->addChild('cbc:TaxAmount', number_format($invoice->tax_amount, 2, '.', ''))->addAttribute('currencyID', 'SAR');

        $taxCategory = $taxSubtotal->addChild('cac:TaxCategory');
        $taxCategory->addChild('cbc:ID', 'S');
        $taxCategory->addChild('cbc:Percent', number_format($invoice->tax_rate, 2, '.', ''));

        $taxScheme = $taxCategory->addChild('cac:TaxScheme');
        $taxScheme->addChild('cbc:ID', 'VAT');

        // إضافة المجاميع
        $legalMonetaryTotal = $xml->addChild('cac:LegalMonetaryTotal');
        $legalMonetaryTotal->addChild('cbc:LineExtensionAmount', number_format($invoice->subtotal, 2, '.', ''))->addAttribute('currencyID', 'SAR');
        $legalMonetaryTotal->addChild('cbc:TaxExclusiveAmount', number_format($invoice->subtotal - $invoice->discount_amount, 2, '.', ''))->addAttribute('currencyID', 'SAR');
        $legalMonetaryTotal->addChild('cbc:TaxInclusiveAmount', number_format($invoice->total_amount, 2, '.', ''))->addAttribute('currencyID', 'SAR');
        $legalMonetaryTotal->addChild('cbc:AllowanceTotalAmount', number_format($invoice->discount_amount, 2, '.', ''))->addAttribute('currencyID', 'SAR');
        $legalMonetaryTotal->addChild('cbc:PayableAmount', number_format($invoice->total_amount, 2, '.', ''))->addAttribute('currencyID', 'SAR');

        // إضافة بنود الفاتورة
        foreach ($items as $index => $item) {
            $invoiceLine = $xml->addChild('cac:InvoiceLine');
            $invoiceLine->addChild('cbc:ID', $index + 1);
            $invoiceLine->addChild('cbc:InvoicedQuantity', number_format($item->quantity, 2, '.', ''))->addAttribute('unitCode', 'EA');
            $invoiceLine->addChild('cbc:LineExtensionAmount', number_format($item->subtotal, 2, '.', ''))->addAttribute('currencyID', 'SAR');

            if ($item->discount_amount > 0) {
                $allowanceCharge = $invoiceLine->addChild('cac:AllowanceCharge');
                $allowanceCharge->addChild('cbc:ChargeIndicator', 'false');
                $allowanceCharge->addChild('cbc:AllowanceChargeReason', 'خصم');
                $allowanceCharge->addChild('cbc:Amount', number_format($item->discount_amount, 2, '.', ''))->addAttribute('currencyID', 'SAR');
            }

            $taxTotal = $invoiceLine->addChild('cac:TaxTotal');
            $taxTotal->addChild('cbc:TaxAmount', number_format($item->tax_amount, 2, '.', ''))->addAttribute('currencyID', 'SAR');
            $taxTotal->addChild('cbc:RoundingAmount', number_format($item->total, 2, '.', ''))->addAttribute('currencyID', 'SAR');

            $item = $invoiceLine->addChild('cac:Item');
            $item->addChild('cbc:Name', $item->description);

            $classifiedTaxCategory = $item->addChild('cac:ClassifiedTaxCategory');
            $classifiedTaxCategory->addChild('cbc:ID', 'S');
            $classifiedTaxCategory->addChild('cbc:Percent', number_format($item->tax_percentage, 2, '.', ''));

            $taxScheme = $classifiedTaxCategory->addChild('cac:TaxScheme');
            $taxScheme->addChild('cbc:ID', 'VAT');

            $price = $invoiceLine->addChild('cac:Price');
            $price->addChild('cbc:PriceAmount', number_format($item->unit_price, 2, '.', ''))->addAttribute('currencyID', 'SAR');
        }

        // حفظ ملف XML
        $xmlString = $xml->asXML();
        $fileName = 'invoice_' . $invoice->invoice_number . '.xml';
        $path = 'invoices/' . $invoice->company_id . '/xml/' . $fileName;

        Storage::disk('public')->put($path, $xmlString);

        // تحديث مسار الملف في الفاتورة
        $invoice->zatca_xml_path = $path;
        $invoice->save();

        return $path;
    }

    /**
     * إنشاء معرف فريد للفاتورة
     *
     * @return string
     */
    private function generateUuid()
    {
        return sprintf(
            '%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0xffff)
        );
    }

    /**
     * تحويل نوع الفاتورة إلى الرمز المناسب
     *
     * @param string $invoiceType
     * @return string
     */
    private function mapInvoiceType($invoiceType)
    {
        $types = [
            'standard' => '388',
            'simplified' => '383',
            'debit_note' => '383',
            'credit_note' => '381'
        ];

        return $types[$invoiceType] ?? '388';
    }

    /**
     * إرسال الفاتورة إلى هيئة الزكاة والدخل
     *
     * @param Invoice $invoice
     * @return bool
     */
    public function reportInvoice(Invoice $invoice)
    {
        try {
            // التأكد من وجود ملف XML
            if (!$invoice->zatca_xml_path) {
                $this->generateXml($invoice);
            }

            // التأكد من وجود رمز الاستجابة السريعة
            if (!$invoice->zatca_qr_code) {
                $invoice->zatca_qr_code = $this->generateQrCode($invoice);
                $invoice->save();
            }

            // هنا يتم إضافة كود الاتصال بواجهة برمجة التطبيقات الخاصة بهيئة الزكاة والدخل
            // ملاحظة: هذا مجرد مثال وسيحتاج إلى تعديل وفقًا لمتطلبات واجهة برمجة التطبيقات الفعلية

            // تحديث حالة الإبلاغ
            $invoice->zatca_reporting_status = 'reported';
            $invoice->zatca_reported_at = Carbon::now();
            $invoice->save();

            return true;
        } catch (\Exception $e) {
            Log::error('فشل في إرسال الفاتورة إلى هيئة الزكاة والدخل: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * التحقق من حالة الفاتورة لدى هيئة الزكاة والدخل
     *
     * @param Invoice $invoice
     * @return array
     */
    public function checkInvoiceStatus(Invoice $invoice)
    {
        try {
            // هنا يتم إضافة كود الاتصال بواجهة برمجة التطبيقات الخاصة بهيئة الزكاة والدخل للتحقق من حالة الفاتورة
            // ملاحظة: هذا مجرد مثال وسيحتاج إلى تعديل وفقًا لمتطلبات واجهة برمجة التطبيقات الفعلية

            return [
                'status' => $invoice->zatca_reporting_status,
                'reported_at' => $invoice->zatca_reported_at,
                'compliance_status' => $invoice->zatca_compliance_status,
                'clearance_status' => $invoice->zatca_clearance_status
            ];
        } catch (\Exception $e) {
            Log::error('فشل في التحقق من حالة الفاتورة لدى هيئة الزكاة والدخل: ' . $e->getMessage());
            return [
                'status' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }
}
