<?php

namespace App\Services\Reports;

use App\Models\InvoiceInstallment;
use App\Models\Invoice;
use App\Models\Customer;
use App\Models\Payment;
use App\Helpers\InstallmentHelper;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class InstallmentReportService
{
    /**
     * إنشاء تقرير الأقساط المستحقة.
     *
     * @param int $companyId
     * @param array $filters
     * @return array
     */
    public function generateUpcomingInstallmentsReport($companyId, array $filters = [])
    {
        try {
            // تحديد الفترة الزمنية
            $startDate = isset($filters['start_date']) ? Carbon::parse($filters['start_date']) : Carbon::now();
            $endDate = isset($filters['end_date']) ? Carbon::parse($filters['end_date']) : Carbon::now()->addDays(30);

            // تحديد العميل
            $customerId = $filters['customer_id'] ?? null;

            // بناء الاستعلام
            $query = InvoiceInstallment::where('status', 'pending')
                ->whereBetween('due_date', [$startDate, $endDate])
                ->whereHas('invoice', function ($query) use ($companyId) {
                    $query->where('company_id', $companyId);
                })
                ->with(['invoice', 'invoice.customer']);

            // تطبيق فلتر العميل
            if ($customerId) {
                $query->whereHas('invoice', function ($query) use ($customerId) {
                    $query->where('customer_id', $customerId);
                });
            }

            // الحصول على الأقساط
            $installments = $query->orderBy('due_date')->get();

            // تجميع البيانات حسب العميل
            $customerData = [];
            $totals = [
                'count' => 0,
                'amount' => 0,
            ];

            foreach ($installments as $installment) {
                $customerId = $installment->invoice->customer_id;
                $customerName = $installment->invoice->customer->name;

                if (!isset($customerData[$customerId])) {
                    $customerData[$customerId] = [
                        'customer_id' => $customerId,
                        'customer_name' => $customerName,
                        'installments' => [],
                        'total_amount' => 0,
                        'count' => 0,
                    ];
                }

                $daysRemaining = Carbon::now()->startOfDay()->diffInDays($installment->due_date, false);

                $customerData[$customerId]['installments'][] = [
                    'id' => $installment->id,
                    'invoice_id' => $installment->invoice_id,
                    'invoice_number' => $installment->invoice->invoice_number,
                    'amount' => $installment->amount,
                    'due_date' => $installment->due_date->format('Y-m-d'),
                    'days_remaining' => $daysRemaining,
                    'notes' => $installment->notes,
                ];

                $customerData[$customerId]['total_amount'] += $installment->amount;
                $customerData[$customerId]['count']++;

                $totals['count']++;
                $totals['amount'] += $installment->amount;
            }

            // تحويل المصفوفة الترابطية إلى مصفوفة عادية
            $customerData = array_values($customerData);

            return [
                'success' => true,
                'data' => [
                    'customers' => $customerData,
                    'totals' => $totals,
                    'period' => [
                        'start_date' => $startDate->format('Y-m-d'),
                        'end_date' => $endDate->format('Y-m-d'),
                    ],
                ],
            ];
        } catch (\Exception $e) {
            \Log::error('خطأ في إنشاء تقرير الأقساط المستحقة: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'حدث خطأ أثناء إنشاء التقرير: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * إنشاء تقرير الأقساط المتأخرة.
     *
     * @param int $companyId
     * @param array $filters
     * @return array
     */
    public function generateOverdueInstallmentsReport($companyId, array $filters = [])
    {
        try {
            // تحديد العميل
            $customerId = $filters['customer_id'] ?? null;

            // تحديد فترة التأخير
            $minDaysOverdue = $filters['min_days_overdue'] ?? null;
            $maxDaysOverdue = $filters['max_days_overdue'] ?? null;

            // بناء الاستعلام
            $query = InvoiceInstallment::where(function ($query) {
                $query->where('status', 'overdue')
                    ->orWhere(function ($q) {
                        $q->where('status', 'pending')
                            ->where('due_date', '<', Carbon::now()->startOfDay());
                    });
            })
                ->where('paid_amount', '<', DB::raw('amount'))
                ->whereHas('invoice', function ($query) use ($companyId) {
                    $query->where('company_id', $companyId);
                })
                ->with(['invoice', 'invoice.customer']);

            // تطبيق فلتر العميل
            if ($customerId) {
                $query->whereHas('invoice', function ($query) use ($customerId) {
                    $query->where('customer_id', $customerId);
                });
            }

            // الحصول على الأقساط
            $installments = $query->orderBy('due_date')->get();

            // تصفية الأقساط حسب فترة التأخير
            if ($minDaysOverdue !== null || $maxDaysOverdue !== null) {
                $installments = $installments->filter(function ($installment) use ($minDaysOverdue, $maxDaysOverdue) {
                    $daysOverdue = Carbon::now()->startOfDay()->diffInDays($installment->due_date, false) * -1;

                    if ($minDaysOverdue !== null && $daysOverdue < $minDaysOverdue) {
                        return false;
                    }

                    if ($maxDaysOverdue !== null && $daysOverdue > $maxDaysOverdue) {
                        return false;
                    }

                    return true;
                });
            }

            // تجميع البيانات حسب العميل
            $customerData = [];
            $totals = [
                'count' => 0,
                'amount' => 0,
                'remaining_amount' => 0,
            ];

            foreach ($installments as $installment) {
                $customerId = $installment->invoice->customer_id;
                $customerName = $installment->invoice->customer->name;

                if (!isset($customerData[$customerId])) {
                    $customerData[$customerId] = [
                        'customer_id' => $customerId,
                        'customer_name' => $customerName,
                        'installments' => [],
                        'total_amount' => 0,
                        'total_remaining' => 0,
                        'count' => 0,
                    ];
                }

                $daysOverdue = Carbon::now()->startOfDay()->diffInDays($installment->due_date, false) * -1;
                $remainingAmount = $installment->amount - $installment->paid_amount;

                $customerData[$customerId]['installments'][] = [
                    'id' => $installment->id,
                    'invoice_id' => $installment->invoice_id,
                    'invoice_number' => $installment->invoice->invoice_number,
                    'amount' => $installment->amount,
                    'paid_amount' => $installment->paid_amount,
                    'remaining_amount' => $remainingAmount,
                    'due_date' => $installment->due_date->format('Y-m-d'),
                    'days_overdue' => $daysOverdue,
                    'notes' => $installment->notes,
                ];

                $customerData[$customerId]['total_amount'] += $installment->amount;
                $customerData[$customerId]['total_remaining'] += $remainingAmount;
                $customerData[$customerId]['count']++;

                $totals['count']++;
                $totals['amount'] += $installment->amount;
                $totals['remaining_amount'] += $remainingAmount;
            }

            // تحويل المصفوفة الترابطية إلى مصفوفة عادية
            $customerData = array_values($customerData);

            return [
                'success' => true,
                'data' => [
                    'customers' => $customerData,
                    'totals' => $totals,
                ],
            ];
        } catch (\Exception $e) {
            \Log::error('خطأ في إنشاء تقرير الأقساط المتأخرة: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'حدث خطأ أثناء إنشاء التقرير: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * إنشاء تقرير مدفوعات الأقساط.
     *
     * @param int $companyId
     * @param array $filters
     * @return array
     */
    public function generateInstallmentPaymentsReport($companyId, array $filters = [])
    {
        try {
            // تحديد الفترة الزمنية
            $startDate = isset($filters['start_date']) ? Carbon::parse($filters['start_date']) : Carbon::now()->subDays(30);
            $endDate = isset($filters['end_date']) ? Carbon::parse($filters['end_date']) : Carbon::now();

            // تحديد العميل
            $customerId = $filters['customer_id'] ?? null;

            // بناء الاستعلام
            $query = Payment::whereHas('installment')
                ->where('company_id', $companyId)
                ->whereBetween('payment_date', [$startDate, $endDate])
                ->with(['installment', 'installment.invoice', 'installment.invoice.customer', 'paymentMethod']);

            // تطبيق فلتر العميل
            if ($customerId) {
                $query->whereHas('installment.invoice', function ($query) use ($customerId) {
                    $query->where('customer_id', $customerId);
                });
            }

            // الحصول على المدفوعات
            $payments = $query->orderBy('payment_date', 'desc')->get();

            // تجميع البيانات حسب العميل
            $customerData = [];
            $totals = [
                'count' => 0,
                'amount' => 0,
            ];

            foreach ($payments as $payment) {
                $customerId = $payment->installment->invoice->customer_id;
                $customerName = $payment->installment->invoice->customer->name;

                if (!isset($customerData[$customerId])) {
                    $customerData[$customerId] = [
                        'customer_id' => $customerId,
                        'customer_name' => $customerName,
                        'payments' => [],
                        'total_amount' => 0,
                        'count' => 0,
                    ];
                }

                $customerData[$customerId]['payments'][] = [
                    'id' => $payment->id,
                    'installment_id' => $payment->installment_id,
                    'invoice_id' => $payment->installment->invoice_id,
                    'invoice_number' => $payment->installment->invoice->invoice_number,
                    'amount' => $payment->amount,
                    'payment_date' => $payment->payment_date->format('Y-m-d'),
                    'payment_method' => $payment->paymentMethod->name,
                    'reference_number' => $payment->reference_number,
                    'notes' => $payment->notes,
                ];

                $customerData[$customerId]['total_amount'] += $payment->amount;
                $customerData[$customerId]['count']++;

                $totals['count']++;
                $totals['amount'] += $payment->amount;
            }

            // تحويل المصفوفة الترابطية إلى مصفوفة عادية
            $customerData = array_values($customerData);

            return [
                'success' => true,
                'data' => [
                    'customers' => $customerData,
                    'totals' => $totals,
                    'period' => [
                        'start_date' => $startDate->format('Y-m-d'),
                        'end_date' => $endDate->format('Y-m-d'),
                    ],
                ],
            ];
        } catch (\Exception $e) {
            \Log::error('خطأ في إنشاء تقرير مدفوعات الأقساط: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'حدث خطأ أثناء إنشاء التقرير: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * إنشاء تقرير أداء تحصيل الأقساط.
     *
     * @param int $companyId
     * @param array $filters
     * @return array
     */
    public function generateInstallmentCollectionPerformanceReport($companyId, array $filters = [])
    {
        try {
            // تحديد الفترة الزمنية
            $year = $filters['year'] ?? Carbon::now()->year;
            $startDate = Carbon::createFromDate($year, 1, 1)->startOfYear();
            $endDate = Carbon::createFromDate($year, 12, 31)->endOfYear();

            // الحصول على الأقساط المستحقة خلال الفترة
            $dueInstallments = InvoiceInstallment::whereBetween('due_date', [$startDate, $endDate])
                ->whereHas('invoice', function ($query) use ($companyId) {
                    $query->where('company_id', $companyId);
                })
                ->get();

            // تجميع البيانات حسب الشهر
            $monthlyData = [];

            for ($month = 1; $month <= 12; $month++) {
                $monthlyData[$month] = [
                    'month' => $month,
                    'month_name' => Carbon::createFromDate($year, $month, 1)->format('F'),
                    'due_count' => 0,
                    'due_amount' => 0,
                    'paid_count' => 0,
                    'paid_amount' => 0,
                    'overdue_count' => 0,
                    'overdue_amount' => 0,
                    'collection_rate' => 0,
                ];
            }

            // حساب الأقساط المستحقة والمدفوعة والمتأخرة
            foreach ($dueInstallments as $installment) {
                $month = $installment->due_date->month;

                $monthlyData[$month]['due_count']++;
                $monthlyData[$month]['due_amount'] += $installment->amount;

                if ($installment->isPaid()) {
                    $monthlyData[$month]['paid_count']++;
                    $monthlyData[$month]['paid_amount'] += $installment->amount;
                } elseif ($installment->isOverdue()) {
                    $monthlyData[$month]['overdue_count']++;
                    $monthlyData[$month]['overdue_amount'] += ($installment->amount - $installment->paid_amount);
                }
            }

            // حساب معدل التحصيل
            foreach ($monthlyData as &$data) {
                if ($data['due_amount'] > 0) {
                    $data['collection_rate'] = round(($data['paid_amount'] / $data['due_amount']) * 100, 2);
                }
            }

            // حساب الإجماليات
            $totals = [
                'due_count' => 0,
                'due_amount' => 0,
                'paid_count' => 0,
                'paid_amount' => 0,
                'overdue_count' => 0,
                'overdue_amount' => 0,
                'collection_rate' => 0,
            ];

            foreach ($monthlyData as $data) {
                $totals['due_count'] += $data['due_count'];
                $totals['due_amount'] += $data['due_amount'];
                $totals['paid_count'] += $data['paid_count'];
                $totals['paid_amount'] += $data['paid_amount'];
                $totals['overdue_count'] += $data['overdue_count'];
                $totals['overdue_amount'] += $data['overdue_amount'];
            }

            if ($totals['due_amount'] > 0) {
                $totals['collection_rate'] = round(($totals['paid_amount'] / $totals['due_amount']) * 100, 2);
            }

            // تحويل المصفوفة الترابطية إلى مصفوفة عادية
            $monthlyData = array_values($monthlyData);

            return [
                'success' => true,
                'data' => [
                    'monthly' => $monthlyData,
                    'totals' => $totals,
                    'year' => $year,
                ],
            ];
        } catch (\Exception $e) {
            \Log::error('خطأ في إنشاء تقرير أداء تحصيل الأقساط: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'حدث خطأ أثناء إنشاء التقرير: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * إنشاء تقرير الأقساط حسب العميل.
     *
     * @param int $companyId
     * @param array $filters
     * @return array
     */
    public function generateInstallmentsByCustomerReport($companyId, array $filters = [])
    {
        try {
            // تحديد حالة القسط
            $status = $filters['status'] ?? null;

            // بناء الاستعلام
            $query = DB::table('invoice_installments')
                ->join('invoices', 'invoice_installments.invoice_id', '=', 'invoices.id')
                ->join('customers', 'invoices.customer_id', '=', 'customers.id')
                ->where('invoices.company_id', $companyId)
                ->select(
                    'customers.id as customer_id',
                    'customers.name as customer_name',
                    DB::raw('COUNT(invoice_installments.id) as count'),
                    DB::raw('SUM(invoice_installments.amount) as total_amount'),
                    DB::raw('SUM(invoice_installments.paid_amount) as paid_amount'),
                    DB::raw('SUM(invoice_installments.amount - invoice_installments.paid_amount) as remaining_amount')
                )
                ->groupBy('customers.id', 'customers.name');

            // تطبيق فلتر الحالة
            if ($status) {
                $query->where('invoice_installments.status', $status);
            }

            // الحصول على البيانات
            $customersData = $query->orderBy('remaining_amount', 'desc')->get();

            // حساب الإجماليات
            $totals = [
                'count' => 0,
                'total_amount' => 0,
                'paid_amount' => 0,
                'remaining_amount' => 0,
            ];

            foreach ($customersData as $customer) {
                $totals['count'] += $customer->count;
                $totals['total_amount'] += $customer->total_amount;
                $totals['paid_amount'] += $customer->paid_amount;
                $totals['remaining_amount'] += $customer->remaining_amount;
            }

            return [
                'success' => true,
                'data' => [
                    'customers' => $customersData,
                    'totals' => $totals,
                ],
            ];
        } catch (\Exception $e) {
            \Log::error('خطأ في إنشاء تقرير الأقساط حسب العميل: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'حدث خطأ أثناء إنشاء التقرير: ' . $e->getMessage(),
            ];
        }
    }
}
