<?php

namespace App\Services;

use App\Models\Notification;
use App\Models\User;
use App\Models\MaintenanceSchedule;
use App\Models\MaintenanceContract;
use App\Models\Elevator;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use App\Mail\MaintenanceNotificationMail;

class NotificationService
{
    /**
     * Create a new notification.
     *
     * @param  array  $data
     * @return \App\Models\Notification
     */
    public function createNotification(array $data)
    {
        // Ensure data is properly serialized to JSON if it's an array
        if (isset($data['data']) && is_array($data['data'])) {
            $data['data'] = json_encode($data['data'], JSON_UNESCAPED_UNICODE);
        }

        $notification = Notification::create($data);

        // Log successful notification creation
        Log::info('تم إنشاء إشعار جديد', [
            'notification_id' => $notification->id,
            'type' => $notification->type,
            'title' => $notification->title,
            'user_id' => $notification->user_id,
            'status' => $notification->status
        ]);

        return $notification;
    }

    /**
     * Send notification to user.
     *
     * @param  \App\Models\Notification  $notification
     * @return bool
     */
    public function sendNotification(Notification $notification)
    {
        try {
            // Mark notification as sent
            $notification->markAsSent();

            // Log notification sending attempt
            Log::info('جاري إرسال الإشعار', [
                'notification_id' => $notification->id,
                'type' => $notification->type,
                'title' => $notification->title,
                'user_id' => $notification->user_id
            ]);

            // Send email notification if user has email
            if ($notification->user && $notification->user->email) {
                Mail::to($notification->user->email)->send(new MaintenanceNotificationMail($notification));

                // إضافة سجل للنشاط - تم تعطيل مؤقتاً (يتطلب تثبيت حزمة spatie/laravel-activitylog)
                // activity()
                //     ->performedOn($notification)
                //     ->causedBy(auth()->user() ?? $notification->user)
                //     ->withProperties([
                //         'type' => 'notification_sent',
                //         'notification_id' => $notification->id,
                //         'notification_type' => $notification->type,
                //         'recipient' => $notification->user->email
                //     ])
                //     ->log('تم إرسال إشعار بالبريد الإلكتروني');

                // Log successful email sending
                Log::info('تم إرسال الإشعار بنجاح بالبريد الإلكتروني', [
                    'notification_id' => $notification->id,
                    'email' => $notification->user->email
                ]);
            } else {
                // Log notification without email
                Log::info('تم تحديث حالة الإشعار إلى مرسل (بدون بريد إلكتروني)', [
                    'notification_id' => $notification->id,
                    'reason' => $notification->user ? 'المستخدم ليس لديه بريد إلكتروني' : 'لا يوجد مستخدم مرتبط'
                ]);
            }

            return true;
        } catch (\Exception $e) {
            Log::error('فشل في إرسال الإشعار: ' . $e->getMessage(), [
                'notification_id' => $notification->id,
                'exception' => get_class($e),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
            return false;
        }
    }

    /**
     * التحقق من جدولة الصيانة القادمة وإنشاء إشعارات متقدمة.
     *
     * @param  int  $daysInAdvance
     * @return int عدد الإشعارات التي تم إنشاؤها
     */
    public function checkUpcomingMaintenanceSchedules($daysInAdvance = 7)
    {
        $count = 0;
        $today = Carbon::today();
        $futureDate = $today->copy()->addDays($daysInAdvance);
        $urgentDate = $today->copy()->addDays(2); // تاريخ للإشعارات العاجلة (خلال يومين)

        // الحصول على جميع جدولات الصيانة المقررة بين اليوم والتاريخ المستقبلي
        $schedules = MaintenanceSchedule::where('status', 'scheduled')
            ->whereBetween('scheduled_date', [$today, $futureDate])
            ->with(['elevator', 'elevator.customer', 'elevator.building', 'maintenanceContract', 'company'])
            ->get();

        foreach ($schedules as $schedule) {
            // تحديد مستوى الأولوية بناءً على قرب موعد الصيانة
            $priority = 'normal';
            $type = 'maintenance_due';

            // إذا كان موعد الصيانة خلال يومين، فالأولوية عالية
            if ($schedule->scheduled_date->between($today, $urgentDate)) {
                $priority = 'high';
                $type = 'maintenance_urgent';
            }

            // التحقق من وجود إشعار مسبق لهذه الجدولة
            $existingNotification = Notification::where('notifiable_type', 'App\\Models\\MaintenanceSchedule')
                ->where('notifiable_id', $schedule->id)
                ->where('type', $type)
                ->first();

            if (!$existingNotification) {
                // إنشاء رسالة مفصلة تتضمن معلومات أكثر
                $daysUntilMaintenance = $today->diffInDays($schedule->scheduled_date);
                $customerInfo = $schedule->elevator->customer ? $schedule->elevator->customer->name : 'غير محدد';
                $buildingInfo = $schedule->elevator->building ? $schedule->elevator->building->name . ' - ' . $schedule->elevator->building->address : 'غير محدد';

                $messageDetails = "تفاصيل المصعد: " . $schedule->elevator->model .
                    "\nالرقم التسلسلي: " . $schedule->elevator->serial_number .
                    "\nالعميل: " . $customerInfo .
                    "\nالموقع: " . $buildingInfo .
                    "\nتاريخ الصيانة: " . $schedule->scheduled_date->format('Y-m-d') .
                    "\nالمتبقي: " . $daysUntilMaintenance . " يوم";

                if ($schedule->maintenanceContract) {
                    $messageDetails .= "\nرقم العقد: " . $schedule->maintenanceContract->contract_number;
                }

                // إنشاء إشعار للمسؤولين
                $admins = User::whereHas('roles', function ($query) {
                    $query->whereIn('name', ['Company Admin', 'Branch Manager', 'Maintenance Manager']);
                })->where('company_id', $schedule->company_id)->get();

                foreach ($admins as $admin) {
                    $title = $priority === 'high' ? 'صيانة عاجلة خلال يومين!' : 'صيانة مستحقة قريباً';

                    $this->createNotification([
                        'company_id' => $schedule->company_id,
                        'user_id' => $admin->id,
                        'type' => $type,
                        'notifiable_type' => 'App\\Models\\MaintenanceSchedule',
                        'notifiable_id' => $schedule->id,
                        'title' => $title,
                        'message' => $priority === 'high'
                            ? 'تنبيه عاجل: هناك صيانة مستحقة للمصعد ' . $schedule->elevator->model . ' خلال يومين بتاريخ ' . $schedule->scheduled_date->format('Y-m-d')
                            : 'هناك صيانة مستحقة للمصعد ' . $schedule->elevator->model . ' بتاريخ ' . $schedule->scheduled_date->format('Y-m-d'),
                        'data' => [
                            'schedule_id' => $schedule->id,
                            'elevator_id' => $schedule->elevator_id,
                            'contract_id' => $schedule->maintenance_contract_id,
                            'scheduled_date' => $schedule->scheduled_date->format('Y-m-d'),
                            'days_until_maintenance' => $daysUntilMaintenance,
                            'details' => $messageDetails,
                            'customer' => $customerInfo,
                            'location' => $buildingInfo,
                        ],
                        'priority' => $priority,
                        'status' => 'pending',
                    ]);
                    $count++;
                }

                // إنشاء إشعار للفنيين
                $technicians = User::whereHas('roles', function ($query) {
                    $query->where('name', 'Technician');
                })->where('company_id', $schedule->company_id)->get();

                foreach ($technicians as $technician) {
                    $title = $priority === 'high' ? 'صيانة عاجلة خلال يومين!' : 'صيانة مستحقة قريباً';

                    $this->createNotification([
                        'company_id' => $schedule->company_id,
                        'user_id' => $technician->id,
                        'type' => $type,
                        'notifiable_type' => 'App\\Models\\MaintenanceSchedule',
                        'notifiable_id' => $schedule->id,
                        'title' => $title,
                        'message' => $priority === 'high'
                            ? 'تنبيه عاجل: هناك صيانة مستحقة للمصعد ' . $schedule->elevator->model . ' خلال يومين بتاريخ ' . $schedule->scheduled_date->format('Y-m-d')
                            : 'هناك صيانة مستحقة للمصعد ' . $schedule->elevator->model . ' بتاريخ ' . $schedule->scheduled_date->format('Y-m-d'),
                        'data' => [
                            'schedule_id' => $schedule->id,
                            'elevator_id' => $schedule->elevator_id,
                            'contract_id' => $schedule->maintenance_contract_id,
                            'scheduled_date' => $schedule->scheduled_date->format('Y-m-d'),
                            'days_until_maintenance' => $daysUntilMaintenance,
                            'details' => $messageDetails,
                            'customer' => $customerInfo,
                            'location' => $buildingInfo,
                        ],
                        'priority' => $priority,
                        'status' => 'pending',
                    ]);
                    $count++;
                }
            }
        }

        return $count;
    }

    /**
     * التحقق من جدولة الصيانة المتأخرة وإنشاء إشعارات متقدمة.
     *
     * @return int عدد الإشعارات التي تم إنشاؤها
     */
    public function checkOverdueMaintenanceSchedules()
    {
        try {
            $count = 0;
            $today = Carbon::today();

            // الحصول على جميع جدولات الصيانة المتأخرة
            $schedules = MaintenanceSchedule::where('status', 'scheduled')
                ->where('scheduled_date', '<', $today)
                ->with(['elevator', 'elevator.customer', 'elevator.building', 'maintenanceContract', 'company'])
                ->get();

            foreach ($schedules as $schedule) {
                // حساب عدد أيام التأخير
                $daysOverdue = $today->diffInDays($schedule->scheduled_date);

                // تحديد مستوى الأولوية بناءً على مدة التأخير
                $priority = 'high';
                $type = 'maintenance_overdue';

                // إذا كان التأخير أكثر من 7 أيام، فالأولوية حرجة
                if ($daysOverdue > 7) {
                    $priority = 'critical';
                    $type = 'maintenance_critical_overdue';
                }

                // التحقق من وجود إشعار مسبق لهذه الجدولة
                $existingNotification = Notification::where('notifiable_type', 'App\\Models\\MaintenanceSchedule')
                    ->where('notifiable_id', $schedule->id)
                    ->where('type', $type)
                    ->first();

                if (!$existingNotification) {
                    // إنشاء رسالة مفصلة تتضمن معلومات أكثر
                    $customerInfo = $schedule->elevator->customer ? $schedule->elevator->customer->name : 'غير محدد';
                    $buildingInfo = $schedule->elevator->building ?
                        ($schedule->elevator->building->name . ' - ' . $schedule->elevator->building->address) :
                        'غير محدد';
                    $contractInfo = $schedule->maintenanceContract ? $schedule->maintenanceContract->contract_number : 'غير محدد';

                    $messageDetails = "تفاصيل المصعد: " . $schedule->elevator->model .
                        "\nالرقم التسلسلي: " . $schedule->elevator->serial_number .
                        "\nالعميل: " . $customerInfo .
                        "\nالموقع: " . $buildingInfo .
                        "\nتاريخ الصيانة المفترض: " . $schedule->scheduled_date->format('Y-m-d') .
                        "\nعدد أيام التأخير: " . $daysOverdue . " يوم" .
                        "\nرقم العقد: " . $contractInfo;

                    // إنشاء إشعار للمسؤولين والمدراء
                    $admins = User::whereHas('roles', function ($query) {
                        $query->whereIn('name', ['Company Admin', 'Branch Manager', 'Maintenance Manager']);
                    })->where('company_id', $schedule->company_id)->get();

                    foreach ($admins as $admin) {
                        $title = $priority === 'critical' ? 'تحذير: صيانة متأخرة بشكل حرج!' : 'صيانة متأخرة';

                        $notificationData = [
                            'company_id' => $schedule->company_id,
                            'user_id' => $admin->id,
                            'type' => $type,
                            'notifiable_type' => 'App\\Models\\MaintenanceSchedule',
                            'notifiable_id' => $schedule->id,
                            'title' => $title,
                            'message' => $priority === 'critical'
                                ? 'تحذير هام: هناك صيانة متأخرة بشكل حرج للمصعد ' . $schedule->elevator->model . ' متأخرة لمدة ' . $daysOverdue . ' يوم. كانت مستحقة بتاريخ ' . $schedule->scheduled_date->format('Y-m-d')
                                : 'هناك صيانة متأخرة للمصعد ' . $schedule->elevator->model . ' متأخرة لمدة ' . $daysOverdue . ' يوم. كانت مستحقة بتاريخ ' . $schedule->scheduled_date->format('Y-m-d'),
                            'data' => [
                                'schedule_id' => $schedule->id,
                                'elevator_id' => $schedule->elevator_id,
                                'contract_id' => $schedule->maintenance_contract_id,
                                'scheduled_date' => $schedule->scheduled_date->format('Y-m-d'),
                                'days_overdue' => $daysOverdue,
                                'details' => $messageDetails,
                                'customer' => $customerInfo,
                                'location' => $buildingInfo,
                                'contract_number' => $contractInfo,
                                'severity' => $priority === 'critical' ? 'حرج' : 'عالي',
                            ],
                            'priority' => $priority,
                            'status' => 'pending',
                        ];

                        $this->createNotification($notificationData);
                        $count++;
                    }

                    // إنشاء إشعار للفنيين
                    $technicians = User::whereHas('roles', function ($query) {
                        $query->where('name', 'Technician');
                    })->where('company_id', $schedule->company_id)->get();

                    foreach ($technicians as $technician) {
                        $title = $priority === 'critical' ? 'تحذير: صيانة متأخرة بشكل حرج!' : 'صيانة متأخرة';

                        $notificationData = [
                            'company_id' => $schedule->company_id,
                            'user_id' => $technician->id,
                            'type' => $type,
                            'notifiable_type' => 'App\\Models\\MaintenanceSchedule',
                            'notifiable_id' => $schedule->id,
                            'title' => $title,
                            'message' => $priority === 'critical'
                                ? 'تحذير هام: هناك صيانة متأخرة بشكل حرج للمصعد ' . $schedule->elevator->model . ' متأخرة لمدة ' . $daysOverdue . ' يوم. كانت مستحقة بتاريخ ' . $schedule->scheduled_date->format('Y-m-d')
                                : 'هناك صيانة متأخرة للمصعد ' . $schedule->elevator->model . ' متأخرة لمدة ' . $daysOverdue . ' يوم. كانت مستحقة بتاريخ ' . $schedule->scheduled_date->format('Y-m-d'),
                            'data' => [
                                'schedule_id' => $schedule->id,
                                'elevator_id' => $schedule->elevator_id,
                                'contract_id' => $schedule->maintenance_contract_id,
                                'scheduled_date' => $schedule->scheduled_date->format('Y-m-d'),
                                'days_overdue' => $daysOverdue,
                                'details' => $messageDetails,
                                'customer' => $customerInfo,
                                'location' => $buildingInfo,
                                'contract_number' => $contractInfo,
                                'severity' => $priority === 'critical' ? 'حرج' : 'عالي',
                            ],
                            'priority' => $priority,
                            'status' => 'pending',
                        ];

                        $this->createNotification($notificationData);
                        $count++;
                    }

                    // إذا كان التأخير حرجاً، أرسل إشعاراً للمدير العام أيضاً
                    if ($priority === 'critical') {
                        $generalManagers = User::whereHas('roles', function ($query) {
                            $query->where('name', 'General Manager');
                        })->where('company_id', $schedule->company_id)->get();

                        foreach ($generalManagers as $manager) {
                            $this->createNotification([
                                'company_id' => $schedule->company_id,
                                'user_id' => $manager->id,
                                'type' => $type,
                                'notifiable_type' => 'App\\Models\\MaintenanceSchedule',
                                'notifiable_id' => $schedule->id,
                                'title' => 'تحذير: صيانة متأخرة بشكل حرج!',
                                'message' => 'تحذير هام: هناك صيانة متأخرة بشكل حرج للمصعد ' . $schedule->elevator->model . ' متأخرة لمدة ' . $daysOverdue . ' يوم. كانت مستحقة بتاريخ ' . $schedule->scheduled_date->format('Y-m-d'),
                                'data' => [
                                    'schedule_id' => $schedule->id,
                                    'elevator_id' => $schedule->elevator_id,
                                    'contract_id' => $schedule->maintenance_contract_id,
                                    'scheduled_date' => $schedule->scheduled_date->format('Y-m-d'),
                                    'days_overdue' => $daysOverdue,
                                    'details' => $messageDetails,
                                    'customer' => $customerInfo,
                                    'location' => $buildingInfo,
                                    'contract_number' => $contractInfo,
                                    'severity' => 'حرج',
                                    'requires_immediate_action' => true,
                                ],
                                'priority' => $priority,
                                'status' => 'pending',
                            ]);
                            $count++;
                        }
                    }
                }
            }

            return $count;
        } catch (\Exception $e) {
            // تسجيل الخطأ بشكل صحيح
            $errorMessage = $e->getMessage();
            // التحقق من نوع الرسالة وتحويلها إلى نص إذا كانت مصفوفة
            if (is_array($errorMessage)) {
                $errorMessage = json_encode($errorMessage, JSON_UNESCAPED_UNICODE);
            }

            Log::error('فشل التحقق من جدولة الصيانة المتأخرة: ' . $errorMessage, [
                'exception' => get_class($e),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);

            return 0;
        }
    }

    /**
     * Check for expiring maintenance contracts and create notifications.
     *
     * @param  int  $daysInAdvance
     * @return int Number of notifications created
     */
    public function checkExpiringContracts($daysInAdvance = 30)
    {
        $count = 0;
        $today = Carbon::today();
        $futureDate = $today->copy()->addDays($daysInAdvance);

        // Get all active contracts expiring soon
        $contracts = MaintenanceContract::where('status', 'active')
            ->whereBetween('end_date', [$today, $futureDate])
            ->with(['company', 'customer'])
            ->get();

        foreach ($contracts as $contract) {
            // Check if notification already exists for this contract
            $existingNotification = Notification::where('notifiable_type', 'App\\Models\\MaintenanceContract')
                ->where('notifiable_id', $contract->id)
                ->where('type', 'contract_expiring')
                ->first();

            if (!$existingNotification) {
                // Create notification for company admins
                $admins = User::whereHas('roles', function ($query) {
                    $query->where('name', 'Company Admin');
                })->where('company_id', $contract->company_id)->get();

                foreach ($admins as $admin) {
                    $this->createNotification([
                        'company_id' => $contract->company_id,
                        'user_id' => $admin->id,
                        'type' => 'contract_expiring',
                        'notifiable_type' => 'App\\Models\\MaintenanceContract',
                        'notifiable_id' => $contract->id,
                        'title' => 'عقد صيانة على وشك الانتهاء',
                        'message' => 'عقد الصيانة رقم ' . $contract->contract_number . ' للعميل ' . $contract->customer->name . ' سينتهي بتاريخ ' . $contract->end_date->format('Y-m-d'),
                        'data' => [
                            'contract_id' => $contract->id,
                            'customer_id' => $contract->customer_id,
                            'end_date' => $contract->end_date->format('Y-m-d'),
                            'days_remaining' => $today->diffInDays($contract->end_date),
                        ],
                        'priority' => 'normal',
                        'status' => 'pending',
                    ]);
                    $count++;
                }

                // Create notification for sales team
                $salesTeam = User::whereHas('roles', function ($query) {
                    $query->where('name', 'Sales');
                })->where('company_id', $contract->company_id)->get();

                foreach ($salesTeam as $salesPerson) {
                    $this->createNotification([
                        'company_id' => $contract->company_id,
                        'user_id' => $salesPerson->id,
                        'type' => 'contract_expiring',
                        'notifiable_type' => 'App\\Models\\MaintenanceContract',
                        'notifiable_id' => $contract->id,
                        'title' => 'عقد صيانة على وشك الانتهاء',
                        'message' => 'عقد الصيانة رقم ' . $contract->contract_number . ' للعميل ' . $contract->customer->name . ' سينتهي بتاريخ ' . $contract->end_date->format('Y-m-d'),
                        'data' => [
                            'contract_id' => $contract->id,
                            'customer_id' => $contract->customer_id,
                            'end_date' => $contract->end_date->format('Y-m-d'),
                            'days_remaining' => $today->diffInDays($contract->end_date),
                        ],
                        'priority' => 'normal',
                        'status' => 'pending',
                    ]);
                    $count++;
                }
            }
        }

        return $count;
    }

    /**
     * Send all pending notifications.
     *
     * @return int Number of notifications sent
     */
    public function sendPendingNotifications()
    {
        $count = 0;

        // Log the start of the process
        Log::info('بدء إرسال الإشعارات المعلقة');

        // Get all pending notifications
        $pendingNotifications = Notification::where('status', 'pending')->get();

        // Log the number of pending notifications found
        Log::info('تم العثور على إشعارات معلقة', [
            'count' => $pendingNotifications->count()
        ]);

        if ($pendingNotifications->isEmpty()) {
            Log::info('لا توجد إشعارات معلقة للإرسال');
            return 0;
        }

        foreach ($pendingNotifications as $notification) {
            Log::info('محاولة إرسال إشعار', [
                'notification_id' => $notification->id,
                'type' => $notification->type,
                'title' => $notification->title
            ]);

            if ($this->sendNotification($notification)) {
                $count++;
                Log::info('تم إرسال الإشعار بنجاح', [
                    'notification_id' => $notification->id
                ]);
            } else {
                Log::warning('فشل في إرسال الإشعار', [
                    'notification_id' => $notification->id
                ]);
            }
        }

        Log::info('اكتمل إرسال الإشعارات المعلقة', [
            'total' => $pendingNotifications->count(),
            'success' => $count,
            'failed' => $pendingNotifications->count() - $count
        ]);

        return $count;
    }

    /**
     * التحقق من جدولة الصيانة المستحقة اليوم وإنشاء إشعارات فورية.
     *
     * @return int عدد الإشعارات التي تم إنشاؤها
     */
    public function checkTodayMaintenanceSchedules()
    {
        $count = 0;
        $today = Carbon::today();

        // الحصول على جميع جدولات الصيانة المقررة لليوم
        $schedules = MaintenanceSchedule::where('status', 'scheduled')
            ->whereDate('scheduled_date', $today)
            ->with(['elevator', 'elevator.customer', 'elevator.building', 'maintenanceContract', 'company'])
            ->get();

        foreach ($schedules as $schedule) {
            // التحقق من وجود إشعار مسبق لهذه الجدولة
            $existingNotification = Notification::where('notifiable_type', 'App\\Models\\MaintenanceSchedule')
                ->where('notifiable_id', $schedule->id)
                ->where('type', 'maintenance_today')
                ->first();

            if (!$existingNotification) {
                // إنشاء رسالة مفصلة تتضمن معلومات أكثر
                $customerInfo = $schedule->elevator->customer ? $schedule->elevator->customer->name : 'غير محدد';
                $buildingInfo = $schedule->elevator->building ? $schedule->elevator->building->name . ' - ' . $schedule->elevator->building->address : 'غير محدد';
                $contractInfo = $schedule->maintenanceContract ? $schedule->maintenanceContract->contract_number : 'غير محدد';

                $messageDetails = "تفاصيل المصعد: " . $schedule->elevator->model .
                    "\nالرقم التسلسلي: " . $schedule->elevator->serial_number .
                    "\nالعميل: " . $customerInfo .
                    "\nالموقع: " . $buildingInfo .
                    "\nتاريخ الصيانة: اليوم " . $today->format('Y-m-d') .
                    "\nرقم العقد: " . $contractInfo;

                // إنشاء إشعار للفنيين (أولوية عالية)
                $technicians = User::whereHas('roles', function ($query) {
                    $query->where('name', 'Technician');
                })->where('company_id', $schedule->company_id)->get();

                foreach ($technicians as $technician) {
                    $this->createNotification([
                        'company_id' => $schedule->company_id,
                        'user_id' => $technician->id,
                        'type' => 'maintenance_today',
                        'notifiable_type' => 'App\\Models\\MaintenanceSchedule',
                        'notifiable_id' => $schedule->id,
                        'title' => 'صيانة مستحقة اليوم!',
                        'message' => 'تنبيه: هناك صيانة مستحقة اليوم للمصعد ' . $schedule->elevator->model . ' (الرقم التسلسلي: ' . $schedule->elevator->serial_number . ')',
                        'data' => [
                            'schedule_id' => $schedule->id,
                            'elevator_id' => $schedule->elevator_id,
                            'contract_id' => $schedule->maintenance_contract_id,
                            'scheduled_date' => $today->format('Y-m-d'),
                            'details' => $messageDetails,
                            'customer' => $customerInfo,
                            'location' => $buildingInfo,
                            'contract_number' => $contractInfo,
                            'is_today' => true,
                            'requires_immediate_action' => true,
                        ],
                        'priority' => 'high',
                        'status' => 'pending',
                    ]);
                    $count++;
                }

                // إنشاء إشعار للمشرفين ومدراء الصيانة
                $managers = User::whereHas('roles', function ($query) {
                    $query->whereIn('name', ['Maintenance Manager', 'Branch Manager']);
                })->where('company_id', $schedule->company_id)->get();

                foreach ($managers as $manager) {
                    $this->createNotification([
                        'company_id' => $schedule->company_id,
                        'user_id' => $manager->id,
                        'type' => 'maintenance_today',
                        'notifiable_type' => 'App\\Models\\MaintenanceSchedule',
                        'notifiable_id' => $schedule->id,
                        'title' => 'صيانة مستحقة اليوم',
                        'message' => 'هناك صيانة مستحقة اليوم للمصعد ' . $schedule->elevator->model . ' (الرقم التسلسلي: ' . $schedule->elevator->serial_number . ')',
                        'data' => [
                            'schedule_id' => $schedule->id,
                            'elevator_id' => $schedule->elevator_id,
                            'contract_id' => $schedule->maintenance_contract_id,
                            'scheduled_date' => $today->format('Y-m-d'),
                            'details' => $messageDetails,
                            'customer' => $customerInfo,
                            'location' => $buildingInfo,
                            'contract_number' => $contractInfo,
                            'is_today' => true,
                        ],
                        'priority' => 'normal',
                        'status' => 'pending',
                    ]);
                    $count++;
                }
            }
        }

        return $count;
    }

    /**
     * إنشاء إشعار عند اكتمال الصيانة
     *
     * @param  \App\Models\MaintenanceLog  $maintenanceLog
     * @return int عدد الإشعارات التي تم إنشاؤها
     */
    public function createMaintenanceCompletedNotification($maintenanceLog)
    {
        $count = 0;

        // الحصول على المصعد وجدول الصيانة
        $elevator = $maintenanceLog->elevator;
        $schedule = $maintenanceLog->maintenanceSchedule;

        if (!$elevator || !$schedule) {
            return $count;
        }

        // إنشاء إشعار للمسؤولين
        $admins = User::whereHas('roles', function ($query) {
            $query->where('name', 'Company Admin');
        })->where('company_id', $maintenanceLog->company_id)->get();

        foreach ($admins as $admin) {
            $this->createNotification([
                'company_id' => $maintenanceLog->company_id,
                'user_id' => $admin->id,
                'type' => 'maintenance_completed',
                'notifiable_type' => 'App\\Models\\MaintenanceLog',
                'notifiable_id' => $maintenanceLog->id,
                'title' => 'تم إكمال صيانة المصعد',
                'message' => 'تم إكمال صيانة المصعد ' . $elevator->model . ' (الرقم التسلسلي: ' . $elevator->serial_number . ') بتاريخ ' . $maintenanceLog->maintenance_date->format('Y-m-d'),
                'data' => [
                    'log_id' => $maintenanceLog->id,
                    'elevator_id' => $elevator->id,
                    'schedule_id' => $schedule->id,
                    'maintenance_date' => $maintenanceLog->maintenance_date->format('Y-m-d'),
                    'technician_id' => $maintenanceLog->technician_id,
                    'issues_found' => $maintenanceLog->issues_found,
                    'actions_taken' => $maintenanceLog->actions_taken,
                    'parts_replaced' => $maintenanceLog->parts_replaced,
                    'next_maintenance_recommendation' => $maintenanceLog->next_maintenance_recommendation,
                    'action_url' => route('maintenance.logs.show', $maintenanceLog->id),
                    'action_text' => 'عرض تفاصيل الصيانة',
                ],
                'priority' => 'normal',
                'status' => 'pending',
            ]);
            $count++;
        }

        // إشعار للعميل إذا كان لديه حساب في النظام
        if ($elevator->customer && $elevator->customer->user_id) {
            $customer = User::find($elevator->customer->user_id);
            if ($customer) {
                $this->createNotification([
                    'company_id' => $maintenanceLog->company_id,
                    'user_id' => $customer->id,
                    'type' => 'maintenance_completed',
                    'notifiable_type' => 'App\\Models\\MaintenanceLog',
                    'notifiable_id' => $maintenanceLog->id,
                    'title' => 'تم إكمال صيانة المصعد',
                    'message' => 'تم إكمال صيانة المصعد ' . $elevator->model . ' (الرقم التسلسلي: ' . $elevator->serial_number . ') بتاريخ ' . $maintenanceLog->maintenance_date->format('Y-m-d'),
                    'data' => [
                        'log_id' => $maintenanceLog->id,
                        'elevator_id' => $elevator->id,
                        'schedule_id' => $schedule->id,
                        'maintenance_date' => $maintenanceLog->maintenance_date->format('Y-m-d'),
                        'action_url' => route('customer.maintenance.logs.show', $maintenanceLog->id),
                        'action_text' => 'عرض تفاصيل الصيانة',
                    ],
                    'priority' => 'normal',
                    'status' => 'pending',
                ]);
                $count++;
            }
        }

        return $count;
    }

    /**
     * إنشاء إشعار عند وجود مشكلة في الصيانة
     *
     * @param  \App\Models\MaintenanceLog  $maintenanceLog
     * @param  string  $issueDetails
     * @return int عدد الإشعارات التي تم إنشاؤها
     */
    public function createMaintenanceIssueNotification($maintenanceLog, $issueDetails)
    {
        $count = 0;

        // الحصول على المصعد وجدول الصيانة
        $elevator = $maintenanceLog->elevator;
        $schedule = $maintenanceLog->maintenanceSchedule;

        if (!$elevator || !$schedule) {
            return $count;
        }

        // إنشاء إشعار للمسؤولين
        $admins = User::whereHas('roles', function ($query) {
            $query->where('name', 'Company Admin');
        })->where('company_id', $maintenanceLog->company_id)->get();

        foreach ($admins as $admin) {
            $this->createNotification([
                'company_id' => $maintenanceLog->company_id,
                'user_id' => $admin->id,
                'type' => 'maintenance_issue',
                'notifiable_type' => 'App\\Models\\MaintenanceLog',
                'notifiable_id' => $maintenanceLog->id,
                'title' => 'مشكلة في صيانة المصعد',
                'message' => 'تم اكتشاف مشكلة أثناء صيانة المصعد ' . $elevator->model . ' (الرقم التسلسلي: ' . $elevator->serial_number . ') بتاريخ ' . $maintenanceLog->maintenance_date->format('Y-m-d') . "\n\n" . $issueDetails,
                'data' => [
                    'log_id' => $maintenanceLog->id,
                    'elevator_id' => $elevator->id,
                    'schedule_id' => $schedule->id,
                    'maintenance_date' => $maintenanceLog->maintenance_date->format('Y-m-d'),
                    'technician_id' => $maintenanceLog->technician_id,
                    'issues_found' => $maintenanceLog->issues_found,
                    'issue_details' => $issueDetails,
                    'action_url' => route('maintenance.logs.show', $maintenanceLog->id),
                    'action_text' => 'عرض تفاصيل المشكلة',
                ],
                'priority' => 'high',
                'status' => 'pending',
            ]);
            $count++;
        }

        // إشعار للفنيين المسؤولين عن الصيانة
        $technicians = User::whereHas('roles', function ($query) {
            $query->where('name', 'Technician');
        })->where('company_id', $maintenanceLog->company_id)->get();

        foreach ($technicians as $technician) {
            $this->createNotification([
                'company_id' => $maintenanceLog->company_id,
                'user_id' => $technician->id,
                'type' => 'maintenance_issue',
                'notifiable_type' => 'App\\Models\\MaintenanceLog',
                'notifiable_id' => $maintenanceLog->id,
                'title' => 'مشكلة في صيانة المصعد',
                'message' => 'تم اكتشاف مشكلة أثناء صيانة المصعد ' . $elevator->model . ' (الرقم التسلسلي: ' . $elevator->serial_number . ') بتاريخ ' . $maintenanceLog->maintenance_date->format('Y-m-d') . "\n\n" . $issueDetails,
                'data' => [
                    'log_id' => $maintenanceLog->id,
                    'elevator_id' => $elevator->id,
                    'schedule_id' => $schedule->id,
                    'maintenance_date' => $maintenanceLog->maintenance_date->format('Y-m-d'),
                    'technician_id' => $maintenanceLog->technician_id,
                    'issues_found' => $maintenanceLog->issues_found,
                    'issue_details' => $issueDetails,
                    'action_url' => route('maintenance.logs.show', $maintenanceLog->id),
                    'action_text' => 'عرض تفاصيل المشكلة',
                ],
                'priority' => 'high',
                'status' => 'pending',
            ]);
            $count++;
        }

        return $count;
    }

    /**
     * إنشاء إشعار عند اقتراب موعد انتهاء ضمان المصعد
     *
     * @param  int  $daysInAdvance
     * @return int عدد الإشعارات التي تم إنشاؤها
     */
    public function checkWarrantyExpirations($daysInAdvance = 30)
    {
        $count = 0;
        $today = Carbon::today();
        $futureDate = $today->copy()->addDays($daysInAdvance);

        // الحصول على المصاعد التي ستنتهي ضماناتها قريباً
        $elevators = Elevator::whereBetween('warranty_end_date', [$today, $futureDate])
            ->with(['company', 'customer'])
            ->get();

        foreach ($elevators as $elevator) {
            // التحقق من وجود إشعار سابق لهذا المصعد
            $existingNotification = Notification::where('notifiable_type', 'App\\Models\\Elevator')
                ->where('notifiable_id', $elevator->id)
                ->where('type', 'warranty_expiring')
                ->first();

            if (!$existingNotification) {
                // إنشاء إشعار للمسؤولين
                $admins = User::whereHas('roles', function ($query) {
                    $query->where('name', 'Company Admin');
                })->where('company_id', $elevator->company_id)->get();

                foreach ($admins as $admin) {
                    $this->createNotification([
                        'company_id' => $elevator->company_id,
                        'user_id' => $admin->id,
                        'type' => 'warranty_expiring',
                        'notifiable_type' => 'App\\Models\\Elevator',
                        'notifiable_id' => $elevator->id,
                        'title' => 'ضمان المصعد على وشك الانتهاء',
                        'message' => 'ضمان المصعد ' . $elevator->model . ' (الرقم التسلسلي: ' . $elevator->serial_number . ') سينتهي بتاريخ ' . $elevator->warranty_end_date->format('Y-m-d'),
                        'data' => [
                            'elevator_id' => $elevator->id,
                            'customer_id' => $elevator->customer_id,
                            'warranty_end_date' => $elevator->warranty_end_date->format('Y-m-d'),
                            'days_remaining' => $today->diffInDays($elevator->warranty_end_date),
                            'action_url' => route('elevators.show', $elevator->id),
                            'action_text' => 'عرض تفاصيل المصعد',
                        ],
                        'priority' => 'normal',
                        'status' => 'pending',
                    ]);
                    $count++;
                }

                // إشعار لفريق المبيعات
                $salesTeam = User::whereHas('roles', function ($query) {
                    $query->where('name', 'Sales');
                })->where('company_id', $elevator->company_id)->get();

                foreach ($salesTeam as $salesPerson) {
                    $this->createNotification([
                        'company_id' => $elevator->company_id,
                        'user_id' => $salesPerson->id,
                        'type' => 'warranty_expiring',
                        'notifiable_type' => 'App\\Models\\Elevator',
                        'notifiable_id' => $elevator->id,
                        'title' => 'فرصة مبيعات: ضمان المصعد على وشك الانتهاء',
                        'message' => 'ضمان المصعد ' . $elevator->model . ' (الرقم التسلسلي: ' . $elevator->serial_number . ') للعميل ' . ($elevator->customer ? $elevator->customer->name : 'غير محدد') . ' سينتهي بتاريخ ' . $elevator->warranty_end_date->format('Y-m-d') . '. فرصة لعرض عقد صيانة.',
                        'data' => [
                            'elevator_id' => $elevator->id,
                            'customer_id' => $elevator->customer_id,
                            'warranty_end_date' => $elevator->warranty_end_date->format('Y-m-d'),
                            'days_remaining' => $today->diffInDays($elevator->warranty_end_date),
                            'action_url' => route('elevators.show', $elevator->id),
                            'action_text' => 'عرض تفاصيل المصعد',
                        ],
                        'priority' => 'normal',
                        'status' => 'pending',
                    ]);
                    $count++;
                }
            }
        }

        return $count;
    }
}
