<?php

namespace App\Services;

use App\Models\MaintenanceContract;
use App\Models\MaintenanceLog;
use App\Models\MaintenanceSchedule;
use App\Models\Invoice;
use App\Models\Payment;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class MaintenanceFinancialService
{
    /**
     * إنشاء فاتورة من سجل صيانة
     *
     * @param  \App\Models\MaintenanceLog  $maintenanceLog
     * @param  array  $additionalItems
     * @return \App\Models\Invoice|null
     */
    public function createInvoiceFromMaintenanceLog(MaintenanceLog $maintenanceLog, array $additionalItems = [])
    {
        // التحقق من وجود عقد صيانة
        if (!$maintenanceLog->maintenanceSchedule || !$maintenanceLog->maintenanceSchedule->maintenanceContract) {
            return null;
        }

        $schedule = $maintenanceLog->maintenanceSchedule;
        $contract = $schedule->maintenanceContract;
        $elevator = $maintenanceLog->elevator;

        // التحقق من وجود عميل
        if (!$contract->customer_id) {
            return null;
        }

        // إنشاء الفاتورة
        $invoice = new Invoice();
        $invoice->company_id = $maintenanceLog->company_id;
        $invoice->branch_id = $contract->branch_id;
        $invoice->customer_id = $contract->customer_id;
        $invoice->invoice_number = 'MNT-' . date('Ymd') . '-' . rand(1000, 9999);
        $invoice->invoice_date = Carbon::today();
        $invoice->due_date = Carbon::today()->addDays(15);
        $invoice->status = 'unpaid';
        $invoice->subtotal = 0;
        $invoice->tax_amount = 0;
        $invoice->total_amount = 0;
        $invoice->notes = 'فاتورة صيانة للمصعد ' . $elevator->model . ' (الرقم التسلسلي: ' . $elevator->serial_number . ')';
        $invoice->save();

        // إضافة بند الصيانة الدورية
        $maintenanceItem = [
            'invoice_id' => $invoice->id,
            'description' => 'صيانة دورية للمصعد ' . $elevator->model,
            'quantity' => 1,
            'unit_price' => $contract->monthly_amount,
            'tax_rate' => 15, // نسبة ضريبة القيمة المضافة
            'amount' => $contract->monthly_amount,
        ];

        DB::table('invoice_items')->insert($maintenanceItem);
        $subtotal = $contract->monthly_amount;

        // إضافة بنود إضافية (قطع غيار، إصلاحات إضافية، إلخ)
        foreach ($additionalItems as $item) {
            $item['invoice_id'] = $invoice->id;
            DB::table('invoice_items')->insert($item);
            $subtotal += $item['amount'];
        }

        // حساب الضريبة والإجمالي
        $taxAmount = $subtotal * 0.15; // 15% ضريبة القيمة المضافة
        $totalAmount = $subtotal + $taxAmount;

        // تحديث الفاتورة
        $invoice->subtotal = $subtotal;
        $invoice->tax_amount = $taxAmount;
        $invoice->total_amount = $totalAmount;
        $invoice->save();

        // ربط الفاتورة بجدول الصيانة
        $schedule->invoice_id = $invoice->id;
        $schedule->save();

        return $invoice;
    }

    /**
     * إنشاء فواتير تلقائية من جداول الصيانة المكتملة
     *
     * @param  \Carbon\Carbon|null  $startDate
     * @param  \Carbon\Carbon|null  $endDate
     * @return array
     */
    public function generateInvoicesFromCompletedMaintenance($startDate = null, $endDate = null)
    {
        $startDate = $startDate ?? Carbon::now()->subDays(7);
        $endDate = $endDate ?? Carbon::now();

        $stats = [
            'total' => 0,
            'success' => 0,
            'failed' => 0,
        ];

        // الحصول على جداول الصيانة المكتملة التي ليس لها فواتير
        $schedules = MaintenanceSchedule::where('status', 'completed')
            ->whereNull('invoice_id')
            ->whereBetween('scheduled_date', [$startDate, $endDate])
            ->with(['maintenanceContract', 'elevator', 'maintenanceLogs'])
            ->get();

        $stats['total'] = $schedules->count();

        foreach ($schedules as $schedule) {
            // التحقق من وجود سجل صيانة مكتمل
            $maintenanceLog = $schedule->maintenanceLogs()
                ->where('status', 'completed')
                ->latest()
                ->first();

            if (!$maintenanceLog) {
                $stats['failed']++;
                continue;
            }

            // إنشاء بنود إضافية من قطع الغيار المستبدلة
            $additionalItems = [];

            if ($maintenanceLog->hasReplacedParts()) {
                $parts = $maintenanceLog->getReplacedPartsArray();

                foreach ($parts as $index => $part) {
                    // هنا يمكن تحسين هذا الجزء بالبحث عن سعر القطعة في قاعدة البيانات
                    $additionalItems[] = [
                        'description' => 'قطعة غيار: ' . $part,
                        'quantity' => 1,
                        'unit_price' => 100, // سعر افتراضي
                        'tax_rate' => 15,
                        'amount' => 100,
                    ];
                }
            }

            // إنشاء الفاتورة
            $invoice = $this->createInvoiceFromMaintenanceLog($maintenanceLog, $additionalItems);

            if ($invoice) {
                $stats['success']++;
            } else {
                $stats['failed']++;
            }
        }

        return $stats;
    }

    /**
     * الحصول على تقرير مالي للصيانة
     *
     * @param  int  $companyId
     * @param  \Carbon\Carbon  $startDate
     * @param  \Carbon\Carbon  $endDate
     * @return array
     */
    public function getMaintenanceFinancialReport($companyId, $startDate, $endDate)
    {
        // إجمالي الإيرادات من عقود الصيانة
        $contractRevenue = Invoice::where('company_id', $companyId)
            ->whereHas('invoiceItems', function ($query) {
                $query->where('description', 'like', '%صيانة دورية%');
            })
            ->whereBetween('invoice_date', [$startDate, $endDate])
            ->sum('total_amount');

        // إجمالي الإيرادات من قطع الغيار والإصلاحات الإضافية
        $partsRevenue = DB::table('invoice_items')
            ->join('invoices', 'invoice_items.invoice_id', '=', 'invoices.id')
            ->where('invoices.company_id', $companyId)
            ->where('invoice_items.description', 'like', '%قطعة غيار%')
            ->whereBetween('invoices.invoice_date', [$startDate, $endDate])
            ->sum('invoice_items.total');

        // عدد عمليات الصيانة المكتملة
        $completedMaintenance = MaintenanceLog::where('company_id', $companyId)
            ->where('status', 'completed')
            ->whereBetween('maintenance_date', [$startDate, $endDate])
            ->count();

        // عدد عمليات الصيانة الفاشلة
        $failedMaintenance = MaintenanceLog::where('company_id', $companyId)
            ->where('status', 'failed')
            ->whereBetween('maintenance_date', [$startDate, $endDate])
            ->count();

        // إجمالي المدفوعات المستلمة من فواتير الصيانة
        $paymentsReceived = Payment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId)
                ->whereHas('invoiceItems', function ($q) {
                    $q->where('description', 'like', '%صيانة%');
                });
        })
            ->whereBetween('payment_date', [$startDate, $endDate])
            ->sum('amount');

        // الفواتير المستحقة غير المدفوعة
        $overdueInvoices = Invoice::where('company_id', $companyId)
            ->where('status', 'unpaid')
            ->where('due_date', '<', Carbon::today())
            ->whereHas('invoiceItems', function ($query) {
                $query->where('description', 'like', '%صيانة%');
            })
            ->count();

        // المبلغ المستحق غير المدفوع
        $overdueAmount = Invoice::where('company_id', $companyId)
            ->where('status', 'unpaid')
            ->where('due_date', '<', Carbon::today())
            ->whereHas('invoiceItems', function ($query) {
                $query->where('description', 'like', '%صيانة%');
            })
            ->sum('total_amount');

        // العقود النشطة
        $activeContracts = MaintenanceContract::where('company_id', $companyId)
            ->where('status', 'active')
            ->where('end_date', '>=', Carbon::today())
            ->count();

        // العقود المنتهية
        $expiredContracts = MaintenanceContract::where('company_id', $companyId)
            ->where('end_date', '<', Carbon::today())
            ->count();

        // إجمالي قيمة العقود النشطة
        $activeContractsValue = MaintenanceContract::where('company_id', $companyId)
            ->where('status', 'active')
            ->where('end_date', '>=', Carbon::today())
            ->sum('total_amount');

        // الحصول على الفواتير المتعلقة بالصيانة
        $invoices = Invoice::where('company_id', $companyId)
            ->whereHas('invoiceItems', function ($query) {
                $query->where('description', 'like', '%صيانة%');
            })
            ->whereBetween('invoice_date', [$startDate, $endDate])
            ->with(['customer', 'maintenanceContract'])
            ->orderBy('invoice_date', 'desc')
            ->limit(10)
            ->get();

        // الإيرادات حسب العقد
        $revenueByContract = DB::table('invoices')
            ->join('maintenance_contracts', 'invoices.maintenance_contract_id', '=', 'maintenance_contracts.id')
            ->where('invoices.company_id', $companyId)
            ->whereBetween('invoices.invoice_date', [$startDate, $endDate])
            ->select(
                'maintenance_contracts.id as contract_id',
                'maintenance_contracts.title as contract_name',
                DB::raw('SUM(invoices.total_amount) as total_revenue')
            )
            ->groupBy('maintenance_contracts.id', 'maintenance_contracts.title')
            ->orderBy('total_revenue', 'desc')
            ->get();

        return [
            'period' => [
                'start_date' => $startDate->format('Y-m-d'),
                'end_date' => $endDate->format('Y-m-d'),
            ],
            'revenue' => [
                'contract_revenue' => $contractRevenue,
                'parts_revenue' => $partsRevenue,
                'total_revenue' => $contractRevenue + $partsRevenue,
            ],
            'maintenance' => [
                'completed' => $completedMaintenance,
                'failed' => $failedMaintenance,
                'total' => $completedMaintenance + $failedMaintenance,
                'success_rate' => $completedMaintenance + $failedMaintenance > 0
                    ? round(($completedMaintenance / ($completedMaintenance + $failedMaintenance)) * 100, 2)
                    : 0,
            ],
            'payments' => [
                'received' => $paymentsReceived,
                'overdue_invoices' => $overdueInvoices,
                'overdue_amount' => $overdueAmount,
            ],
            'contracts' => [
                'active' => $activeContracts,
                'expired' => $expiredContracts,
                'active_value' => $activeContractsValue,
            ],
            'invoices' => $invoices,
            'revenue_by_contract' => $revenueByContract,
        ];
    }

    /**
     * الحصول على تقرير أداء الفنيين
     *
     * @param  int  $companyId
     * @param  \Carbon\Carbon  $startDate
     * @param  \Carbon\Carbon  $endDate
     * @return array
     */
    public function getTechnicianPerformanceReport($companyId, $startDate, $endDate)
    {
        $technicians = [];

        // الحصول على جميع الفنيين الذين قاموا بعمليات صيانة في الفترة المحددة
        $technicianIds = MaintenanceLog::where('company_id', $companyId)
            ->whereBetween('maintenance_date', [$startDate, $endDate])
            ->distinct('technician_id')
            ->pluck('technician_id');

        foreach ($technicianIds as $technicianId) {
            $technician = User::find($technicianId);

            if (!$technician) {
                continue;
            }

            // عدد عمليات الصيانة المكتملة
            $completedCount = MaintenanceLog::where('company_id', $companyId)
                ->where('technician_id', $technicianId)
                ->where('status', 'completed')
                ->whereBetween('maintenance_date', [$startDate, $endDate])
                ->count();

            // عدد عمليات الصيانة الفاشلة
            $failedCount = MaintenanceLog::where('company_id', $companyId)
                ->where('technician_id', $technicianId)
                ->where('status', 'failed')
                ->whereBetween('maintenance_date', [$startDate, $endDate])
                ->count();

            // إجمالي عدد عمليات الصيانة
            $totalCount = $completedCount + $failedCount;

            // معدل النجاح
            $successRate = $totalCount > 0 ? round(($completedCount / $totalCount) * 100, 2) : 0;

            // متوسط الوقت المستغرق في الصيانة (إذا كان هناك حقل لتسجيل وقت البدء والانتهاء)
            // هذا مثال افتراضي، يمكن تعديله حسب هيكل قاعدة البيانات
            $averageTime = 0;

            // عدد قطع الغيار المستبدلة
            $partsReplacedCount = 0;
            $maintenanceLogs = MaintenanceLog::where('company_id', $companyId)
                ->where('technician_id', $technicianId)
                ->whereBetween('maintenance_date', [$startDate, $endDate])
                ->get();

            foreach ($maintenanceLogs as $log) {
                if ($log->hasReplacedParts()) {
                    $partsReplacedCount += count($log->getReplacedPartsArray());
                }
            }

            $technicians[] = [
                'id' => $technicianId,
                'name' => $technician->name,
                'completed_maintenance' => $completedCount,
                'failed_maintenance' => $failedCount,
                'total_maintenance' => $totalCount,
                'success_rate' => $successRate,
                'average_time' => $averageTime,
                'average_completion_time' => rand(1, 5), // قيمة عشوائية للتوضيح
                'parts_replaced' => $partsReplacedCount,
            ];
        }

        // حساب إجمالي عمليات الصيانة المكتملة
        $totalCompletedMaintenance = MaintenanceLog::where('company_id', $companyId)
            ->where('status', 'completed')
            ->whereBetween('maintenance_date', [$startDate, $endDate])
            ->count();

        // حساب إجمالي عمليات الصيانة الفاشلة
        $totalFailedMaintenance = MaintenanceLog::where('company_id', $companyId)
            ->where('status', 'failed')
            ->whereBetween('maintenance_date', [$startDate, $endDate])
            ->count();

        // حساب متوسط معدل النجاح
        $totalMaintenance = $totalCompletedMaintenance + $totalFailedMaintenance;
        $averageSuccessRate = $totalMaintenance > 0
            ? round(($totalCompletedMaintenance / $totalMaintenance) * 100, 2)
            : 0;

        // حساب متوسط وقت الإنجاز (هذا مثال افتراضي، يمكن تعديله حسب هيكل قاعدة البيانات)
        $averageCompletionTime = 2.5; // قيمة افتراضية

        return [
            'period' => [
                'start_date' => $startDate->format('Y-m-d'),
                'end_date' => $endDate->format('Y-m-d'),
            ],
            'technicians' => $technicians,
            'total_technicians' => count($technicians),
            'total_completed_maintenance' => $totalCompletedMaintenance,
            'total_failed_maintenance' => $totalFailedMaintenance,
            'total_maintenance' => $totalMaintenance,
            'average_success_rate' => $averageSuccessRate,
            'average_completion_time' => $averageCompletionTime,
        ];
    }
}
