<?php

namespace App\Services;

use App\Models\JournalEntry;
use App\Models\JournalEntryItem;
use App\Models\ChartOfAccount;
use App\Models\Invoice;
use App\Models\Customer;
use App\Models\FiscalYear;
use App\Models\AccountingPeriod;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class JournalEntryService
{
    /**
     * إنشاء قيد محاسبي للفاتورة
     *
     * @param Invoice $invoice الفاتورة
     * @return JournalEntry|null القيد المحاسبي أو null في حالة الفشل
     */
    public function createInvoiceJournalEntry(Invoice $invoice)
    {
        try {
            DB::beginTransaction();

            // البحث عن السنة المالية والفترة المحاسبية النشطة
            $fiscalYear = FiscalYear::where('company_id', $invoice->company_id)
                ->where('is_active', true)
                ->where('is_closed', false)
                ->first();

            if (!$fiscalYear) {
                throw new \Exception('لا توجد سنة مالية نشطة');
            }

            $entryDate = $invoice->invoice_date;

            $accountingPeriod = AccountingPeriod::where('fiscal_year_id', $fiscalYear->id)
                ->where('is_closed', false)
                ->where('start_date', '<=', $entryDate)
                ->where('end_date', '>=', $entryDate)
                ->first();

            if (!$accountingPeriod) {
                throw new \Exception('لا توجد فترة محاسبية نشطة للتاريخ المحدد');
            }

            // إنشاء القيد المحاسبي
            $entry = new JournalEntry();
            $entry->company_id = $invoice->company_id;
            $entry->fiscal_year_id = $fiscalYear->id;
            $entry->accounting_period_id = $accountingPeriod->id;
            $entry->entry_number = JournalEntry::generateEntryNumber($invoice->company_id);
            $entry->entry_date = $entryDate;
            $entry->reference_type = 'App\Models\Invoice';
            $entry->reference_id = $invoice->id;
            $entry->description = 'قيد فاتورة رقم: ' . $invoice->invoice_number;
            $entry->is_posted = false;
            $entry->is_approved = false;
            $entry->created_by = $invoice->created_by;
            $entry->save();

            // إنشاء بنود القيد المحاسبي
            $this->createInvoiceJournalEntryItems($entry, $invoice);

            // ترحيل القيد المحاسبي
            $entry->post();

            // إنشاء كشف حساب للعميل من القيد المحاسبي
            \App\Models\CustomerStatement::createForJournalEntry($entry);

            DB::commit();

            return $entry;
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('فشل في إنشاء قيد محاسبي للفاتورة: ' . $e->getMessage(), [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return null;
        }
    }

    /**
     * إنشاء بنود القيد المحاسبي للفاتورة
     *
     * @param JournalEntry $entry القيد المحاسبي
     * @param Invoice $invoice الفاتورة
     * @return void
     */
    private function createInvoiceJournalEntryItems(JournalEntry $entry, Invoice $invoice)
    {
        // 1. حساب المدينين (ذمم العملاء)
        $accountReceivable = $this->getOrCreateAccountReceivable($invoice->company_id, $invoice->customer);

        // 2. حساب الإيرادات
        $revenueAccount = $this->getOrCreateRevenueAccount($invoice->company_id);

        // 3. حساب ضريبة القيمة المضافة
        $vatAccount = $this->getOrCreateVatAccount($invoice->company_id);

        // إنشاء بند مدين (ذمم العملاء)
        $debitItem = new JournalEntryItem();
        $debitItem->journal_entry_id = $entry->id;
        $debitItem->account_id = $accountReceivable->id;
        $debitItem->description = 'فاتورة رقم: ' . $invoice->invoice_number . ' - ' . $invoice->customer->name;
        $debitItem->debit = $invoice->total_amount;
        $debitItem->credit = 0;
        $debitItem->save();

        // إنشاء بند دائن (الإيرادات)
        $creditItem = new JournalEntryItem();
        $creditItem->journal_entry_id = $entry->id;
        $creditItem->account_id = $revenueAccount->id;
        $creditItem->description = 'فاتورة رقم: ' . $invoice->invoice_number . ' - إيرادات';
        $creditItem->debit = 0;
        $creditItem->credit = $invoice->subtotal - $invoice->discount_amount;
        $creditItem->save();

        // إنشاء بند دائن (ضريبة القيمة المضافة)
        if ($invoice->tax_amount > 0) {
            $vatItem = new JournalEntryItem();
            $vatItem->journal_entry_id = $entry->id;
            $vatItem->account_id = $vatAccount->id;
            $vatItem->description = 'فاتورة رقم: ' . $invoice->invoice_number . ' - ضريبة القيمة المضافة';
            $vatItem->debit = 0;
            $vatItem->credit = $invoice->tax_amount;
            $vatItem->save();
        }
    }

    /**
     * الحصول على أو إنشاء حساب المدينين (ذمم العملاء)
     *
     * @param int $companyId معرف الشركة
     * @param Customer $customer العميل
     * @return ChartOfAccount حساب المدينين
     */
    private function getOrCreateAccountReceivable($companyId, Customer $customer)
    {
        // البحث عن حساب الذمم المدينة الرئيسي (1200)
        $mainAccountReceivable = ChartOfAccount::where('company_id', $companyId)
            ->where('account_code', '1200')
            ->first();

        // إذا لم يوجد حساب الذمم المدينة (1200)، نستخدم خدمة حسابات العملاء لإنشائه
        if (!$mainAccountReceivable) {
            $customerAccountService = new \App\Services\CustomerAccountService();

            // البحث عن حساب الأصول المتداولة
            $currentAssetsAccount = ChartOfAccount::where('account_code', '1100')
                ->where('company_id', $companyId)
                ->first();

            if (!$currentAssetsAccount) {
                // إذا لم يوجد حساب الأصول المتداولة، نبحث عن حساب الأصول
                $assetsAccount = ChartOfAccount::where('account_code', '1000')
                    ->where('company_id', $companyId)
                    ->first();

                if (!$assetsAccount) {
                    // إنشاء هيكل حسابات الأصول الأساسي بالكامل
                    $assetsAccount = $customerAccountService->createAssetsAccountStructure($companyId);
                    $currentAssetsAccount = $customerAccountService->createCurrentAssetsAccount($assetsAccount);
                } else {
                    // إنشاء حساب الأصول المتداولة تحت حساب الأصول
                    $currentAssetsAccount = $customerAccountService->createCurrentAssetsAccount($assetsAccount);
                }
            }

            // إنشاء حساب الذمم المدينة (1200) تحت حساب الأصول المتداولة
            $mainAccountReceivable = $customerAccountService->createReceivablesAccount($currentAssetsAccount);
        }

        // البحث عن حساب العميل تحت حساب الذمم المدينة (1200)
        $customerAccount = ChartOfAccount::where('company_id', $companyId)
            ->where('parent_id', $mainAccountReceivable->id)
            ->where('name', 'like', '%' . $customer->name . '%')
            ->first();

        // إنشاء حساب للعميل إذا لم يكن موجوداً
        if (!$customerAccount) {
            // توليد رمز حساب فريد
            $lastAccountCode = ChartOfAccount::where('parent_id', $mainAccountReceivable->id)
                ->orderBy('account_code', 'desc')
                ->value('account_code');

            $newAccountCode = $lastAccountCode ? (intval($lastAccountCode) + 1) : ($mainAccountReceivable->account_code . '01');

            $customerAccount = new ChartOfAccount();
            $customerAccount->company_id = $companyId;
            $customerAccount->account_code = $newAccountCode;
            $customerAccount->name = 'حساب العميل: ' . $customer->name;
            $customerAccount->name_en = 'Customer Account: ' . $customer->name;
            $customerAccount->account_type = 'asset';
            $customerAccount->sub_type = 'customer_receivable';
            $customerAccount->parent_id = $mainAccountReceivable->id;
            $customerAccount->description = 'حساب ذمم العميل: ' . $customer->name;
            $customerAccount->is_active = true;
            $customerAccount->is_system = false;
            $customerAccount->opening_balance = 0;
            $customerAccount->current_balance = 0;
            $customerAccount->currency = 'SAR';
            $customerAccount->balance_type = 'debit';
            $customerAccount->is_cash_account = false;
            $customerAccount->is_bank_account = false;
            $customerAccount->level = $mainAccountReceivable->level + 1;
            $customerAccount->is_parent = false;
            $customerAccount->save();
        }

        return $customerAccount;
    }

    /**
     * إنشاء حساب المدينين الرئيسي
     *
     * @param int $companyId معرف الشركة
     * @return ChartOfAccount حساب المدينين الرئيسي
     */
    private function createMainAccountsReceivable($companyId)
    {
        // البحث عن حساب الأصول الرئيسي
        $assetsAccount = ChartOfAccount::where('company_id', $companyId)
            ->where('account_type', 'asset')
            ->where('parent_id', null)
            ->first();

        // إنشاء حساب الأصول الرئيسي إذا لم يكن موجوداً
        if (!$assetsAccount) {
            $assetsAccount = new ChartOfAccount();
            $assetsAccount->company_id = $companyId;
            $assetsAccount->account_code = 'ASSETS';
            $assetsAccount->name = 'الأصول';
            $assetsAccount->name_en = 'Assets';
            $assetsAccount->account_type = 'asset';
            $assetsAccount->sub_type = 'current_assets';
            $assetsAccount->parent_id = null;
            $assetsAccount->description = 'حساب الأصول الرئيسي';
            $assetsAccount->is_active = true;
            $assetsAccount->is_system = true;
            $assetsAccount->opening_balance = 0;
            $assetsAccount->current_balance = 0;
            $assetsAccount->currency = 'SAR';
            $assetsAccount->balance_type = 'debit';
            $assetsAccount->is_cash_account = false;
            $assetsAccount->is_bank_account = false;
            $assetsAccount->level = 1;
            $assetsAccount->is_parent = true;
            $assetsAccount->save();
        }

        // إنشاء حساب المدينين الرئيسي
        $accountReceivable = new ChartOfAccount();
        $accountReceivable->company_id = $companyId;
        $accountReceivable->account_code = 'AR';
        $accountReceivable->name = 'ذمم العملاء';
        $accountReceivable->name_en = 'Accounts Receivable';
        $accountReceivable->account_type = 'asset';
        $accountReceivable->sub_type = 'accounts_receivable';
        $accountReceivable->parent_id = $assetsAccount->id;
        $accountReceivable->description = 'حساب ذمم العملاء الرئيسي';
        $accountReceivable->is_active = true;
        $accountReceivable->is_system = true;
        $accountReceivable->opening_balance = 0;
        $accountReceivable->current_balance = 0;
        $accountReceivable->currency = 'SAR';
        $accountReceivable->balance_type = 'debit';
        $accountReceivable->is_cash_account = false;
        $accountReceivable->is_bank_account = false;
        $accountReceivable->level = $assetsAccount->level + 1;
        $accountReceivable->is_parent = true;
        $accountReceivable->save();

        return $accountReceivable;
    }

    /**
     * الحصول على أو إنشاء حساب الإيرادات
     *
     * @param int $companyId معرف الشركة
     * @return ChartOfAccount حساب الإيرادات
     */
    private function getOrCreateRevenueAccount($companyId)
    {
        // البحث عن حساب الإيرادات
        $revenueAccount = ChartOfAccount::where('company_id', $companyId)
            ->where('account_type', 'revenue')
            ->where('sub_type', 'sales_revenue')
            ->first();

        // إنشاء حساب الإيرادات إذا لم يكن موجوداً
        if (!$revenueAccount) {
            // البحث عن حساب الإيرادات الرئيسي
            $mainRevenueAccount = ChartOfAccount::where('company_id', $companyId)
                ->where('account_type', 'revenue')
                ->where('parent_id', null)
                ->first();

            // إنشاء حساب الإيرادات الرئيسي إذا لم يكن موجوداً
            if (!$mainRevenueAccount) {
                $mainRevenueAccount = new ChartOfAccount();
                $mainRevenueAccount->company_id = $companyId;
                $mainRevenueAccount->account_code = 'REVENUE';
                $mainRevenueAccount->name = 'الإيرادات';
                $mainRevenueAccount->name_en = 'Revenue';
                $mainRevenueAccount->account_type = 'revenue';
                $mainRevenueAccount->sub_type = 'operating_revenue';
                $mainRevenueAccount->parent_id = null;
                $mainRevenueAccount->description = 'حساب الإيرادات الرئيسي';
                $mainRevenueAccount->is_active = true;
                $mainRevenueAccount->is_system = true;
                $mainRevenueAccount->opening_balance = 0;
                $mainRevenueAccount->current_balance = 0;
                $mainRevenueAccount->currency = 'SAR';
                $mainRevenueAccount->balance_type = 'credit';
                $mainRevenueAccount->is_cash_account = false;
                $mainRevenueAccount->is_bank_account = false;
                $mainRevenueAccount->level = 1;
                $mainRevenueAccount->is_parent = true;
                $mainRevenueAccount->save();
            }

            // إنشاء حساب إيرادات المبيعات
            $revenueAccount = new ChartOfAccount();
            $revenueAccount->company_id = $companyId;
            $revenueAccount->account_code = 'SALES';
            $revenueAccount->name = 'إيرادات المبيعات';
            $revenueAccount->name_en = 'Sales Revenue';
            $revenueAccount->account_type = 'revenue';
            $revenueAccount->sub_type = 'sales_revenue';
            $revenueAccount->parent_id = $mainRevenueAccount->id;
            $revenueAccount->description = 'حساب إيرادات المبيعات';
            $revenueAccount->is_active = true;
            $revenueAccount->is_system = true;
            $revenueAccount->opening_balance = 0;
            $revenueAccount->current_balance = 0;
            $revenueAccount->currency = 'SAR';
            $revenueAccount->balance_type = 'credit';
            $revenueAccount->is_cash_account = false;
            $revenueAccount->is_bank_account = false;
            $revenueAccount->level = $mainRevenueAccount->level + 1;
            $revenueAccount->is_parent = false;
            $revenueAccount->save();
        }

        return $revenueAccount;
    }

    /**
     * الحصول على أو إنشاء حساب ضريبة القيمة المضافة
     *
     * @param int $companyId معرف الشركة
     * @return ChartOfAccount حساب ضريبة القيمة المضافة
     */
    private function getOrCreateVatAccount($companyId)
    {
        // البحث عن حساب ضريبة القيمة المضافة
        $vatAccount = ChartOfAccount::where('company_id', $companyId)
            ->where('account_type', 'liability')
            ->where('sub_type', 'vat')
            ->first();

        // إنشاء حساب ضريبة القيمة المضافة إذا لم يكن موجوداً
        if (!$vatAccount) {
            // البحث عن حساب الخصوم الرئيسي
            $liabilitiesAccount = ChartOfAccount::where('company_id', $companyId)
                ->where('account_type', 'liability')
                ->where('parent_id', null)
                ->first();

            // إنشاء حساب الخصوم الرئيسي إذا لم يكن موجوداً
            if (!$liabilitiesAccount) {
                $liabilitiesAccount = new ChartOfAccount();
                $liabilitiesAccount->company_id = $companyId;
                $liabilitiesAccount->account_code = 'LIABILITIES';
                $liabilitiesAccount->name = 'الخصوم';
                $liabilitiesAccount->name_en = 'Liabilities';
                $liabilitiesAccount->account_type = 'liability';
                $liabilitiesAccount->sub_type = 'current_liabilities';
                $liabilitiesAccount->parent_id = null;
                $liabilitiesAccount->description = 'حساب الخصوم الرئيسي';
                $liabilitiesAccount->is_active = true;
                $liabilitiesAccount->is_system = true;
                $liabilitiesAccount->opening_balance = 0;
                $liabilitiesAccount->current_balance = 0;
                $liabilitiesAccount->currency = 'SAR';
                $liabilitiesAccount->balance_type = 'credit';
                $liabilitiesAccount->is_cash_account = false;
                $liabilitiesAccount->is_bank_account = false;
                $liabilitiesAccount->level = 1;
                $liabilitiesAccount->is_parent = true;
                $liabilitiesAccount->save();
            }

            // إنشاء حساب ضريبة القيمة المضافة
            $vatAccount = new ChartOfAccount();
            $vatAccount->company_id = $companyId;
            $vatAccount->account_code = 'VAT';
            $vatAccount->name = 'ضريبة القيمة المضافة';
            $vatAccount->name_en = 'VAT';
            $vatAccount->account_type = 'liability';
            $vatAccount->sub_type = 'vat';
            $vatAccount->parent_id = $liabilitiesAccount->id;
            $vatAccount->description = 'حساب ضريبة القيمة المضافة';
            $vatAccount->is_active = true;
            $vatAccount->is_system = true;
            $vatAccount->opening_balance = 0;
            $vatAccount->current_balance = 0;
            $vatAccount->currency = 'SAR';
            $vatAccount->balance_type = 'credit';
            $vatAccount->is_cash_account = false;
            $vatAccount->is_bank_account = false;
            $vatAccount->level = $liabilitiesAccount->level + 1;
            $vatAccount->is_parent = false;
            $vatAccount->save();
        }

        return $vatAccount;
    }

    /**
     * تحديث القيد المحاسبي للفاتورة
     *
     * @param Invoice $invoice الفاتورة
     * @return JournalEntry|null القيد المحاسبي أو null في حالة الفشل
     */
    public function updateInvoiceJournalEntry(Invoice $invoice)
    {
        try {
            DB::beginTransaction();

            // البحث عن القيد المحاسبي الحالي للفاتورة
            $entry = JournalEntry::where('reference_type', 'App\Models\Invoice')
                ->where('reference_id', $invoice->id)
                ->first();

            // إذا لم يكن هناك قيد محاسبي، نقوم بإنشاء واحد جديد
            if (!$entry) {
                return $this->createInvoiceJournalEntry($invoice);
            }

            // التحقق من أن القيد غير معتمد
            if ($entry->is_approved) {
                throw new \Exception('لا يمكن تحديث القيد المحاسبي لأنه معتمد');
            }

            // إلغاء ترحيل القيد المحاسبي
            if ($entry->is_posted) {
                $entry->unpost();
            }

            // حذف بنود القيد المحاسبي الحالية
            $entry->items()->delete();

            // تحديث تاريخ القيد
            $entry->entry_date = $invoice->invoice_date;
            $entry->description = 'قيد فاتورة رقم: ' . $invoice->invoice_number;
            $entry->save();

            // إنشاء بنود القيد المحاسبي الجديدة
            $this->createInvoiceJournalEntryItems($entry, $invoice);

            // ترحيل القيد المحاسبي
            $entry->post();

            // إنشاء كشف حساب للعميل من القيد المحاسبي
            \App\Models\CustomerStatement::createForJournalEntry($entry);

            DB::commit();

            return $entry;
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('فشل في تحديث القيد المحاسبي للفاتورة: ' . $e->getMessage(), [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return null;
        }
    }
}
