<?php

namespace App\Services;

use App\Models\Invoice;
use App\Models\InvoiceTemplate;
use Illuminate\Support\Facades\Storage;
use Barryvdh\DomPDF\Facade\Pdf;

class InvoiceTemplateService
{
    /**
     * Generate an invoice HTML based on template
     *
     * @param Invoice $invoice
     * @param InvoiceTemplate|null $template
     * @return string
     */
    public function generateInvoiceHtml(Invoice $invoice, InvoiceTemplate $template = null)
    {
        // If no template provided, get the default template for the invoice language
        if (!$template) {
            $template = InvoiceTemplate::where('company_id', $invoice->company_id)
                ->where('language', $invoice->language)
                ->where('is_default', true)
                ->where('is_active', true)
                ->first();

            // If still no template, get any active template for the language
            if (!$template) {
                $template = InvoiceTemplate::where('company_id', $invoice->company_id)
                    ->where('language', $invoice->language)
                    ->where('is_active', true)
                    ->first();
            }

            // If still no template, create a basic one
            if (!$template) {
                $template = $this->createBasicTemplate($invoice->company_id, $invoice->language);
            }
        }

        // Get invoice data
        $data = $this->prepareInvoiceData($invoice);

        // Generate HTML
        return $this->renderTemplate($template, $data);
    }

    /**
     * Generate a PDF for an invoice
     *
     * @param Invoice $invoice
     * @param InvoiceTemplate|null $template
     * @return \Barryvdh\DomPDF\PDF
     */
    public function generateInvoicePdf(Invoice $invoice, InvoiceTemplate $template = null)
    {
        $html = $this->generateInvoiceHtml($invoice, $template);

        $isRtl = $invoice->language === 'ar';
        $options = [
            'isRemoteEnabled' => true,
            'isHtml5ParserEnabled' => true,
            'defaultFont' => $isRtl ? 'XB Riyaz' : 'DejaVu Sans',
        ];

        $pdf = PDF::loadHTML($html)->setOptions($options);

        if ($isRtl) {
            $pdf->setOption('isPhpEnabled', true);
            $pdf->getDomPDF()->getOptions()->set('isPhpEnabled', true);
        }

        return $pdf;
    }

    /**
     * Save invoice PDF to storage
     *
     * @param Invoice $invoice
     * @param InvoiceTemplate|null $template
     * @return string File path
     */
    public function saveInvoicePdf(Invoice $invoice, InvoiceTemplate $template = null)
    {
        $pdf = $this->generateInvoicePdf($invoice, $template);

        $fileName = 'invoice_' . $invoice->invoice_number . '.pdf';
        $path = 'invoices/' . $invoice->company_id . '/' . $fileName;

        Storage::disk('public')->put($path, $pdf->output());

        return $path;
    }

    /**
     * Create a basic template if none exists
     *
     * @param int $companyId
     * @param string $language
     * @return InvoiceTemplate
     */
    protected function createBasicTemplate($companyId, $language)
    {
        $isRtl = $language === 'ar';
        $direction = $isRtl ? 'rtl' : 'ltr';
        $align = $isRtl ? 'right' : 'left';
        $alignOpposite = $isRtl ? 'left' : 'right';

        $headerHtml = '<div class="header" style="text-align: center; margin-bottom: 20px;">
            <h1 style="color: #333;">فاتورة</h1>
        </div>';

        $footerHtml = '<div class="footer" style="text-align: center; margin-top: 20px; font-size: 12px; color: #777;">
            <p>شكراً لتعاملكم معنا</p>
        </div>';

        $cssStyles = 'body {
            font-family: ' . ($isRtl ? '"XB Riyaz", Arial' : '"DejaVu Sans", Arial') . ';
            direction: ' . $direction . ';
            text-align: ' . $align . ';
            padding: 20px;
        }
        .header {
            border-bottom: 1px solid #ddd;
            padding-bottom: 10px;
        }
        .footer {
            border-top: 1px solid #ddd;
            padding-top: 10px;
        }
        .invoice-info {
            display: flex;
            justify-content: space-between;
            margin: 20px 0;
        }
        .company-info, .customer-info, .invoice-details {
            width: 30%;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
        }
        th, td {
            padding: 10px;
            border: 1px solid #ddd;
            text-align: ' . $align . ';
        }
        th {
            background-color: #f5f5f5;
        }
        .text-' . $alignOpposite . ' {
            text-align: ' . $alignOpposite . ';
        }
        .invoice-total {
            width: 50%;
            margin-' . $alignOpposite . ': auto;
        }
        .invoice-notes, .invoice-terms {
            margin-top: 20px;
        }';

        return InvoiceTemplate::create([
            'company_id' => $companyId,
            'name' => $language === 'ar' ? 'قالب أساسي (عربي)' : 'Basic Template (English)',
            'language' => $language,
            'header_html' => $headerHtml,
            'footer_html' => $footerHtml,
            'css_styles' => $cssStyles,
            'is_default' => true,
            'is_active' => true,
        ]);
    }

    /**
     * Prepare invoice data for template
     *
     * @param Invoice $invoice
     * @return array
     */
    protected function prepareInvoiceData(Invoice $invoice)
    {
        $company = $invoice->company;
        $customer = $invoice->customer;
        $items = $invoice->items;

        // Format status in Arabic or English
        $statusLabels = [
            'ar' => [
                'draft' => 'مسودة',
                'sent' => 'مرسلة',
                'paid' => 'مدفوعة',
                'partially_paid' => 'مدفوعة جزئياً',
                'overdue' => 'متأخرة',
                'cancelled' => 'ملغاة',
            ],
            'en' => [
                'draft' => 'Draft',
                'sent' => 'Sent',
                'paid' => 'Paid',
                'partially_paid' => 'Partially Paid',
                'overdue' => 'Overdue',
                'cancelled' => 'Cancelled',
            ],
        ];

        $status = $statusLabels[$invoice->language][$invoice->status] ?? $invoice->status;

        return [
            'company' => [
                'name' => $company->name,
                'address' => $company->address,
                'phone' => $company->phone,
                'email' => $company->email,
                'vat_number' => $company->vat_number,
                'logo' => $company->logo ? asset('storage/' . $company->logo) : null,
            ],
            'customer' => [
                'name' => $customer->name,
                'address' => $customer->address,
                'phone' => $customer->phone,
                'email' => $customer->email,
                'vat_number' => $customer->vat_number,
            ],
            'invoice' => [
                'invoice_number' => $invoice->invoice_number,
                'invoice_date' => $invoice->invoice_date->format('Y-m-d'),
                'due_date' => $invoice->due_date->format('Y-m-d'),
                'status' => $status,
                'subtotal' => $invoice->subtotal,
                'tax_rate' => $invoice->tax_rate,
                'tax_amount' => $invoice->tax_amount,
                'discount_percentage' => $invoice->discount_percentage,
                'discount_amount' => $invoice->discount_amount,
                'total_amount' => $invoice->total_amount,
                'paid_amount' => $invoice->paid_amount,
                'due_amount' => $invoice->due_amount,
                'notes' => $invoice->notes,
                'terms_conditions' => $invoice->terms_conditions,
                'language' => $invoice->language,
                'is_recurring' => $invoice->is_recurring,
                'recurring_cycle' => $invoice->recurring_cycle,
                'zatca_qr_code' => $invoice->zatca_qr_code,
            ],
            'items' => $items->map(function ($item) {
                return [
                    'description' => $item->description,
                    'quantity' => $item->quantity,
                    'unit_price' => $item->unit_price,
                    'discount_percentage' => $item->discount_percentage,
                    'discount_amount' => $item->discount_amount,
                    'tax_percentage' => $item->tax_percentage,
                    'tax_amount' => $item->tax_amount,
                    'subtotal' => $item->subtotal,
                    'total' => $item->total,
                ];
            })->toArray(),
            'payments' => $invoice->payments->map(function ($payment) {
                return [
                    'payment_date' => $payment->payment_date->format('Y-m-d'),
                    'amount' => $payment->amount,
                    'payment_method' => $payment->paymentMethod->name,
                    'reference' => $payment->reference,
                ];
            })->toArray(),
        ];
    }

    /**
     * Render template with data
     *
     * @param InvoiceTemplate $template
     * @param array $data
     * @return string
     */
    protected function renderTemplate(InvoiceTemplate $template, array $data)
    {
        $headerHtml = $this->replaceVariables($template->header_html, $data);
        $footerHtml = $this->replaceVariables($template->footer_html, $data);
        $cssStyles = $template->css_styles;

        // Build invoice items HTML
        $itemsHtml = '';
        $labels = $data['invoice']['language'] === 'ar' ? [
            'description' => 'الوصف',
            'quantity' => 'الكمية',
            'unit_price' => 'سعر الوحدة',
            'discount' => 'الخصم',
            'tax' => 'الضريبة',
            'total' => 'الإجمالي',
            'subtotal' => 'المجموع الفرعي',
            'discount_total' => 'إجمالي الخصم',
            'tax_total' => 'إجمالي الضريبة',
            'grand_total' => 'المجموع الكلي',
            'paid_amount' => 'المبلغ المدفوع',
            'due_amount' => 'المبلغ المستحق',
            'notes' => 'ملاحظات',
            'terms' => 'الشروط والأحكام',
            'invoice_number' => 'رقم الفاتورة',
            'invoice_date' => 'تاريخ الفاتورة',
            'due_date' => 'تاريخ الاستحقاق',
            'status' => 'الحالة',
        ] : [
            'description' => 'Description',
            'quantity' => 'Quantity',
            'unit_price' => 'Unit Price',
            'discount' => 'Discount',
            'tax' => 'Tax',
            'total' => 'Total',
            'subtotal' => 'Subtotal',
            'discount_total' => 'Discount Total',
            'tax_total' => 'Tax Total',
            'grand_total' => 'Grand Total',
            'paid_amount' => 'Paid Amount',
            'due_amount' => 'Due Amount',
            'notes' => 'Notes',
            'terms' => 'Terms & Conditions',
            'invoice_number' => 'Invoice Number',
            'invoice_date' => 'Invoice Date',
            'due_date' => 'Due Date',
            'status' => 'Status',
        ];

        // Build items table
        foreach ($data['items'] as $item) {
            $itemsHtml .= '<tr>
                <td>' . $item['description'] . '</td>
                <td>' . $item['quantity'] . '</td>
                <td>' . number_format($item['unit_price'], 2) . '</td>
                <td>' . number_format($item['discount_amount'], 2) . '</td>
                <td>' . number_format($item['tax_amount'], 2) . '</td>
                <td>' . number_format($item['total'], 2) . '</td>
            </tr>';
        }

        // Build payments table if any
        $paymentsHtml = '';
        if (!empty($data['payments'])) {
            $paymentLabels = $data['invoice']['language'] === 'ar' ? [
                'date' => 'التاريخ',
                'amount' => 'المبلغ',
                'method' => 'طريقة الدفع',
                'reference' => 'المرجع',
                'payments' => 'المدفوعات',
            ] : [
                'date' => 'Date',
                'amount' => 'Amount',
                'method' => 'Payment Method',
                'reference' => 'Reference',
                'payments' => 'Payments',
            ];

            $paymentsHtml = '<div class="invoice-payments">
                <h3>' . $paymentLabels['payments'] . '</h3>
                <table border="1" cellspacing="0" cellpadding="5" width="100%">
                    <thead>
                        <tr>
                            <th>' . $paymentLabels['date'] . '</th>
                            <th>' . $paymentLabels['amount'] . '</th>
                            <th>' . $paymentLabels['method'] . '</th>
                            <th>' . $paymentLabels['reference'] . '</th>
                        </tr>
                    </thead>
                    <tbody>';

            foreach ($data['payments'] as $payment) {
                $paymentsHtml .= '<tr>
                    <td>' . $payment['payment_date'] . '</td>
                    <td>' . number_format($payment['amount'], 2) . '</td>
                    <td>' . $payment['payment_method'] . '</td>
                    <td>' . $payment['reference'] . '</td>
                </tr>';
            }

            $paymentsHtml .= '</tbody></table></div>';
        }

        // Build the complete HTML
        $html = '<!DOCTYPE html>
        <html lang="' . $data['invoice']['language'] . '" dir="' . ($data['invoice']['language'] === 'ar' ? 'rtl' : 'ltr') . '">
        <head>
            <meta charset="UTF-8">
            <title>' . $labels['invoice_number'] . ': ' . $data['invoice']['invoice_number'] . '</title>
            <style>' . $cssStyles . '</style>
        </head>
        <body>
            ' . $headerHtml . '
            
            <div class="invoice-info">
                <div class="company-info">
                    ' . ($data['company']['logo'] ? '<img src="' . $data['company']['logo'] . '" alt="Logo" style="max-width: 200px; max-height: 100px;">' : '') . '
                    <h2>' . $data['company']['name'] . '</h2>
                    <p>' . $data['company']['address'] . '</p>
                    <p>' . $data['company']['phone'] . '</p>
                    <p>' . $data['company']['email'] . '</p>
                    ' . ($data['company']['vat_number'] ? '<p>VAT: ' . $data['company']['vat_number'] . '</p>' : '') . '
                </div>
                
                <div class="invoice-details">
                    <h2>' . $labels['invoice_number'] . ': ' . $data['invoice']['invoice_number'] . '</h2>
                    <p><strong>' . $labels['invoice_date'] . ':</strong> ' . $data['invoice']['invoice_date'] . '</p>
                    <p><strong>' . $labels['due_date'] . ':</strong> ' . $data['invoice']['due_date'] . '</p>
                    <p><strong>' . $labels['status'] . ':</strong> ' . $data['invoice']['status'] . '</p>
                    ' . ($data['invoice']['zatca_qr_code'] ? '
                    <div class="zatca-qr-code" style="margin-top: 10px;">
                        <h4>' . ($data['invoice']['language'] === 'ar' ? 'رمز الاستجابة السريعة للزكاة والدخل' : 'ZATCA QR Code') . '</h4>
                        <img src="data:image/png;base64,' . $data['invoice']['zatca_qr_code'] . '" alt="ZATCA QR Code" style="max-width: 150px; max-height: 150px;">
                    </div>' : '') . '
                </div>
                
                <div class="customer-info">
                    <h3>' . ($data['invoice']['language'] === 'ar' ? 'العميل' : 'Customer') . '</h3>
                    <p>' . $data['customer']['name'] . '</p>
                    <p>' . $data['customer']['address'] . '</p>
                    <p>' . $data['customer']['phone'] . '</p>
                    <p>' . $data['customer']['email'] . '</p>
                    ' . ($data['customer']['vat_number'] ? '<p>VAT: ' . $data['customer']['vat_number'] . '</p>' : '') . '
                </div>
            </div>
            
            <div class="invoice-items">
                <table border="1" cellspacing="0" cellpadding="5" width="100%">
                    <thead>
                        <tr>
                            <th>' . $labels['description'] . '</th>
                            <th>' . $labels['quantity'] . '</th>
                            <th>' . $labels['unit_price'] . '</th>
                            <th>' . $labels['discount'] . '</th>
                            <th>' . $labels['tax'] . '</th>
                            <th>' . $labels['total'] . '</th>
                        </tr>
                    </thead>
                    <tbody>
                        ' . $itemsHtml . '
                    </tbody>
                    <tfoot>
                        <tr>
                            <td colspan="5" class="text-' . ($data['invoice']['language'] === 'ar' ? 'left' : 'right') . '"><strong>' . $labels['subtotal'] . ':</strong></td>
                            <td>' . number_format($data['invoice']['subtotal'], 2) . '</td>
                        </tr>
                        <tr>
                            <td colspan="5" class="text-' . ($data['invoice']['language'] === 'ar' ? 'left' : 'right') . '"><strong>' . $labels['discount_total'] . ':</strong></td>
                            <td>' . number_format($data['invoice']['discount_amount'], 2) . '</td>
                        </tr>
                        <tr>
                            <td colspan="5" class="text-' . ($data['invoice']['language'] === 'ar' ? 'left' : 'right') . '"><strong>' . $labels['tax_total'] . ' (' . $data['invoice']['tax_rate'] . '%):</strong></td>
                            <td>' . number_format($data['invoice']['tax_amount'], 2) . '</td>
                        </tr>
                        <tr>
                            <td colspan="5" class="text-' . ($data['invoice']['language'] === 'ar' ? 'left' : 'right') . '"><strong>' . $labels['grand_total'] . ':</strong></td>
                            <td>' . number_format($data['invoice']['total_amount'], 2) . '</td>
                        </tr>
                        <tr>
                            <td colspan="5" class="text-' . ($data['invoice']['language'] === 'ar' ? 'left' : 'right') . '"><strong>' . $labels['paid_amount'] . ':</strong></td>
                            <td>' . number_format($data['invoice']['paid_amount'], 2) . '</td>
                        </tr>
                        <tr>
                            <td colspan="5" class="text-' . ($data['invoice']['language'] === 'ar' ? 'left' : 'right') . '"><strong>' . $labels['due_amount'] . ':</strong></td>
                            <td>' . number_format($data['invoice']['due_amount'], 2) . '</td>
                        </tr>
                    </tfoot>
                </table>
            </div>
            
            ' . $paymentsHtml . '
            
            <div class="invoice-notes">
                <h3>' . $labels['notes'] . '</h3>
                <p>' . $data['invoice']['notes'] . '</p>
            </div>
            
            <div class="invoice-terms">
                <h3>' . $labels['terms'] . '</h3>
                <p>' . $data['invoice']['terms_conditions'] . '</p>
            </div>
            
            ' . $footerHtml . '
        </body>
        </html>';

        return $html;
    }

    /**
     * Replace variables in template
     *
     * @param string $content
     * @param array $data
     * @return string
     */
    protected function replaceVariables($content, $data)
    {
        if (empty($content)) {
            return $content;
        }

        // Replace company variables
        $content = str_replace('{company_name}', $data['company']['name'], $content);
        $content = str_replace('{company_address}', $data['company']['address'], $content);
        $content = str_replace('{company_phone}', $data['company']['phone'], $content);
        $content = str_replace('{company_email}', $data['company']['email'], $content);
        $content = str_replace('{company_vat}', $data['company']['vat_number'], $content);

        // Replace customer variables
        $content = str_replace('{customer_name}', $data['customer']['name'], $content);
        $content = str_replace('{customer_address}', $data['customer']['address'], $content);
        $content = str_replace('{customer_phone}', $data['customer']['phone'], $content);
        $content = str_replace('{customer_email}', $data['customer']['email'], $content);
        $content = str_replace('{customer_vat}', $data['customer']['vat_number'], $content);

        // Replace invoice variables
        $content = str_replace('{invoice_number}', $data['invoice']['invoice_number'], $content);
        $content = str_replace('{invoice_date}', $data['invoice']['invoice_date'], $content);
        $content = str_replace('{due_date}', $data['invoice']['due_date'], $content);
        $content = str_replace('{invoice_status}', $data['invoice']['status'], $content);
        $content = str_replace('{invoice_subtotal}', number_format($data['invoice']['subtotal'], 2), $content);
        $content = str_replace('{invoice_tax}', number_format($data['invoice']['tax_amount'], 2), $content);
        $content = str_replace('{invoice_discount}', number_format($data['invoice']['discount_amount'], 2), $content);
        $content = str_replace('{invoice_total}', number_format($data['invoice']['total_amount'], 2), $content);
        $content = str_replace('{invoice_paid}', number_format($data['invoice']['paid_amount'], 2), $content);
        $content = str_replace('{invoice_due}', number_format($data['invoice']['due_amount'], 2), $content);

        return $content;
    }
}
