<?php

namespace App\Services;

use App\Models\InventoryTransaction;
use App\Models\InventoryStock;
use Illuminate\Support\Facades\DB;

class InventoryService
{
    /**
     * تحديث المخزون عند إنشاء مرتجع مبيعات
     *
     * @param \App\Models\SalesReturn $salesReturn
     * @return void
     */
    public function updateInventoryForSalesReturn($salesReturn)
    {
        DB::beginTransaction();

        try {
            foreach ($salesReturn->items as $item) {
                // البحث عن رصيد المخزون أو إنشاء واحد جديد
                $stock = InventoryStock::firstOrNew([
                    'company_id' => $salesReturn->company_id,
                    'warehouse_id' => $salesReturn->warehouse_id,
                    'item_id' => $item->item_id,
                ]);

                if (!$stock->exists) {
                    $stock->quantity = 0;
                    $stock->reserved_quantity = 0;
                }

                // زيادة كمية المخزون (لأن المنتج يعود للمخزون)
                $stock->quantity += $item->quantity;
                $stock->save();

                // إنشاء معاملة مخزون
                $transaction = new InventoryTransaction();
                $transaction->company_id = $salesReturn->company_id;
                $transaction->transaction_type = 'return_from_customer';
                $transaction->item_id = $item->item_id;
                $transaction->warehouse_id = $salesReturn->warehouse_id;
                $transaction->quantity = $item->quantity;
                $transaction->unit_price = $item->unit_price;
                $transaction->reference_number = $salesReturn->return_number;
                $transaction->reference_type = 'sales_return';
                $transaction->reference_id = $salesReturn->id;
                $transaction->created_by = $salesReturn->created_by;
                $transaction->notes = 'إضافة للمخزون من مرتجع المبيعات #' . $salesReturn->return_number;
                $transaction->save();
            }

            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * إلغاء تحديث المخزون عند إلغاء مرتجع مبيعات
     *
     * @param \App\Models\SalesReturn $salesReturn
     * @return void
     */
    public function reverseInventoryForSalesReturn($salesReturn)
    {
        DB::beginTransaction();

        try {
            foreach ($salesReturn->items as $item) {
                // البحث عن رصيد المخزون
                $stock = InventoryStock::where([
                    'company_id' => $salesReturn->company_id,
                    'warehouse_id' => $salesReturn->warehouse_id,
                    'item_id' => $item->item_id,
                ])->first();

                if ($stock) {
                    // تقليل كمية المخزون (لأن المنتج يخرج من المخزون مرة أخرى)
                    $stock->quantity -= $item->quantity;
                    $stock->save();
                }

                // إنشاء معاملة مخزون عكسية
                $transaction = new InventoryTransaction();
                $transaction->company_id = $salesReturn->company_id;
                $transaction->transaction_type = 'sale';
                $transaction->item_id = $item->item_id;
                $transaction->warehouse_id = $salesReturn->warehouse_id;
                $transaction->quantity = -$item->quantity;
                $transaction->unit_price = $item->unit_price;
                $transaction->reference_number = $salesReturn->return_number;
                $transaction->reference_type = 'sales_return_cancel';
                $transaction->reference_id = $salesReturn->id;
                $transaction->created_by = auth()->id();
                $transaction->notes = 'إلغاء إضافة للمخزون من مرتجع المبيعات #' . $salesReturn->return_number;
                $transaction->save();
            }

            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }
}
