<?php

namespace App\Services;

use App\Models\Invoice;
use App\Models\Payment;
use App\Models\MaintenanceContract;
use App\Models\Customer;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class FinancialReportService
{
    /**
     * إنشاء تقرير الإيرادات
     *
     * @param int $companyId
     * @param string $startDate
     * @param string $endDate
     * @param int|null $branchId
     * @return array
     */
    public function generateRevenueReport($companyId, $startDate, $endDate, $branchId = null)
    {
        $startDate = Carbon::parse($startDate)->startOfDay();
        $endDate = Carbon::parse($endDate)->endOfDay();

        $query = Invoice::where('company_id', $companyId)
            ->whereBetween('invoice_date', [$startDate, $endDate]);

        if ($branchId) {
            $query->where('branch_id', $branchId);
        }

        // إجمالي الفواتير
        $totalInvoices = $query->count();
        $totalAmount = $query->sum('total_amount');
        $paidAmount = $query->sum('paid_amount');
        $dueAmount = $query->sum('due_amount');

        // تحليل الإيرادات حسب الشهر
        $monthlyRevenue = $query->select(
            DB::raw('YEAR(invoice_date) as year'),
            DB::raw('MONTH(invoice_date) as month'),
            DB::raw('SUM(total_amount) as total'),
            DB::raw('SUM(paid_amount) as paid'),
            DB::raw('SUM(due_amount) as due')
        )
            ->groupBy('year', 'month')
            ->orderBy('year')
            ->orderBy('month')
            ->get();

        // تحليل الإيرادات حسب العميل
        $customerRevenue = $query->select(
            'customer_id',
            DB::raw('SUM(total_amount) as total'),
            DB::raw('SUM(paid_amount) as paid'),
            DB::raw('SUM(due_amount) as due'),
            DB::raw('COUNT(*) as invoice_count')
        )
            ->with('customer:id,name')
            ->groupBy('customer_id')
            ->orderByDesc('total')
            ->limit(10)
            ->get();

        // تحليل الإيرادات حسب نوع الفاتورة (عقد صيانة أو غيره)
        $typeRevenue = $query->select(
            DB::raw('CASE WHEN maintenance_contract_id IS NOT NULL THEN "عقد صيانة" ELSE "أخرى" END as type'),
            DB::raw('SUM(total_amount) as total'),
            DB::raw('COUNT(*) as count')
        )
            ->groupBy(DB::raw('CASE WHEN maintenance_contract_id IS NOT NULL THEN "عقد صيانة" ELSE "أخرى" END'))
            ->get();

        return [
            'period' => [
                'start_date' => $startDate->format('Y-m-d'),
                'end_date' => $endDate->format('Y-m-d'),
            ],
            'summary' => [
                'total_invoices' => $totalInvoices,
                'total_amount' => $totalAmount,
                'paid_amount' => $paidAmount,
                'due_amount' => $dueAmount,
                'collection_rate' => $totalAmount > 0 ? round(($paidAmount / $totalAmount) * 100, 2) : 0,
            ],
            'monthly_revenue' => $monthlyRevenue,
            'customer_revenue' => $customerRevenue,
            'type_revenue' => $typeRevenue,
        ];
    }

    /**
     * إنشاء تقرير المدفوعات
     *
     * @param int $companyId
     * @param string $startDate
     * @param string $endDate
     * @param int|null $branchId
     * @return array
     */
    public function generatePaymentsReport($companyId, $startDate, $endDate, $branchId = null)
    {
        $startDate = Carbon::parse($startDate)->startOfDay();
        $endDate = Carbon::parse($endDate)->endOfDay();

        $query = Payment::where('company_id', $companyId)
            ->whereBetween('payment_date', [$startDate, $endDate]);

        if ($branchId) {
            $query->where('branch_id', $branchId);
        }

        // إجمالي المدفوعات
        $totalPayments = $query->count();
        $totalAmount = $query->sum('amount');

        // تحليل المدفوعات حسب الشهر
        $monthlyPayments = $query->select(
            DB::raw('YEAR(payment_date) as year'),
            DB::raw('MONTH(payment_date) as month'),
            DB::raw('SUM(amount) as total'),
            DB::raw('COUNT(*) as count')
        )
            ->groupBy('year', 'month')
            ->orderBy('year')
            ->orderBy('month')
            ->get();

        // تحليل المدفوعات حسب طريقة الدفع
        $paymentMethodStats = $query->select(
            'payment_method_id',
            DB::raw('SUM(amount) as total'),
            DB::raw('COUNT(*) as count')
        )
            ->with('paymentMethod:id,name')
            ->groupBy('payment_method_id')
            ->orderByDesc('total')
            ->get();

        // تحليل المدفوعات حسب العميل
        $customerPayments = $query->select(
            'customer_id',
            DB::raw('SUM(amount) as total'),
            DB::raw('COUNT(*) as count')
        )
            ->with('customer:id,name')
            ->groupBy('customer_id')
            ->orderByDesc('total')
            ->limit(10)
            ->get();

        return [
            'period' => [
                'start_date' => $startDate->format('Y-m-d'),
                'end_date' => $endDate->format('Y-m-d'),
            ],
            'summary' => [
                'total_payments' => $totalPayments,
                'total_amount' => $totalAmount,
            ],
            'monthly_payments' => $monthlyPayments,
            'payment_methods' => $paymentMethodStats,
            'customer_payments' => $customerPayments,
        ];
    }

    /**
     * إنشاء تقرير العقود
     *
     * @param int $companyId
     * @param string|null $startDate
     * @param string|null $endDate
     * @param int|null $branchId
     * @return array
     */
    public function generateContractsReport($companyId, $startDate = null, $endDate = null, $branchId = null)
    {
        $query = MaintenanceContract::where('company_id', $companyId);

        if ($startDate && $endDate) {
            $startDate = Carbon::parse($startDate)->startOfDay();
            $endDate = Carbon::parse($endDate)->endOfDay();
            $query->where(function ($q) use ($startDate, $endDate) {
                $q->whereBetween('start_date', [$startDate, $endDate])
                    ->orWhereBetween('end_date', [$startDate, $endDate])
                    ->orWhere(function ($q2) use ($startDate, $endDate) {
                        $q2->where('start_date', '<=', $startDate)
                            ->where('end_date', '>=', $endDate);
                    });
            });
        }

        if ($branchId) {
            $query->where('branch_id', $branchId);
        }

        // إحصائيات العقود
        $totalContracts = $query->count();
        $activeContracts = $query->where('status', 'active')->count();
        $expiredContracts = $query->where('status', 'expired')->count();
        $cancelledContracts = $query->where('status', 'cancelled')->count();
        $totalValue = $query->sum('total_amount');

        // العقود حسب دورة الفوترة
        $billingCycleStats = $query->select(
            'billing_cycle',
            DB::raw('COUNT(*) as count'),
            DB::raw('SUM(total_amount) as total')
        )
            ->groupBy('billing_cycle')
            ->get();

        // العقود حسب عدد الزيارات السنوية
        $visitsStats = $query->select(
            'visits_per_year',
            DB::raw('COUNT(*) as count')
        )
            ->groupBy('visits_per_year')
            ->orderBy('visits_per_year')
            ->get();

        // العقود التي ستنتهي قريبًا
        $today = Carbon::today();
        $expiringContracts = MaintenanceContract::where('company_id', $companyId)
            ->where('status', 'active')
            ->where('end_date', '>=', $today)
            ->where('end_date', '<=', $today->copy()->addDays(30))
            ->with('customer:id,name')
            ->orderBy('end_date')
            ->get();

        return [
            'summary' => [
                'total_contracts' => $totalContracts,
                'active_contracts' => $activeContracts,
                'expired_contracts' => $expiredContracts,
                'cancelled_contracts' => $cancelledContracts,
                'total_value' => $totalValue,
            ],
            'billing_cycles' => $billingCycleStats,
            'visits_stats' => $visitsStats,
            'expiring_contracts' => $expiringContracts,
        ];
    }

    /**
     * إنشاء تقرير الذمم المدينة
     *
     * @param int $companyId
     * @param int|null $branchId
     * @return array
     */
    public function generateAccountsReceivableReport($companyId, $branchId = null)
    {
        $query = Invoice::where('company_id', $companyId)
            ->where('due_amount', '>', 0);

        if ($branchId) {
            $query->where('branch_id', $branchId);
        }

        // إجمالي الذمم المدينة
        $totalDue = $query->sum('due_amount');
        $totalInvoices = $query->count();

        // تحليل الذمم المدينة حسب العمر
        $today = Carbon::today();
        $current = $query->where('due_date', '>=', $today)->sum('due_amount');
        $overdue1To30 = $query->where('due_date', '<', $today)
            ->where('due_date', '>=', $today->copy()->subDays(30))
            ->sum('due_amount');
        $overdue31To60 = $query->where('due_date', '<', $today->copy()->subDays(30))
            ->where('due_date', '>=', $today->copy()->subDays(60))
            ->sum('due_amount');
        $overdue61To90 = $query->where('due_date', '<', $today->copy()->subDays(60))
            ->where('due_date', '>=', $today->copy()->subDays(90))
            ->sum('due_amount');
        $overdue90Plus = $query->where('due_date', '<', $today->copy()->subDays(90))
            ->sum('due_amount');

        // الذمم المدينة حسب العميل
        $customerAR = $query->select(
            'customer_id',
            DB::raw('SUM(due_amount) as total_due'),
            DB::raw('COUNT(*) as invoice_count')
        )
            ->with('customer:id,name')
            ->groupBy('customer_id')
            ->orderByDesc('total_due')
            ->limit(10)
            ->get();

        // الفواتير المتأخرة
        $overdueInvoices = $query->where('due_date', '<', $today)
            ->with('customer:id,name')
            ->orderBy('due_date')
            ->limit(20)
            ->get();

        return [
            'summary' => [
                'total_due' => $totalDue,
                'total_invoices' => $totalInvoices,
            ],
            'aging' => [
                'current' => $current,
                'overdue_1_30' => $overdue1To30,
                'overdue_31_60' => $overdue31To60,
                'overdue_61_90' => $overdue61To90,
                'overdue_90_plus' => $overdue90Plus,
            ],
            'customer_ar' => $customerAR,
            'overdue_invoices' => $overdueInvoices,
        ];
    }

    /**
     * إنشاء تقرير الإيرادات والمصروفات
     *
     * @param int $companyId
     * @param string $startDate
     * @param string $endDate
     * @param string $groupBy
     * @param bool $includeDetails
     * @return array
     */
    public function generateIncomeExpenseReport($companyId, $startDate, $endDate, $groupBy = 'month', $includeDetails = false)
    {
        $startDate = Carbon::parse($startDate)->startOfDay();
        $endDate = Carbon::parse($endDate)->endOfDay();

        // الحصول على الإيرادات
        $incomeQuery = Payment::where('company_id', $companyId)
            ->whereBetween('payment_date', [$startDate, $endDate]);

        $totalIncome = $incomeQuery->sum('amount');

        // الحصول على المصروفات
        // هنا نفترض وجود جدول للمصروفات، يمكن تعديله حسب هيكل قاعدة البيانات
        $totalExpenses = 0; // يتم استبداله بالاستعلام الفعلي للمصروفات

        // تجميع البيانات حسب الفترة المحددة
        $periods = [];

        if ($groupBy == 'month') {
            // تجميع حسب الشهر
            $currentDate = $startDate->copy()->startOfMonth();
            $endMonth = $endDate->copy()->endOfMonth();

            while ($currentDate->lte($endMonth)) {
                $periodStart = $currentDate->copy()->startOfMonth();
                $periodEnd = $currentDate->copy()->endOfMonth();

                $periodIncome = $incomeQuery->whereBetween('payment_date', [$periodStart, $periodEnd])->sum('amount');
                $periodExpenses = 0; // يتم استبداله بالاستعلام الفعلي للمصروفات

                $periods[] = [
                    'name' => $currentDate->format('F Y'),
                    'start_date' => $periodStart->format('Y-m-d'),
                    'end_date' => $periodEnd->format('Y-m-d'),
                    'income' => $periodIncome,
                    'expenses' => $periodExpenses,
                    'net_income' => $periodIncome - $periodExpenses
                ];

                $currentDate->addMonth();
            }
        } elseif ($groupBy == 'quarter') {
            // تجميع حسب ربع السنة
            $currentDate = $startDate->copy()->startOfQuarter();
            $endQuarter = $endDate->copy()->endOfQuarter();

            while ($currentDate->lte($endQuarter)) {
                $periodStart = $currentDate->copy()->startOfQuarter();
                $periodEnd = $currentDate->copy()->endOfQuarter();

                $periodIncome = $incomeQuery->whereBetween('payment_date', [$periodStart, $periodEnd])->sum('amount');
                $periodExpenses = 0; // يتم استبداله بالاستعلام الفعلي للمصروفات

                $periods[] = [
                    'name' => 'Q' . $currentDate->quarter . ' ' . $currentDate->year,
                    'start_date' => $periodStart->format('Y-m-d'),
                    'end_date' => $periodEnd->format('Y-m-d'),
                    'income' => $periodIncome,
                    'expenses' => $periodExpenses,
                    'net_income' => $periodIncome - $periodExpenses
                ];

                $currentDate->addQuarter();
            }
        } elseif ($groupBy == 'year') {
            // تجميع حسب السنة
            $currentDate = $startDate->copy()->startOfYear();
            $endYear = $endDate->copy()->endOfYear();

            while ($currentDate->lte($endYear)) {
                $periodStart = $currentDate->copy()->startOfYear();
                $periodEnd = $currentDate->copy()->endOfYear();

                $periodIncome = $incomeQuery->whereBetween('payment_date', [$periodStart, $periodEnd])->sum('amount');
                $periodExpenses = 0; // يتم استبداله بالاستعلام الفعلي للمصروفات

                $periods[] = [
                    'name' => $currentDate->year,
                    'start_date' => $periodStart->format('Y-m-d'),
                    'end_date' => $periodEnd->format('Y-m-d'),
                    'income' => $periodIncome,
                    'expenses' => $periodExpenses,
                    'net_income' => $periodIncome - $periodExpenses
                ];

                $currentDate->addYear();
            }
        }

        // تفاصيل الإيرادات والمصروفات
        $incomeDetails = [];
        $expenseDetails = [];

        if ($includeDetails) {
            // تفاصيل الإيرادات حسب المصدر
            $incomeDetails = [
                'فواتير الصيانة' => $totalIncome * 0.6, // مثال، يتم استبداله بالاستعلام الفعلي
                'فواتير الإصلاح' => $totalIncome * 0.3, // مثال، يتم استبداله بالاستعلام الفعلي
                'فواتير أخرى' => $totalIncome * 0.1, // مثال، يتم استبداله بالاستعلام الفعلي
            ];

            // تفاصيل المصروفات حسب الفئة
            $expenseDetails = [
                'رواتب الموظفين' => $totalExpenses * 0.5, // مثال، يتم استبداله بالاستعلام الفعلي
                'قطع غيار' => $totalExpenses * 0.3, // مثال، يتم استبداله بالاستعلام الفعلي
                'مصروفات إدارية' => $totalExpenses * 0.2, // مثال، يتم استبداله بالاستعلام الفعلي
            ];
        }

        return [
            'period' => [
                'start_date' => $startDate->format('Y-m-d'),
                'end_date' => $endDate->format('Y-m-d'),
            ],
            'group_by' => $groupBy,
            'summary' => [
                'total_income' => $totalIncome,
                'total_expenses' => $totalExpenses,
                'net_income' => $totalIncome - $totalExpenses,
            ],
            'periods' => $periods,
            'income_details' => $incomeDetails,
            'expense_details' => $expenseDetails,
        ];
    }

    /**
     * إنشاء تقرير الربحية
     *
     * @param int $companyId
     * @param string $startDate
     * @param string $endDate
     * @param string $groupBy
     * @param bool $includeInactive
     * @return array
     */
    public function generateProfitabilityReport($companyId, $startDate, $endDate, $groupBy = 'customer', $includeInactive = false)
    {
        $startDate = Carbon::parse($startDate)->startOfDay();
        $endDate = Carbon::parse($endDate)->endOfDay();

        // الحصول على البيانات حسب نوع التجميع
        $items = [];
        $totalRevenue = 0;
        $totalCost = 0;
        $totalProfit = 0;

        if ($groupBy == 'customer') {
            // تجميع حسب العميل
            $customers = Customer::where('company_id', $companyId);

            if (!$includeInactive) {
                $customers->where('status', 'active');
            }

            $customers = $customers->get();

            foreach ($customers as $customer) {
                // الإيرادات من الفواتير المدفوعة
                $revenue = Payment::where('company_id', $companyId)
                    ->where('customer_id', $customer->id)
                    ->whereBetween('payment_date', [$startDate, $endDate])
                    ->sum('amount');

                // التكاليف المباشرة (مثال، يمكن تعديله حسب هيكل قاعدة البيانات)
                $cost = $revenue * 0.6; // نفترض أن التكلفة 60% من الإيرادات

                // الربح
                $profit = $revenue - $cost;

                // هامش الربح
                $profitMargin = $revenue > 0 ? ($profit / $revenue) * 100 : 0;

                $items[] = [
                    'id' => $customer->id,
                    'name' => $customer->name,
                    'revenue' => $revenue,
                    'cost' => $cost,
                    'profit' => $profit,
                    'profit_margin' => $profitMargin,
                ];

                $totalRevenue += $revenue;
                $totalCost += $cost;
                $totalProfit += $profit;
            }
        } elseif ($groupBy == 'elevator') {
            // تجميع حسب المصعد
            // يتم تنفيذه بطريقة مماثلة للتجميع حسب العميل
            // هذا مجرد مثال، يمكن تعديله حسب هيكل قاعدة البيانات
        } elseif ($groupBy == 'contract') {
            // تجميع حسب العقد
            // يتم تنفيذه بطريقة مماثلة للتجميع حسب العميل
            // هذا مجرد مثال، يمكن تعديله حسب هيكل قاعدة البيانات
        }

        // حساب هامش الربح الإجمالي
        $profitMargin = $totalRevenue > 0 ? ($totalProfit / $totalRevenue) * 100 : 0;

        return [
            'period' => [
                'start_date' => $startDate->format('Y-m-d'),
                'end_date' => $endDate->format('Y-m-d'),
            ],
            'group_by' => $groupBy,
            'summary' => [
                'total_revenue' => $totalRevenue,
                'total_cost' => $totalCost,
                'total_profit' => $totalProfit,
                'profit_margin' => $profitMargin,
            ],
            'items' => $items,
        ];
    }

    /**
     * إنشاء تقرير كشف حساب العميل
     *
     * @param int $companyId
     * @param int $customerId
     * @param string|null $startDate
     * @param string|null $endDate
     * @return array
     */
    public function generateCustomerStatementReport($companyId, $customerId, $startDate = null, $endDate = null)
    {
        $customer = Customer::findOrFail($customerId);

        if ($startDate) {
            $startDate = Carbon::parse($startDate)->startOfDay();
        } else {
            $startDate = Carbon::now()->subMonths(6)->startOfDay();
        }

        if ($endDate) {
            $endDate = Carbon::parse($endDate)->endOfDay();
        } else {
            $endDate = Carbon::now()->endOfDay();
        }

        // الرصيد الافتتاحي (مجموع الفواتير المستحقة قبل تاريخ البداية)
        $openingBalance = Invoice::where('company_id', $companyId)
            ->where('customer_id', $customerId)
            ->where('invoice_date', '<', $startDate)
            ->sum('due_amount');

        // جميع المعاملات خلال الفترة
        $transactions = [];

        // الفواتير
        $invoices = Invoice::where('company_id', $companyId)
            ->where('customer_id', $customerId)
            ->whereBetween('invoice_date', [$startDate, $endDate])
            ->get();

        foreach ($invoices as $invoice) {
            $transactions[] = [
                'date' => $invoice->invoice_date,
                'type' => 'invoice',
                'reference' => $invoice->invoice_number,
                'description' => 'فاتورة رقم ' . $invoice->invoice_number,
                'debit' => $invoice->total_amount,
                'credit' => 0,
                'balance' => 0, // سيتم حسابه لاحقًا
            ];
        }

        // المدفوعات
        $payments = Payment::where('company_id', $companyId)
            ->where('customer_id', $customerId)
            ->whereBetween('payment_date', [$startDate, $endDate])
            ->get();

        foreach ($payments as $payment) {
            $transactions[] = [
                'date' => $payment->payment_date,
                'type' => 'payment',
                'reference' => $payment->reference_number,
                'description' => 'دفعة رقم ' . $payment->reference_number . ' (' . $payment->paymentMethod->name . ')',
                'debit' => 0,
                'credit' => $payment->amount,
                'balance' => 0, // سيتم حسابه لاحقًا
            ];
        }

        // ترتيب المعاملات حسب التاريخ
        usort($transactions, function ($a, $b) {
            return strtotime($a['date']) - strtotime($b['date']);
        });

        // حساب الرصيد التراكمي
        $runningBalance = $openingBalance;
        foreach ($transactions as &$transaction) {
            $runningBalance = $runningBalance + $transaction['debit'] - $transaction['credit'];
            $transaction['balance'] = $runningBalance;
        }

        // ملخص الحساب
        $totalInvoices = $invoices->sum('total_amount');
        $totalPayments = $payments->sum('amount');
        $closingBalance = $openingBalance + $totalInvoices - $totalPayments;

        return [
            'customer' => $customer,
            'period' => [
                'start_date' => $startDate->format('Y-m-d'),
                'end_date' => $endDate->format('Y-m-d'),
            ],
            'summary' => [
                'opening_balance' => $openingBalance,
                'total_invoices' => $totalInvoices,
                'total_payments' => $totalPayments,
                'closing_balance' => $closingBalance,
            ],
            'transactions' => $transactions,
        ];
    }
}
