<?php

namespace App\Services;

use App\Models\Customer;
use App\Models\Notification;
use App\Models\User;
use Illuminate\Support\Facades\Log;

class CustomerNotificationService
{
    /**
     * خدمة الإشعارات العامة
     *
     * @var \App\Services\NotificationService
     */
    protected $notificationService;

    /**
     * خدمة تسجيل النشاط
     *
     * @var \App\Services\ActivityLogService
     */
    protected $activityLogService;

    /**
     * إنشاء مثيل جديد من الخدمة
     *
     * @param \App\Services\NotificationService $notificationService
     * @param \App\Services\ActivityLogService $activityLogService
     */
    public function __construct(NotificationService $notificationService, ActivityLogService $activityLogService)
    {
        $this->notificationService = $notificationService;
        $this->activityLogService = $activityLogService;
    }

    /**
     * إنشاء إشعار لعميل جديد
     *
     * @param \App\Models\Customer $customer
     * @return \App\Models\Notification|null
     */
    public function newCustomerNotification(Customer $customer)
    {
        try {
            // تحديد المستخدمين الذين سيتلقون الإشعار (مثلاً المديرين ومسؤولي المبيعات)
            $users = User::where('company_id', $customer->company_id)
                ->get()
                ->filter(function ($user) {
                    return $user->hasPermission('view_customers');
                });

            if ($users->isEmpty()) {
                Log::warning('لا يوجد مستخدمين لديهم صلاحية عرض العملاء في الشركة: ' . $customer->company_id);
            }

            $lastNotification = null;

            // إنشاء إشعار لكل مستخدم
            foreach ($users as $user) {
                $notificationData = [
                    'company_id' => $customer->company_id,
                    'user_id' => $user->id,
                    'type' => 'customer_new',
                    'notifiable_type' => 'App\\Models\\Customer',
                    'notifiable_id' => $customer->id,
                    'title' => 'عميل جديد',
                    'message' => 'تم إضافة عميل جديد: ' . $customer->name,
                    'data' => [
                        'customer_id' => $customer->id,
                        'customer_name' => $customer->name,
                        'customer_phone' => $customer->phone,
                        'customer_email' => $customer->email,
                        'created_by' => auth()->id(),
                    ],
                    'status' => 'pending',
                    'priority' => 'normal'
                ];

                $notification = $this->notificationService->createNotification($notificationData);
                $lastNotification = $notification;

                // تسجيل النشاط
                $this->activityLogService->log(
                    'customer_notification',
                    'تم إنشاء إشعار لعميل جديد',
                    $customer,
                    null,
                    ['notification_id' => $notification->id, 'user_id' => $user->id]
                );

                Log::info('تم إنشاء إشعار للمستخدم: ' . $user->email . ' للعميل الجديد: ' . $customer->name, [
                    'user_id' => $user->id,
                    'customer_id' => $customer->id,
                    'notification_id' => $notification->id
                ]);
            }

            // إرجاع الإشعار الأخير (للتوافق مع الكود القديم)
            return $lastNotification;
        } catch (\Exception $e) {
            Log::error('فشل إنشاء إشعار للعميل الجديد: ' . $e->getMessage(), [
                'exception' => get_class($e),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'customer_id' => $customer->id
            ]);
            return null;
        }
    }

    /**
     * إنشاء إشعار لتحديث بيانات العميل
     *
     * @param \App\Models\Customer $customer
     * @param array $updatedFields
     * @return \App\Models\Notification|null
     */
    public function customerUpdateNotification(Customer $customer, array $updatedFields)
    {
        try {
            // تحديد المستخدمين الذين سيتلقون الإشعار - استخدام الطريقة الصحيحة للحصول على المستخدمين
            $users = User::all()->filter(function ($user) {
                return $user->hasPermission('view_customers');
            });

            // إضافة سجل للتشخيص
            Log::info('عدد المستخدمين الذين سيتلقون الإشعار: ' . $users->count());

            // إنشاء الإشعار
            // إنشاء إشعار لكل مستخدم
            foreach ($users as $user) {
                $notificationData = [
                    'company_id' => $customer->company_id ?? auth()->user()->company_id,
                    'user_id' => $user->id,
                    'type' => 'customer_update',
                    'notifiable_type' => 'App\\Models\\Customer',
                    'notifiable_id' => $customer->id,
                    'title' => 'تحديث بيانات عميل',
                    'message' => 'تم تحديث بيانات العميل: ' . $customer->name,
                    'data' => [
                        'customer_id' => $customer->id,
                        'customer_name' => $customer->name,
                        'updated_fields' => $updatedFields,
                        'updated_by' => auth()->id(),
                    ],
                    'status' => 'pending',
                    'priority' => 'normal'
                ];

                $notification = $this->notificationService->createNotification($notificationData);

                // تسجيل النشاط
                $this->activityLogService->log(
                    'customer_notification',
                    'تم إنشاء إشعار لتحديث بيانات العميل',
                    $customer,
                    null,
                    ['notification_id' => $notification->id]
                );
            }

            // إرجاع الإشعار الأخير (للتوافق مع الكود القديم)
            return $notification ?? null;
        } catch (\Exception $e) {
            Log::error('فشل إنشاء إشعار لتحديث بيانات العميل: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * إنشاء إشعار لطلب خدمة من العميل
     *
     * @param \App\Models\Customer $customer
     * @param string $serviceType
     * @param string $description
     * @return \App\Models\Notification|null
     */
    public function customerServiceRequestNotification(Customer $customer, string $serviceType, string $description)
    {
        try {
            // تحديد المستخدمين الذين سيتلقون الإشعار (مثلاً الفنيين والمديرين)
            $users = User::all()->filter(function ($user) {
                return $user->hasPermission('handle_service_requests');
            });

            // إنشاء إشعار لكل مستخدم
            foreach ($users as $user) {
                $notificationData = [
                    'company_id' => $customer->company_id ?? auth()->user()->company_id,
                    'user_id' => $user->id,
                    'type' => 'customer_service_request',
                    'notifiable_type' => 'App\\Models\\Customer',
                    'notifiable_id' => $customer->id,
                    'title' => 'طلب خدمة جديد',
                    'message' => 'طلب خدمة جديد من العميل: ' . $customer->name,
                    'data' => [
                        'customer_id' => $customer->id,
                        'customer_name' => $customer->name,
                        'service_type' => $serviceType,
                        'description' => $description,
                        'priority' => 'متوسط',
                        'created_at' => now()->format('Y-m-d H:i:s'),
                    ],
                    'status' => 'pending',
                    'priority' => 'normal'
                ];

                $notification = $this->notificationService->createNotification($notificationData);

                // تسجيل النشاط
                $this->activityLogService->log(
                    'customer_notification',
                    'تم إنشاء إشعار لطلب خدمة من العميل',
                    $customer,
                    null,
                    [
                        'notification_id' => $notification->id,
                        'service_type' => $serviceType
                    ]
                );
            }

            // إرجاع الإشعار الأخير (للتوافق مع الكود القديم)
            return $notification ?? null;
        } catch (\Exception $e) {
            Log::error('فشل إنشاء إشعار لطلب خدمة من العميل: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * إنشاء إشعار لشكوى من العميل
     *
     * @param \App\Models\Customer $customer
     * @param string $complaintType
     * @param string $description
     * @param string $priority
     * @return \App\Models\Notification|null
     */
    public function customerComplaintNotification(Customer $customer, string $complaintType, string $description, string $priority = 'عالي')
    {
        try {
            // تحديد المستخدمين الذين سيتلقون الإشعار (مثلاً المديرين ومسؤولي خدمة العملاء)
            $users = User::all()->filter(function ($user) {
                return $user->hasPermission('handle_customer_complaints');
            });

            // إنشاء إشعار لكل مستخدم
            foreach ($users as $user) {
                $notificationData = [
                    'company_id' => $customer->company_id ?? auth()->user()->company_id,
                    'user_id' => $user->id,
                    'type' => 'customer_complaint',
                    'notifiable_type' => 'App\\Models\\Customer',
                    'notifiable_id' => $customer->id,
                    'title' => 'شكوى عميل جديدة',
                    'message' => 'شكوى جديدة من العميل: ' . $customer->name,
                    'data' => [
                        'customer_id' => $customer->id,
                        'customer_name' => $customer->name,
                        'complaint_type' => $complaintType,
                        'description' => $description,
                        'priority' => $priority,
                        'created_at' => now()->format('Y-m-d H:i:s'),
                    ],
                    'status' => 'pending',
                    'priority' => $priority == 'عالي' ? 'high' : 'normal'
                ];

                $notification = $this->notificationService->createNotification($notificationData);

                // تسجيل النشاط
                $this->activityLogService->log(
                    'customer_notification',
                    'تم إنشاء إشعار لشكوى من العميل',
                    $customer,
                    null,
                    [
                        'notification_id' => $notification->id,
                        'complaint_type' => $complaintType,
                        'priority' => $priority
                    ]
                );
            }

            // إرجاع الإشعار الأخير (للتوافق مع الكود القديم)
            return $notification ?? null;
        } catch (\Exception $e) {
            Log::error('فشل إنشاء إشعار لشكوى من العميل: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * إنشاء إشعار تذكير لمتابعة العميل
     *
     * @param \App\Models\Customer $customer
     * @param \DateTime $followUpDate
     * @param string $notes
     * @param \App\Models\User|null $assignedTo
     * @return \App\Models\Notification|null
     */
    public function customerFollowUpReminderNotification(Customer $customer, \DateTime $followUpDate, string $notes, ?User $assignedTo = null)
    {
        try {
            // تحديد المستخدمين الذين سيتلقون الإشعار
            $users = collect();

            if ($assignedTo) {
                $users->push($assignedTo);
            } else {
                $users = User::all()->filter(function ($user) {
                    return $user->hasPermission('follow_up_customers');
                });
            }

            // إنشاء إشعار لكل مستخدم
            foreach ($users as $user) {
                $notificationData = [
                    'company_id' => $customer->company_id ?? auth()->user()->company_id,
                    'user_id' => $user->id,
                    'type' => 'customer_follow_up',
                    'notifiable_type' => 'App\\Models\\Customer',
                    'notifiable_id' => $customer->id,
                    'title' => 'تذكير متابعة عميل',
                    'message' => 'تذكير لمتابعة العميل: ' . $customer->name,
                    'data' => [
                        'customer_id' => $customer->id,
                        'customer_name' => $customer->name,
                        'follow_up_date' => $followUpDate->format('Y-m-d'),
                        'notes' => $notes,
                        'assigned_to' => $assignedTo ? $assignedTo->id : null,
                        'created_by' => auth()->id(),
                    ],
                    'status' => 'pending',
                    'priority' => 'normal'
                ];

                $notification = $this->notificationService->createNotification($notificationData);

                // تسجيل النشاط
                $this->activityLogService->log(
                    'customer_notification',
                    'تم إنشاء إشعار تذكير لمتابعة العميل',
                    $customer,
                    null,
                    [
                        'notification_id' => $notification->id,
                        'follow_up_date' => $followUpDate->format('Y-m-d'),
                        'assigned_to' => $assignedTo ? $assignedTo->id : null
                    ]
                );
            }

            // إرجاع الإشعار الأخير (للتوافق مع الكود القديم)
            return $notification ?? null;
        } catch (\Exception $e) {
            Log::error('فشل إنشاء إشعار تذكير لمتابعة العميل: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * فحص وإرسال تذكيرات متابعة العملاء المستحقة
     *
     * @return int عدد الإشعارات التي تم إنشاؤها
     */
    public function checkCustomerFollowUpReminders()
    {
        try {
            // هنا يمكن إضافة المنطق الخاص بفحص تذكيرات متابعة العملاء المستحقة
            // وإنشاء الإشعارات المناسبة

            // مثال: البحث عن تذكيرات المتابعة المستحقة اليوم
            // $followUps = CustomerFollowUp::whereDate('follow_up_date', today())->get();

            // عدد الإشعارات التي تم إنشاؤها
            $count = 0;

            // تسجيل النشاط
            $this->activityLogService->log(
                'customer_follow_up_check',
                'تم فحص تذكيرات متابعة العملاء',
                null,
                null,
                ['created_count' => $count]
            );

            return $count;
        } catch (\Exception $e) {
            Log::error('فشل فحص تذكيرات متابعة العملاء: ' . $e->getMessage());
            return 0;
        }
    }
}
