<?php

namespace App\Services;

use App\Models\Customer;
use App\Models\ChartOfAccount;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CustomerAccountService
{
    /**
     * إنشاء أو تحديث حساب في شجرة الحسابات للعميل
     *
     * @param Customer $customer
     * @return ChartOfAccount|null
     */
    public function createOrUpdateCustomerAccount(Customer $customer)
    {
        try {
            DB::beginTransaction();

            // البحث عن حساب الذمم المدينة (الحساب الأب للعملاء)
            $receivablesAccount = ChartOfAccount::where('account_code', '1200')
                ->where('company_id', $customer->company_id)
                ->first();

            if (!$receivablesAccount) {
                // إذا لم يوجد حساب الذمم المدينة، نبحث عن حساب الأصول المتداولة
                $currentAssetsAccount = ChartOfAccount::where('account_code', '1100')
                    ->where('company_id', $customer->company_id)
                    ->first();

                if (!$currentAssetsAccount) {
                    // إذا لم يوجد حساب الأصول المتداولة، نبحث عن حساب الأصول
                    $assetsAccount = ChartOfAccount::where('account_code', '1000')
                        ->where('company_id', $customer->company_id)
                        ->first();

                    if (!$assetsAccount) {
                        // إنشاء هيكل حسابات الأصول الأساسي بالكامل
                        $assetsAccount = $this->createAssetsAccountStructure($customer->company_id);
                        $currentAssetsAccount = $this->createCurrentAssetsAccount($assetsAccount);
                    } else {
                        // إنشاء حساب الأصول المتداولة تحت حساب الأصول
                        $currentAssetsAccount = $this->createCurrentAssetsAccount($assetsAccount);
                    }
                }

                // إنشاء حساب الذمم المدينة تحت حساب الأصول المتداولة
                $receivablesAccount = $this->createReceivablesAccount($currentAssetsAccount);
            }

            // البحث عن الحساب المرتبط بالعميل إن وجد
            $customerAccount = ChartOfAccount::where('company_id', $customer->company_id)
                ->where('sub_type', 'customer_receivable')
                ->where('name', 'like', '%' . $customer->name . '%')
                ->first();

            if ($customerAccount) {
                // إذا كان هناك حساب مرتبط بالعميل، نقوم بتحديثه
                $customerAccount->name = 'حساب العميل: ' . $customer->name;
                $customerAccount->name_en = 'Customer Account: ' . $customer->name;
                $customerAccount->description = 'حساب آلي للعميل ' . $customer->name;
                $customerAccount->save();
            } else {
                // إنشاء حساب جديد في شجرة الحسابات للعميل
                $customerAccount = $this->createCustomerChartAccount($customer, $receivablesAccount);
            }

            DB::commit();
            return $customerAccount;
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('خطأ في إنشاء حساب العميل: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * إنشاء حساب جديد في شجرة الحسابات للعميل
     *
     * @param Customer $customer
     * @param ChartOfAccount $parentAccount
     * @return ChartOfAccount
     */
    private function createCustomerChartAccount(Customer $customer, ChartOfAccount $parentAccount)
    {
        // توليد رمز حساب فريد
        $lastAccountCode = ChartOfAccount::where('parent_id', $parentAccount->id)
            ->orderBy('account_code', 'desc')
            ->value('account_code');

        $newAccountCode = $lastAccountCode ? (intval($lastAccountCode) + 1) : ($parentAccount->account_code . '01');

        // إنشاء حساب جديد في شجرة الحسابات
        return ChartOfAccount::create([
            'company_id' => $customer->company_id,
            'parent_id' => $parentAccount->id,
            'account_code' => $newAccountCode,
            'name' => 'حساب العميل: ' . $customer->name,
            'name_en' => 'Customer Account: ' . $customer->name,
            'account_type' => 'asset',
            'sub_type' => 'customer_receivable',
            'description' => 'حساب آلي للعميل ' . $customer->name,
            'is_active' => true,
            'is_system' => false,
            'opening_balance' => 0,
            'current_balance' => 0,
            'currency' => 'SAR',
            'balance_type' => 'debit',
            'is_cash_account' => false,
            'is_bank_account' => false,
            'level' => $parentAccount->level + 1,
            'is_parent' => false
        ]);
    }

    /**
     * إنشاء هيكل حسابات الأصول الأساسي
     *
     * @param int $companyId
     * @return ChartOfAccount
     */
    public function createAssetsAccountStructure($companyId)
    {
        // إنشاء حساب الأصول (1000)
        return ChartOfAccount::create([
            'company_id' => $companyId,
            'parent_id' => null,
            'account_code' => '1000',
            'name' => 'الأصول',
            'name_en' => 'Assets',
            'account_type' => 'asset',
            'sub_type' => 'asset',
            'description' => 'حساب الأصول الرئيسي',
            'is_active' => true,
            'is_system' => true,
            'opening_balance' => 0,
            'current_balance' => 0,
            'currency' => 'SAR',
            'balance_type' => 'debit',
            'is_cash_account' => false,
            'is_bank_account' => false,
            'level' => 1,
            'is_parent' => true
        ]);
    }

    /**
     * إنشاء حساب الأصول المتداولة تحت حساب الأصول
     *
     * @param ChartOfAccount $assetsAccount
     * @return ChartOfAccount
     */
    public function createCurrentAssetsAccount(ChartOfAccount $assetsAccount)
    {
        // إنشاء حساب الأصول المتداولة (1100)
        return ChartOfAccount::create([
            'company_id' => $assetsAccount->company_id,
            'parent_id' => $assetsAccount->id,
            'account_code' => '1100',
            'name' => 'الأصول المتداولة',
            'name_en' => 'Current Assets',
            'account_type' => 'asset',
            'sub_type' => 'current_asset',
            'description' => 'حساب الأصول المتداولة',
            'is_active' => true,
            'is_system' => true,
            'opening_balance' => 0,
            'current_balance' => 0,
            'currency' => 'SAR',
            'balance_type' => 'debit',
            'is_cash_account' => false,
            'is_bank_account' => false,
            'level' => $assetsAccount->level + 1,
            'is_parent' => true
        ]);
    }

    /**
     * إنشاء حساب الذمم المدينة تحت حساب الأصول المتداولة
     *
     * @param ChartOfAccount $currentAssetsAccount
     * @return ChartOfAccount
     */
    public function createReceivablesAccount(ChartOfAccount $currentAssetsAccount)
    {
        // إنشاء حساب الذمم المدينة (1200)
        return ChartOfAccount::create([
            'company_id' => $currentAssetsAccount->company_id,
            'parent_id' => $currentAssetsAccount->id,
            'account_code' => '1200',
            'name' => 'الذمم المدينة',
            'name_en' => 'Accounts Receivable',
            'account_type' => 'asset',
            'sub_type' => 'receivable',
            'description' => 'حساب الذمم المدينة',
            'is_active' => true,
            'is_system' => true,
            'opening_balance' => 0,
            'current_balance' => 0,
            'currency' => 'SAR',
            'balance_type' => 'debit',
            'is_cash_account' => false,
            'is_bank_account' => false,
            'level' => $currentAssetsAccount->level + 1,
            'is_parent' => true
        ]);
    }
}
