<?php

namespace App\Services;

use App\Models\CashRegister;
use App\Models\ChartOfAccount;
use App\Models\CashRegisterAccount;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CashRegisterService
{
    /**
     * إنشاء أو تحديث حساب في شجرة الحسابات للخزينة
     *
     * @param CashRegister $cashRegister
     * @return void
     */
    public function createOrUpdateCashRegisterAccount(CashRegister $cashRegister)
    {
        try {
            DB::beginTransaction();

            // البحث عن حساب النقدية والبنوك (الحساب الأب للخزائن)
            $cashAccountsParent = ChartOfAccount::where('account_code', '1110')
                ->where('company_id', $cashRegister->company_id)
                ->first();

            if (!$cashAccountsParent) {
                // إذا لم يوجد حساب النقدية والبنوك، نبحث عن حساب الأصول المتداولة
                $currentAssetsAccount = ChartOfAccount::where('account_code', '1100')
                    ->where('company_id', $cashRegister->company_id)
                    ->first();

                if (!$currentAssetsAccount) {
                    // إذا لم يوجد حساب الأصول المتداولة، نبحث عن حساب الأصول
                    $assetsAccount = ChartOfAccount::where('account_code', '1000')
                        ->where('company_id', $cashRegister->company_id)
                        ->first();

                    if (!$assetsAccount) {
                        // إنشاء هيكل حسابات الأصول الأساسي بالكامل
                        $cashAccountsParent = $this->createAssetsAccountStructure($cashRegister->company_id);
                    } else {
                        // إنشاء حساب الأصول المتداولة تحت حساب الأصول
                        $currentAssetsAccount = $this->createCurrentAssetsAccount($assetsAccount);
                        // ثم إنشاء حساب النقدية والبنوك تحت حساب الأصول المتداولة
                        $cashAccountsParent = $this->createCashAccountUnderCurrentAssets($currentAssetsAccount);
                    }
                } else {
                    // إنشاء حساب النقدية والبنوك تحت حساب الأصول المتداولة
                    $cashAccountsParent = $this->createCashAccountUnderCurrentAssets($currentAssetsAccount);
                }
            }

            // البحث عن الحساب المرتبط بالخزينة إن وجد
            $cashRegisterAccount = $cashRegister->cashRegisterAccounts()->first();
            $chartAccount = null;

            if ($cashRegisterAccount) {
                // إذا كان هناك حساب مرتبط بالخزينة، نقوم بتحديثه
                $chartAccount = ChartOfAccount::find($cashRegisterAccount->account_id);

                if ($chartAccount) {
                    $chartAccount->name = $cashRegister->name;
                    $chartAccount->description = 'حساب خزينة: ' . $cashRegister->name;
                    $chartAccount->save();
                } else {
                    // إذا لم يوجد الحساب في شجرة الحسابات، نقوم بإنشائه
                    $chartAccount = $this->createChartAccount($cashRegister, $cashAccountsParent);

                    // تحديث العلاقة في جدول cash_register_accounts
                    $cashRegisterAccount->account_id = $chartAccount->id;
                    $cashRegisterAccount->save();
                }
            } else {
                // إنشاء حساب جديد في شجرة الحسابات
                $chartAccount = $this->createChartAccount($cashRegister, $cashAccountsParent);

                // إنشاء علاقة جديدة في جدول cash_register_accounts
                CashRegisterAccount::create([
                    'cash_register_id' => $cashRegister->id,
                    'account_id' => $chartAccount->id
                ]);
            }

            DB::commit();
            return $chartAccount;
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('خطأ في إنشاء حساب الخزينة: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * إنشاء حساب جديد في شجرة الحسابات للخزينة
     *
     * @param CashRegister $cashRegister
     * @param ChartOfAccount $parentAccount
     * @return ChartOfAccount
     */
    private function createChartAccount(CashRegister $cashRegister, ChartOfAccount $parentAccount)
    {
        // توليد رمز حساب فريد
        $lastAccountCode = ChartOfAccount::where('parent_id', $parentAccount->id)
            ->orderBy('account_code', 'desc')
            ->value('account_code');

        $newAccountCode = $lastAccountCode ? (intval($lastAccountCode) + 1) : ($parentAccount->account_code . '01');

        // إنشاء حساب جديد في شجرة الحسابات
        return ChartOfAccount::create([
            'company_id' => $cashRegister->company_id,
            'parent_id' => $parentAccount->id,
            'account_code' => $newAccountCode,
            'name' => $cashRegister->name,
            'name_en' => $cashRegister->name,
            'account_type' => 'asset',
            'sub_type' => 'cash',
            'description' => 'حساب خزينة: ' . $cashRegister->name,
            'is_active' => true,
            'is_system' => false,
            'opening_balance' => $cashRegister->initial_balance,
            'current_balance' => $cashRegister->current_balance,
            'currency' => $cashRegister->currency,
            'balance_type' => 'debit',
            'is_cash_account' => true,
            'is_bank_account' => false,
            'level' => $parentAccount->level + 1,
            'is_parent' => false
        ]);
    }

    /**
     * إنشاء هيكل حسابات الأصول الأساسي
     *
     * @param int $companyId
     * @return ChartOfAccount
     */
    private function createAssetsAccountStructure($companyId)
    {
        // إنشاء حساب الأصول (1000)
        $assetsAccount = ChartOfAccount::create([
            'company_id' => $companyId,
            'parent_id' => null,
            'account_code' => '1000',
            'name' => 'الأصول',
            'name_en' => 'Assets',
            'account_type' => 'asset',
            'sub_type' => 'asset',
            'description' => 'حساب الأصول الرئيسي',
            'is_active' => true,
            'is_system' => true,
            'opening_balance' => 0,
            'current_balance' => 0,
            'currency' => 'SAR',
            'balance_type' => 'debit',
            'is_cash_account' => false,
            'is_bank_account' => false,
            'level' => 1,
            'is_parent' => true
        ]);

        // إنشاء حساب الأصول المتداولة (1100)
        $currentAssetsAccount = ChartOfAccount::create([
            'company_id' => $companyId,
            'parent_id' => $assetsAccount->id,
            'account_code' => '1100',
            'name' => 'الأصول المتداولة',
            'name_en' => 'Current Assets',
            'account_type' => 'asset',
            'sub_type' => 'current_asset',
            'description' => 'حساب الأصول المتداولة',
            'is_active' => true,
            'is_system' => true,
            'opening_balance' => 0,
            'current_balance' => 0,
            'currency' => 'SAR',
            'balance_type' => 'debit',
            'is_cash_account' => false,
            'is_bank_account' => false,
            'level' => 2,
            'is_parent' => true
        ]);

        // إنشاء حساب النقدية والبنوك (1110)
        $cashAccount = ChartOfAccount::create([
            'company_id' => $companyId,
            'parent_id' => $currentAssetsAccount->id,
            'account_code' => '1110',
            'name' => 'النقدية والبنوك',
            'name_en' => 'Cash and Banks',
            'account_type' => 'asset',
            'sub_type' => 'cash',
            'description' => 'حساب النقدية والبنوك',
            'is_active' => true,
            'is_system' => true,
            'opening_balance' => 0,
            'current_balance' => 0,
            'currency' => 'SAR',
            'balance_type' => 'debit',
            'is_cash_account' => true,
            'is_bank_account' => false,
            'level' => 3,
            'is_parent' => true
        ]);

        return $cashAccount;
    }

    /**
     * إنشاء حساب النقدية والبنوك تحت حساب الأصول المتداولة
     *
     * @param ChartOfAccount $currentAssetsAccount
     * @return ChartOfAccount
     */
    private function createCashAccountUnderCurrentAssets(ChartOfAccount $currentAssetsAccount)
    {
        // إنشاء حساب النقدية والبنوك (1110)
        return ChartOfAccount::create([
            'company_id' => $currentAssetsAccount->company_id,
            'parent_id' => $currentAssetsAccount->id,
            'account_code' => '1110',
            'name' => 'النقدية والبنوك',
            'name_en' => 'Cash and Banks',
            'account_type' => 'asset',
            'sub_type' => 'cash',
            'description' => 'حساب النقدية والبنوك',
            'is_active' => true,
            'is_system' => true,
            'opening_balance' => 0,
            'current_balance' => 0,
            'currency' => 'SAR',
            'balance_type' => 'debit',
            'is_cash_account' => true,
            'is_bank_account' => false,
            'level' => $currentAssetsAccount->level + 1,
            'is_parent' => true
        ]);
    }

    /**
     * إنشاء حساب الأصول المتداولة تحت حساب الأصول
     *
     * @param ChartOfAccount $assetsAccount
     * @return ChartOfAccount
     */
    private function createCurrentAssetsAccount(ChartOfAccount $assetsAccount)
    {
        // إنشاء حساب الأصول المتداولة (1100)
        return ChartOfAccount::create([
            'company_id' => $assetsAccount->company_id,
            'parent_id' => $assetsAccount->id,
            'account_code' => '1100',
            'name' => 'الأصول المتداولة',
            'name_en' => 'Current Assets',
            'account_type' => 'asset',
            'sub_type' => 'current_asset',
            'description' => 'حساب الأصول المتداولة',
            'is_active' => true,
            'is_system' => true,
            'opening_balance' => 0,
            'current_balance' => 0,
            'currency' => 'SAR',
            'balance_type' => 'debit',
            'is_cash_account' => false,
            'is_bank_account' => false,
            'level' => $assetsAccount->level + 1,
            'is_parent' => true
        ]);
    }
}
