<?php

namespace App\Services\Calendar;

use App\Models\InvoiceInstallment;
use App\Models\CalendarEvent;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class InstallmentCalendarService
{
    /**
     * إنشاء أحداث التقويم للأقساط.
     *
     * @param int $companyId
     * @param array $filters
     * @return array
     */
    public function syncInstallmentsToCalendar($companyId, array $filters = [])
    {
        try {
            DB::beginTransaction();

            // تحديد الفترة الزمنية
            $startDate = isset($filters['start_date']) ? Carbon::parse($filters['start_date']) : Carbon::now()->startOfMonth();
            $endDate = isset($filters['end_date']) ? Carbon::parse($filters['end_date']) : Carbon::now()->addMonths(3)->endOfMonth();

            // حذف أحداث التقويم القديمة للأقساط
            CalendarEvent::where('company_id', $companyId)
                ->where('event_type', 'installment')
                ->delete();

            // الحصول على الأقساط المستحقة خلال الفترة
            $installments = InvoiceInstallment::whereBetween('due_date', [$startDate, $endDate])
                ->whereHas('invoice', function ($query) use ($companyId) {
                    $query->where('company_id', $companyId);
                })
                ->with(['invoice', 'invoice.customer'])
                ->get();

            $createdCount = 0;

            // إنشاء أحداث التقويم للأقساط
            foreach ($installments as $installment) {
                $invoice = $installment->invoice;
                $customer = $invoice->customer;

                $eventTitle = "قسط مستحق - {$customer->name}";
                $eventDescription = "قسط مستحق للفاتورة رقم {$invoice->invoice_number} بقيمة " . number_format($installment->amount, 2) . " ريال";

                $event = new CalendarEvent([
                    'company_id' => $companyId,
                    'title' => $eventTitle,
                    'description' => $eventDescription,
                    'start_date' => $installment->due_date,
                    'end_date' => $installment->due_date->copy()->addHours(1),
                    'all_day' => true,
                    'event_type' => 'installment',
                    'reference_type' => 'installment',
                    'reference_id' => $installment->id,
                    'color' => $this->getEventColor($installment),
                    'created_by' => auth()->id() ?? 1,
                ]);

                $event->save();
                $createdCount++;
            }

            DB::commit();

            return [
                'success' => true,
                'message' => "تم مزامنة {$createdCount} قسط مع التقويم بنجاح.",
                'count' => $createdCount,
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('خطأ في مزامنة الأقساط مع التقويم: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'حدث خطأ أثناء مزامنة الأقساط مع التقويم: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * الحصول على لون حدث التقويم بناءً على حالة القسط.
     *
     * @param InvoiceInstallment $installment
     * @return string
     */
    protected function getEventColor(InvoiceInstallment $installment)
    {
        if ($installment->isPaid()) {
            return '#28a745'; // أخضر
        } elseif ($installment->isOverdue()) {
            return '#dc3545'; // أحمر
        } else {
            $daysRemaining = Carbon::now()->startOfDay()->diffInDays($installment->due_date, false);

            if ($daysRemaining <= 3) {
                return '#ffc107'; // أصفر
            } else {
                return '#17a2b8'; // أزرق
            }
        }
    }

    /**
     * الحصول على أحداث التقويم للأقساط.
     *
     * @param int $companyId
     * @param string $startDate
     * @param string $endDate
     * @return array
     */
    public function getInstallmentCalendarEvents($companyId, $startDate, $endDate)
    {
        try {
            $startDate = Carbon::parse($startDate)->startOfDay();
            $endDate = Carbon::parse($endDate)->endOfDay();

            $events = CalendarEvent::where('company_id', $companyId)
                ->where('event_type', 'installment')
                ->whereBetween('start_date', [$startDate, $endDate])
                ->get();

            $formattedEvents = [];

            foreach ($events as $event) {
                $formattedEvents[] = [
                    'id' => $event->id,
                    'title' => $event->title,
                    'description' => $event->description,
                    'start' => $event->start_date->format('Y-m-d\TH:i:s'),
                    'end' => $event->end_date->format('Y-m-d\TH:i:s'),
                    'allDay' => $event->all_day,
                    'color' => $event->color,
                    'reference_type' => $event->reference_type,
                    'reference_id' => $event->reference_id,
                ];
            }

            return [
                'success' => true,
                'data' => $formattedEvents,
            ];
        } catch (\Exception $e) {
            \Log::error('خطأ في الحصول على أحداث التقويم للأقساط: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'حدث خطأ أثناء الحصول على أحداث التقويم: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * تحديث حدث تقويم للقسط.
     *
     * @param InvoiceInstallment $installment
     * @return array
     */
    public function updateInstallmentCalendarEvent(InvoiceInstallment $installment)
    {
        try {
            $companyId = $installment->invoice->company_id;

            // البحث عن حدث التقويم للقسط
            $event = CalendarEvent::where('company_id', $companyId)
                ->where('event_type', 'installment')
                ->where('reference_type', 'installment')
                ->where('reference_id', $installment->id)
                ->first();

            if (!$event) {
                // إنشاء حدث جديد إذا لم يكن موجوداً
                $invoice = $installment->invoice;
                $customer = $invoice->customer;

                $eventTitle = "قسط مستحق - {$customer->name}";
                $eventDescription = "قسط مستحق للفاتورة رقم {$invoice->invoice_number} بقيمة " . number_format($installment->amount, 2) . " ريال";

                $event = new CalendarEvent([
                    'company_id' => $companyId,
                    'title' => $eventTitle,
                    'description' => $eventDescription,
                    'start_date' => $installment->due_date,
                    'end_date' => $installment->due_date->copy()->addHours(1),
                    'all_day' => true,
                    'event_type' => 'installment',
                    'reference_type' => 'installment',
                    'reference_id' => $installment->id,
                    'color' => $this->getEventColor($installment),
                    'created_by' => auth()->id() ?? 1,
                ]);

                $event->save();

                return [
                    'success' => true,
                    'message' => 'تم إنشاء حدث التقويم للقسط بنجاح.',
                ];
            } else {
                // تحديث الحدث الموجود
                $invoice = $installment->invoice;
                $customer = $invoice->customer;

                $eventTitle = "قسط مستحق - {$customer->name}";
                $eventDescription = "قسط مستحق للفاتورة رقم {$invoice->invoice_number} بقيمة " . number_format($installment->amount, 2) . " ريال";

                $event->title = $eventTitle;
                $event->description = $eventDescription;
                $event->start_date = $installment->due_date;
                $event->end_date = $installment->due_date->copy()->addHours(1);
                $event->color = $this->getEventColor($installment);
                $event->save();

                return [
                    'success' => true,
                    'message' => 'تم تحديث حدث التقويم للقسط بنجاح.',
                ];
            }
        } catch (\Exception $e) {
            \Log::error('خطأ في تحديث حدث التقويم للقسط: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'حدث خطأ أثناء تحديث حدث التقويم: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * حذف حدث تقويم للقسط.
     *
     * @param InvoiceInstallment $installment
     * @return array
     */
    public function deleteInstallmentCalendarEvent(InvoiceInstallment $installment)
    {
        try {
            $companyId = $installment->invoice->company_id;

            // حذف حدث التقويم للقسط
            $deleted = CalendarEvent::where('company_id', $companyId)
                ->where('event_type', 'installment')
                ->where('reference_type', 'installment')
                ->where('reference_id', $installment->id)
                ->delete();

            return [
                'success' => true,
                'message' => 'تم حذف حدث التقويم للقسط بنجاح.',
                'deleted' => $deleted,
            ];
        } catch (\Exception $e) {
            \Log::error('خطأ في حذف حدث التقويم للقسط: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'حدث خطأ أثناء حذف حدث التقويم: ' . $e->getMessage(),
            ];
        }
    }
}
