<?php

namespace App\Services;

use App\Models\Bank;
use App\Models\ChartOfAccount;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class BankAccountService
{
    /**
     * إنشاء حساب للبنك في شجرة الحسابات وربطه به
     * تم تعديل الدالة لإنشاء حساب البنك الأب فقط دون إنشاء حساب فرعي
     *
     * @param Bank $bank
     * @return ChartOfAccount|null
     */
    public function createBankAccount(Bank $bank)
    {
        try {
            // البحث عن حساب الأصول المتداولة (الحساب الأب للبنوك)
            $currentAssetsAccount = $this->findOrCreateCurrentAssetsAccount($bank->company_id);

            if (!$currentAssetsAccount) {
                Log::error('فشل في العثور على أو إنشاء حساب الأصول المتداولة', [
                    'bank_id' => $bank->id,
                    'company_id' => $bank->company_id
                ]);
                return null;
            }

            // البحث عن حساب البنك الأب (بنفس اسم البنك) أو إنشائه إذا لم يكن موجوداً
            $parentBankAccount = $this->findOrCreateBankParentAccount($bank, $currentAssetsAccount);

            if (!$parentBankAccount) {
                Log::error('فشل في العثور على أو إنشاء حساب البنك الأب', [
                    'bank_id' => $bank->id,
                    'bank_name' => $bank->name,
                    'company_id' => $bank->company_id
                ]);
                return null;
            }

            // إرجاع حساب البنك الأب مباشرة دون إنشاء حساب فرعي
            Log::info('تم إنشاء/استخدام حساب البنك الأب فقط في شجرة الحسابات', [
                'bank_id' => $bank->id,
                'bank_name' => $bank->name,
                'account_id' => $parentBankAccount->id,
                'account_code' => $parentBankAccount->account_code,
                'user_id' => auth()->id() ?? 'system'
            ]);

            return $parentBankAccount;
        } catch (\Exception $e) {
            Log::error('خطأ عام في إنشاء حساب البنك: ' . $e->getMessage(), [
                'bank_id' => $bank->id,
                'bank_name' => $bank->name,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return null;
        }
    }

    /**
     * البحث عن حساب البنك الأب أو إنشائه إذا لم يكن موجوداً
     *
     * @param Bank $bank
     * @param ChartOfAccount $currentAssetsAccount
     * @return ChartOfAccount|null
     */
    private function findOrCreateBankParentAccount(Bank $bank, $currentAssetsAccount)
    {
        try {
            // البحث عن حساب البنك الأب باستخدام اسم البنك
            $parentBankAccount = ChartOfAccount::withTrashed()
                ->where('company_id', $bank->company_id)
                ->where('name', $bank->name)
                ->where('is_parent', true)
                ->where('is_bank_account', true)
                ->first();

            if ($parentBankAccount) {
                // إذا كان الحساب محذوفاً، نقوم باسترجاعه
                if ($parentBankAccount->trashed()) {
                    $parentBankAccount->restore();
                    Log::info('تم استرجاع حساب البنك الأب المحذوف', [
                        'account_id' => $parentBankAccount->id,
                        'bank_id' => $bank->id
                    ]);
                }

                // تحديث parent_id إذا كان مختلفاً
                if ($parentBankAccount->parent_id != $currentAssetsAccount->id) {
                    $parentBankAccount->parent_id = $currentAssetsAccount->id;
                    $parentBankAccount->level = $currentAssetsAccount->level + 1;
                    $parentBankAccount->save();
                    Log::info('تم تحديث الحساب الأب لحساب البنك', [
                        'account_id' => $parentBankAccount->id,
                        'parent_id' => $currentAssetsAccount->id
                    ]);
                }

                return $parentBankAccount;
            }

            // إنشاء حساب البنك الأب
            $bankCode = 'BANK-' . $bank->id . '-PARENT';
            $parentBankAccount = ChartOfAccount::create([
                'company_id' => $bank->company_id,
                'account_code' => $bankCode,
                'name' => $bank->name,
                'name_en' => $bank->name,
                'account_type' => 'asset',
                'sub_type' => 'bank',
                'parent_id' => $currentAssetsAccount->id,
                'description' => 'حساب البنك الأب: ' . $bank->name,
                'is_active' => $bank->is_active,
                'is_system' => true,
                'opening_balance' => 0,
                'current_balance' => 0,
                'currency' => 'SAR',
                'balance_type' => 'debit',
                'is_cash_account' => false,
                'is_bank_account' => true,
                'level' => $currentAssetsAccount->level + 1,
                'is_parent' => true
            ]);

            Log::info('تم إنشاء حساب البنك الأب', [
                'account_id' => $parentBankAccount->id,
                'bank_id' => $bank->id,
                'bank_name' => $bank->name
            ]);

            return $parentBankAccount;
        } catch (\Exception $e) {
            Log::error('خطأ في البحث عن أو إنشاء حساب البنك الأب', [
                'bank_id' => $bank->id,
                'bank_name' => $bank->name,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * البحث عن حساب الأصول المتداولة أو إنشائه إذا لم يكن موجوداً
     *
     * @param int $companyId
     * @return ChartOfAccount
     */
    private function findOrCreateCurrentAssetsAccount($companyId)
    {
        // البحث عن حساب الأصول المتداولة
        $currentAssetsAccount = ChartOfAccount::where('company_id', $companyId)
            ->where('account_code', '1100')
            ->where('name', 'الأصول المتداولة')
            ->first();

        if (!$currentAssetsAccount) {
            // البحث عن حساب الأصول
            $assetsAccount = $this->findOrCreateAssetsAccount($companyId);

            // إنشاء حساب الأصول المتداولة تحت حساب الأصول
            $currentAssetsAccount = ChartOfAccount::create([
                'company_id' => $companyId,
                'parent_id' => $assetsAccount->id,
                'account_code' => '1100',
                'name' => 'الأصول المتداولة',
                'name_en' => 'Current Assets',
                'account_type' => 'asset',
                'sub_type' => 'current_asset',
                'description' => 'حساب الأصول المتداولة',
                'is_active' => true,
                'is_system' => true,
                'opening_balance' => 0,
                'current_balance' => 0,
                'currency' => 'SAR',
                'balance_type' => 'debit',
                'is_cash_account' => false,
                'is_bank_account' => false,
                'level' => $assetsAccount->level + 1,
                'is_parent' => true
            ]);

            Log::info('تم إنشاء حساب الأصول المتداولة في شجرة الحسابات', [
                'company_id' => $companyId,
                'account_id' => $currentAssetsAccount->id,
                'user_id' => auth()->id()
            ]);
        }

        return $currentAssetsAccount;
    }

    /**
     * البحث عن حساب الأصول أو إنشائه إذا لم يكن موجوداً
     *
     * @param int $companyId
     * @return ChartOfAccount
     */
    private function findOrCreateAssetsAccount($companyId)
    {
        // البحث عن حساب الأصول
        $assetsAccount = ChartOfAccount::where('company_id', $companyId)
            ->where('account_code', '1000')
            ->where('name', 'الأصول')
            ->first();

        if (!$assetsAccount) {
            // إنشاء حساب الأصول
            $assetsAccount = ChartOfAccount::create([
                'company_id' => $companyId,
                'parent_id' => null,
                'account_code' => '1000',
                'name' => 'الأصول',
                'name_en' => 'Assets',
                'account_type' => 'asset',
                'sub_type' => 'asset',
                'description' => 'حساب الأصول الرئيسي',
                'is_active' => true,
                'is_system' => true,
                'opening_balance' => 0,
                'current_balance' => 0,
                'currency' => 'SAR',
                'balance_type' => 'debit',
                'is_cash_account' => false,
                'is_bank_account' => false,
                'level' => 1,
                'is_parent' => true
            ]);

            Log::info('تم إنشاء حساب الأصول في شجرة الحسابات', [
                'company_id' => $companyId,
                'account_id' => $assetsAccount->id,
                'user_id' => auth()->id()
            ]);
        }

        return $assetsAccount;
    }

    /**
     * تحديث حساب البنك في شجرة الحسابات
     *
     * @param Bank $bank
     * @param ChartOfAccount $account
     * @return bool
     */
    public function updateBankAccount(Bank $bank, ChartOfAccount $account)
    {
        try {
            $account->update([
                'name' => $bank->name,
                'name_en' => $bank->name,
                'description' => 'حساب البنك: ' . $bank->name,
                'is_active' => $bank->is_active
            ]);

            Log::info('تم تحديث حساب البنك في شجرة الحسابات', [
                'bank_id' => $bank->id,
                'bank_name' => $bank->name,
                'account_id' => $account->id,
                'user_id' => auth()->id()
            ]);

            return true;
        } catch (\Exception $e) {
            Log::error('خطأ في تحديث حساب البنك في شجرة الحسابات', [
                'bank_id' => $bank->id,
                'bank_name' => $bank->name,
                'account_id' => $account->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return false;
        }
    }

    /**
     * البحث عن حساب البنك في شجرة الحسابات
     * تم تعديل الدالة للبحث عن حساب البنك الأب مباشرة
     *
     * @param Bank $bank
     * @return ChartOfAccount|null
     */
    public function findBankAccount(Bank $bank)
    {
        // البحث عن حساب الأصول المتداولة
        $currentAssetsAccount = $this->findOrCreateCurrentAssetsAccount($bank->company_id);

        // البحث عن حساب البنك الأب مباشرة
        return ChartOfAccount::where('company_id', $bank->company_id)
            ->where('name', $bank->name)
            ->where('is_bank_account', true)
            ->where('is_parent', true)
            ->where('parent_id', $currentAssetsAccount->id)
            ->first();
    }

    /**
     * إنشاء أو تحديث حساب البنك في شجرة الحسابات
     *
     * @param Bank $bank
     * @return ChartOfAccount|null
     */
    public function createOrUpdateBankAccount(Bank $bank)
    {
        // البحث عن حساب البنك في شجرة الحسابات
        $account = $this->findBankAccount($bank);

        if ($account) {
            // تحديث الحساب إذا كان موجوداً
            $this->updateBankAccount($bank, $account);
            return $account;
        } else {
            // إنشاء حساب جديد إذا لم يكن موجوداً
            return $this->createBankAccount($bank);
        }
    }

    /**
     * البحث عن حساب البنوك الرئيسي أو إنشائه إذا لم يكن موجوداً
     *
     * @param int $companyId
     * @param ChartOfAccount $currentAssetsAccount
     * @return ChartOfAccount
     */
    private function findOrCreateParentBankAccount($companyId, $currentAssetsAccount)
    {
        try {
            // إذا وصلنا إلى هنا، فقد فشلت كل المحاولات
            // نرجع حساب الأصول المتداولة كبديل مؤقت
            Log::warning('فشلت جميع محاولات إنشاء حساب البنوك الرئيسي، استخدام حساب الأصول المتداولة كبديل مؤقت', [
                'company_id' => $companyId,
                'current_assets_id' => $currentAssetsAccount->id
            ]);

            return $currentAssetsAccount;
        } catch (\Exception $e) {
            Log::error('خطأ في البحث عن أو إنشاء حساب البنوك الرئيسي', [
                'company_id' => $companyId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            // في حالة الخطأ، نحاول العثور على حساب البنوك الرئيسي مرة أخرى
            $existingAccount = ChartOfAccount::withTrashed()
                ->where('company_id', $companyId)
                ->where('account_code', 'BANK')
                ->first();

            if ($existingAccount) {
                if ($existingAccount->trashed()) {
                    $existingAccount->restore();
                }
                return $existingAccount;
            }

            // إذا لم نجد الحساب، نرجع حساب الأصول المتداولة كبديل مؤقت
            return $currentAssetsAccount;
        }
    }
}
