<?php

namespace App\Services;

use App\Models\ActivityLog;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Auth;

class ActivityLogService
{
    /**
     * تسجيل نشاط جديد.
     *
     * @param string $action
     * @param string|null $description
     * @param \Illuminate\Database\Eloquent\Model|null $subject
     * @param array|null $oldValues
     * @param array|null $newValues
     * @param string $level
     * @return \App\Models\ActivityLog
     */
    public function log($action, $description = null, $subject = null, $oldValues = null, $newValues = null, $level = 'info')
    {
        $user = Auth::user();

        $log = new ActivityLog();
        $log->user_id = $user ? $user->id : null;
        $log->company_id = $user ? $user->company_id : null;
        $log->branch_id = $user ? $user->branch_id : null;
        $log->action = $action;
        $log->description = $description;

        if ($subject instanceof Model) {
            $log->subject_type = get_class($subject);
            $log->subject_id = $subject->id;
        }

        $log->old_values = $oldValues;
        $log->new_values = $newValues;
        $log->ip_address = request()->ip();
        $log->user_agent = request()->userAgent();
        $log->url = request()->fullUrl();
        $log->method = request()->method();
        $log->level = $level;

        $log->save();

        return $log;
    }

    /**
     * تسجيل نشاط تسجيل الدخول.
     *
     * @param \App\Models\User $user
     * @return \App\Models\ActivityLog
     */
    public function logLogin($user)
    {
        return $this->log(
            'login',
            'تم تسجيل الدخول للمستخدم ' . $user->name,
            $user,
            null,
            null,
            'info'
        );
    }

    /**
     * تسجيل نشاط تسجيل الخروج.
     *
     * @param \App\Models\User $user
     * @return \App\Models\ActivityLog
     */
    public function logLogout($user)
    {
        return $this->log(
            'logout',
            'تم تسجيل الخروج للمستخدم ' . $user->name,
            $user,
            null,
            null,
            'info'
        );
    }

    /**
     * تسجيل نشاط فشل تسجيل الدخول.
     *
     * @param string $email
     * @param string $reason
     * @return \App\Models\ActivityLog
     */
    public function logFailedLogin($email, $reason = null)
    {
        return $this->log(
            'failed_login',
            'فشل تسجيل الدخول للبريد الإلكتروني ' . $email,
            null,
            null,
            ['email' => $email, 'reason' => $reason],
            'warning'
        );
    }

    /**
     * تسجيل نشاط إنشاء سجل.
     *
     * @param \Illuminate\Database\Eloquent\Model $model
     * @return \App\Models\ActivityLog
     */
    public function logCreated(Model $model)
    {
        $modelName = class_basename($model);

        return $this->log(
            'created',
            'تم إنشاء سجل جديد في ' . $modelName,
            $model,
            null,
            $model->toArray(),
            'info'
        );
    }

    /**
     * تسجيل نشاط تحديث سجل.
     *
     * @param \Illuminate\Database\Eloquent\Model $model
     * @param array $oldValues
     * @return \App\Models\ActivityLog
     */
    public function logUpdated(Model $model, array $oldValues)
    {
        $modelName = class_basename($model);

        return $this->log(
            'updated',
            'تم تحديث سجل في ' . $modelName,
            $model,
            $oldValues,
            $model->toArray(),
            'info'
        );
    }

    /**
     * تسجيل نشاط حذف سجل.
     *
     * @param \Illuminate\Database\Eloquent\Model $model
     * @return \App\Models\ActivityLog
     */
    public function logDeleted(Model $model)
    {
        $modelName = class_basename($model);

        return $this->log(
            'deleted',
            'تم حذف سجل من ' . $modelName,
            $model,
            $model->toArray(),
            null,
            'warning'
        );
    }

    /**
     * تسجيل نشاط استعادة سجل محذوف.
     *
     * @param \Illuminate\Database\Eloquent\Model $model
     * @return \App\Models\ActivityLog
     */
    public function logRestored(Model $model)
    {
        $modelName = class_basename($model);

        return $this->log(
            'restored',
            'تم استعادة سجل محذوف في ' . $modelName,
            $model,
            null,
            $model->toArray(),
            'info'
        );
    }

    /**
     * تسجيل نشاط تغيير الصلاحيات.
     *
     * @param \App\Models\User $user
     * @param array $oldPermissions
     * @param array $newPermissions
     * @return \App\Models\ActivityLog
     */
    public function logPermissionChange($user, array $oldPermissions, array $newPermissions)
    {
        return $this->log(
            'permission_change',
            'تم تغيير صلاحيات المستخدم ' . $user->name,
            $user,
            ['permissions' => $oldPermissions],
            ['permissions' => $newPermissions],
            'warning'
        );
    }

    /**
     * تسجيل نشاط تغيير الأدوار.
     *
     * @param \App\Models\User $user
     * @param array $oldRoles
     * @param array $newRoles
     * @return \App\Models\ActivityLog
     */
    public function logRoleChange($user, array $oldRoles, array $newRoles)
    {
        return $this->log(
            'role_change',
            'تم تغيير أدوار المستخدم ' . $user->name,
            $user,
            ['roles' => $oldRoles],
            ['roles' => $newRoles],
            'warning'
        );
    }
}
