<?php

namespace App\Services;

use App\Models\ChartOfAccount;
use Illuminate\Support\Facades\DB;

class AccountingService
{
    /**
     * التحقق من وجود الحسابات الرئيسية وإنشائها إذا لم تكن موجودة
     *
     * @param int $companyId
     * @return array
     */
    public function ensureParentAccounts($companyId)
    {
        DB::beginTransaction();

        try {
            // حساب الأصول
            $assetsAccount = $this->findOrCreateAccount([
                'company_id' => $companyId,
                'account_code' => '1000',
                'name' => 'الأصول',
                'name_en' => 'Assets',
                'account_type' => 'asset',
                'sub_type' => 'current_asset',
                'parent_id' => null,
                'description' => 'حساب الأصول الرئيسي',
                'is_active' => true,
                'is_system' => true,
                'balance_type' => 'debit',
                'level' => 1,
                'is_parent' => true
            ]);

            // حساب المخزون
            $inventoryAccount = $this->findOrCreateAccount([
                'company_id' => $companyId,
                'account_code' => '1200',
                'name' => 'المخزون',
                'name_en' => 'Inventory',
                'account_type' => 'asset',
                'sub_type' => 'inventory',
                'parent_id' => $assetsAccount->id,
                'description' => 'حساب المخزون الرئيسي',
                'is_active' => true,
                'is_system' => true,
                'balance_type' => 'debit',
                'level' => 2,
                'is_parent' => true
            ]);

            // حساب الإيرادات
            $revenueAccount = $this->findOrCreateAccount([
                'company_id' => $companyId,
                'account_code' => '4000',
                'name' => 'الإيرادات',
                'name_en' => 'Revenue',
                'account_type' => 'revenue',
                'sub_type' => 'sales',
                'parent_id' => null,
                'description' => 'حساب الإيرادات الرئيسي',
                'is_active' => true,
                'is_system' => true,
                'balance_type' => 'credit',
                'level' => 1,
                'is_parent' => true
            ]);

            // حساب المبيعات
            $salesAccount = $this->findOrCreateAccount([
                'company_id' => $companyId,
                'account_code' => '4100',
                'name' => 'المبيعات',
                'name_en' => 'Sales',
                'account_type' => 'revenue',
                'sub_type' => 'sales',
                'parent_id' => $revenueAccount->id,
                'description' => 'حساب المبيعات',
                'is_active' => true,
                'is_system' => true,
                'balance_type' => 'credit',
                'level' => 2,
                'is_parent' => true
            ]);

            // حساب مرتجع المبيعات
            $salesReturnsAccount = $this->findOrCreateAccount([
                'company_id' => $companyId,
                'account_code' => '4200',
                'name' => 'مرتجع المبيعات',
                'name_en' => 'Sales Returns',
                'account_type' => 'revenue',
                'sub_type' => 'sales_returns',
                'parent_id' => $revenueAccount->id,
                'description' => 'حساب مرتجع المبيعات',
                'is_active' => true,
                'is_system' => true,
                'balance_type' => 'debit',
                'level' => 2,
                'is_parent' => true
            ]);

            // حساب الخصوم
            $liabilitiesAccount = $this->findOrCreateAccount([
                'company_id' => $companyId,
                'account_code' => '2000',
                'name' => 'الخصوم',
                'name_en' => 'Liabilities',
                'account_type' => 'liability',
                'sub_type' => 'current_liability',
                'parent_id' => null,
                'description' => 'حساب الخصوم الرئيسي',
                'is_active' => true,
                'is_system' => true,
                'balance_type' => 'credit',
                'level' => 1,
                'is_parent' => true
            ]);

            // حساب ضريبة القيمة المضافة
            $vatAccount = $this->findOrCreateAccount([
                'company_id' => $companyId,
                'account_code' => '2100',
                'name' => 'ضريبة القيمة المضافة',
                'name_en' => 'VAT',
                'account_type' => 'liability',
                'sub_type' => 'tax',
                'parent_id' => $liabilitiesAccount->id,
                'description' => 'حساب ضريبة القيمة المضافة',
                'is_active' => true,
                'is_system' => true,
                'balance_type' => 'credit',
                'level' => 2,
                'is_parent' => true
            ]);

            // حساب المدينون
            $accountsReceivableAccount = $this->findOrCreateAccount([
                'company_id' => $companyId,
                'account_code' => '1300',
                'name' => 'المدينون',
                'name_en' => 'Accounts Receivable',
                'account_type' => 'asset',
                'sub_type' => 'receivable',
                'parent_id' => $assetsAccount->id,
                'description' => 'حساب المدينون',
                'is_active' => true,
                'is_system' => true,
                'balance_type' => 'debit',
                'level' => 2,
                'is_parent' => true
            ]);

            DB::commit();

            return [
                'assets' => $assetsAccount,
                'inventory' => $inventoryAccount,
                'revenue' => $revenueAccount,
                'sales' => $salesAccount,
                'sales_returns' => $salesReturnsAccount,
                'liabilities' => $liabilitiesAccount,
                'vat' => $vatAccount,
                'accounts_receivable' => $accountsReceivableAccount
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * البحث عن حساب أو إنشائه إذا لم يكن موجوداً
     *
     * @param array $data
     * @return ChartOfAccount
     */
    public function findOrCreateAccount($data)
    {
        $account = ChartOfAccount::where('company_id', $data['company_id'])
            ->where('account_code', $data['account_code'])
            ->first();

        if (!$account) {
            $account = ChartOfAccount::create($data);
        }

        return $account;
    }

    /**
     * إنشاء حساب عميل إذا لم يكن موجوداً
     *
     * @param int $companyId
     * @param int $customerId
     * @param string $customerName
     * @return ChartOfAccount
     */
    public function ensureCustomerAccount($companyId, $customerId, $customerName)
    {
        // البحث عن حساب المدينون الرئيسي
        $accountsReceivableAccount = ChartOfAccount::where('company_id', $companyId)
            ->where('account_type', 'asset')
            ->where('sub_type', 'receivable')
            ->where('is_parent', true)
            ->first();

        if (!$accountsReceivableAccount) {
            // إنشاء حساب المدينون الرئيسي إذا لم يكن موجوداً
            $accounts = $this->ensureParentAccounts($companyId);
            $accountsReceivableAccount = $accounts['accounts_receivable'];
        }

        // البحث عن حساب العميل - نبحث بناءً على اسم الحساب بدلاً من reference_id
        $customerAccount = ChartOfAccount::where('company_id', $companyId)
            ->where('sub_type', 'customer')
            ->where('name', 'like', '%' . $customerName . '%')
            ->first();

        if (!$customerAccount) {
            // إنشاء حساب العميل إذا لم يكن موجوداً
            $customerAccount = ChartOfAccount::create([
                'company_id' => $companyId,
                'account_code' => 'AR-' . $customerId,
                'name' => 'حساب العميل: ' . $customerName,
                'name_en' => 'Customer Account: ' . $customerName,
                'account_type' => 'asset',
                'sub_type' => 'customer',
                'parent_id' => $accountsReceivableAccount->id,
                'description' => 'حساب العميل: ' . $customerName,
                'is_active' => true,
                'is_system' => false,
                'balance_type' => 'debit',
                'level' => $accountsReceivableAccount->level + 1,
                'is_parent' => false
                // حذف الحقول غير الموجودة في الجدول
                // 'reference_type' => 'customer',
                // 'reference_id' => $customerId
            ]);
        }

        return $customerAccount;
    }

    /**
     * إنشاء قيد محاسبي
     *
     * @param array $data
     * @return \App\Models\JournalEntry
     */
    public function createJournalEntry($data)
    {
        return DB::transaction(function () use ($data) {
            // البحث عن السنة المالية النشطة للشركة
            $fiscalYear = \App\Models\FiscalYear::where('company_id', $data['company_id'])
                ->where('is_active', true)
                ->where('is_closed', false)
                ->first();

            if (!$fiscalYear) {
                throw new \Exception('لا توجد سنة مالية نشطة للشركة');
            }

            // البحث عن الفترة المحاسبية المناسبة للتاريخ
            $entryDate = $data['entry_date'];
            $accountingPeriod = \App\Models\AccountingPeriod::where('fiscal_year_id', $fiscalYear->id)
                ->where('is_closed', false)
                ->where('start_date', '<=', $entryDate)
                ->where('end_date', '>=', $entryDate)
                ->first();

            if (!$accountingPeriod) {
                throw new \Exception('لا توجد فترة محاسبية نشطة للتاريخ المحدد');
            }

            // توليد رقم قيد محاسبي جديد إذا لم يتم تحديده
            $entryNumber = $data['entry_number'] ?? \App\Models\JournalEntry::generateEntryNumber($data['company_id']);

            $journalEntry = \App\Models\JournalEntry::create([
                'company_id' => $data['company_id'],
                'fiscal_year_id' => $fiscalYear->id,
                'accounting_period_id' => $accountingPeriod->id,
                'entry_number' => $entryNumber,
                'entry_date' => $data['entry_date'],
                'reference_number' => $data['reference_number'],
                'reference_type' => $data['reference_type'],
                'description' => $data['description'],
                'created_by' => $data['created_by'],
                'status' => $data['status'] ?? 'approved',
                'journalable_type' => $data['journalable_type'] ?? null,
                'journalable_id' => $data['journalable_id'] ?? null,
            ]);

            foreach ($data['items'] as $item) {
                $journalEntry->items()->create([
                    'account_id' => $item['account_id'],
                    'description' => $item['description'],
                    'debit' => $item['debit'] ?? 0,
                    'credit' => $item['credit'] ?? 0,
                ]);
            }

            return $journalEntry;
        });
    }
}
