<?php

namespace App\Policies;

use App\Models\InstallationProject;
use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class InstallationProjectPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any models.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function viewAny(User $user)
    {
        return $user->hasPermissionTo('view_installation_projects');
    }

    /**
     * Determine whether the user can view the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\InstallationProject  $project
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function view(User $user, InstallationProject $project)
    {
        // يمكن للمستخدم عرض المشروع إذا كان لديه صلاحية عرض المشاريع
        // أو إذا كان مدير المشروع أو عضو في فريق المشروع
        return $user->hasPermissionTo('view_installation_projects') ||
            $project->project_manager_id === $user->id ||
            $project->teamMembers()->where('user_id', $user->id)->exists();
    }

    /**
     * Determine whether the user can create models.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function create(User $user)
    {
        return $user->hasPermissionTo('create_installation_projects');
    }

    /**
     * Determine whether the user can update the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\InstallationProject  $project
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function update(User $user, InstallationProject $project)
    {
        // يمكن للمستخدم تعديل المشروع إذا كان لديه صلاحية تعديل المشاريع
        // أو إذا كان مدير المشروع
        return $user->hasPermissionTo('edit_installation_projects') ||
            $project->project_manager_id === $user->id;
    }

    /**
     * Determine whether the user can delete the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\InstallationProject  $project
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function delete(User $user, InstallationProject $project)
    {
        return $user->hasPermissionTo('delete_installation_projects');
    }

    /**
     * Determine whether the user can restore the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\InstallationProject  $project
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function restore(User $user, InstallationProject $project)
    {
        return $user->hasPermissionTo('delete_installation_projects');
    }

    /**
     * Determine whether the user can permanently delete the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\InstallationProject  $project
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function forceDelete(User $user, InstallationProject $project)
    {
        return $user->hasPermissionTo('delete_installation_projects');
    }

    /**
     * Determine whether the user can approve inventory requests.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\InstallationProject  $project
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function approveInventoryRequests(User $user, InstallationProject $project)
    {
        // يمكن للمستخدم اعتماد طلبات المخزون إذا كان لديه صلاحية اعتماد طلبات المخزون للمشاريع
        // أو إذا كان مدير المشروع
        return $user->hasPermissionTo('approve_project_inventory_requests') ||
            $project->project_manager_id === $user->id;
    }

    /**
     * Determine whether the user can reject inventory requests.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\InstallationProject  $project
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function rejectInventoryRequests(User $user, InstallationProject $project)
    {
        // يمكن للمستخدم رفض طلبات المخزون إذا كان لديه صلاحية رفض طلبات المخزون للمشاريع
        // أو إذا كان مدير المشروع
        return $user->hasPermissionTo('reject_project_inventory_requests') ||
            $project->project_manager_id === $user->id;
    }

    /**
     * Determine whether the user can fulfill inventory requests.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\InstallationProject  $project
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function fulfillInventoryRequests(User $user, InstallationProject $project)
    {
        // يمكن للمستخدم تنفيذ طلبات المخزون إذا كان لديه صلاحية تنفيذ طلبات المخزون للمشاريع
        return $user->hasPermissionTo('fulfill_project_inventory_requests');
    }
}
