<?php

namespace App\Observers;

use App\Models\Customer;
use App\Services\CustomerNotificationService;
use Illuminate\Support\Facades\Log;

class CustomerObserver
{
    /**
     * خدمة إشعارات العملاء
     *
     * @var \App\Services\CustomerNotificationService
     */
    protected $customerNotificationService;

    /**
     * إنشاء مثيل جديد من المراقب
     *
     * @param \App\Services\CustomerNotificationService $customerNotificationService
     */
    public function __construct(CustomerNotificationService $customerNotificationService)
    {
        $this->customerNotificationService = $customerNotificationService;
    }

    /**
     * التعامل مع حدث "created" للعميل.
     *
     * @param  \App\Models\Customer  $customer
     * @return void
     */
    public function created(Customer $customer)
    {
        try {
            // إرسال إشعار للعميل الجديد
            $this->customerNotificationService->newCustomerNotification($customer);
        } catch (\Exception $e) {
            Log::error('فشل إرسال إشعار للعميل الجديد: ' . $e->getMessage());
        }
    }

    /**
     * التعامل مع حدث "updated" للعميل.
     *
     * @param  \App\Models\Customer  $customer
     * @return void
     */
    public function updated(Customer $customer)
    {
        try {
            // تحديد الحقول المهمة التي تم تغييرها
            $changedFields = [];
            $importantFields = ['name', 'email', 'phone', 'mobile', 'address', 'status', 'is_active'];

            foreach ($importantFields as $field) {
                if ($customer->wasChanged($field)) {
                    $changedFields[$field] = [
                        'old' => $customer->getOriginal($field),
                        'new' => $customer->$field
                    ];
                }
            }

            // إذا كانت هناك تغييرات مهمة، أرسل إشعاراً
            if (!empty($changedFields)) {
                $this->customerNotificationService->customerUpdateNotification($customer, $changedFields);
            }

            // إضافة سجل للتشخيص
            Log::info('تحديث العميل: ' . $customer->id . ', التغييرات: ' . json_encode($changedFields));
        } catch (\Exception $e) {
            Log::error('فشل إرسال إشعار لتحديث العميل: ' . $e->getMessage());
        }
    }
}
