<?php

namespace App\Notifications;

use App\Models\InvoiceInstallment;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class InstallmentPaymentReceivedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * القسط.
     *
     * @var \App\Models\InvoiceInstallment
     */
    protected $installment;

    /**
     * مبلغ الدفعة.
     *
     * @var float
     */
    protected $amount;

    /**
     * إنشاء نسخة جديدة من الإشعار.
     *
     * @param \App\Models\InvoiceInstallment $installment
     * @param float $amount
     * @return void
     */
    public function __construct(InvoiceInstallment $installment, $amount)
    {
        $this->installment = $installment;
        $this->amount = $amount;
    }

    /**
     * الحصول على قنوات تسليم الإشعار.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function via($notifiable)
    {
        return ['mail', 'database'];
    }

    /**
     * الحصول على تمثيل البريد الإلكتروني للإشعار.
     *
     * @param  mixed  $notifiable
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toMail($notifiable)
    {
        $invoice = $this->installment->invoice;
        $customer = $invoice->customer;
        $remainingAmount = $this->installment->amount - $this->installment->paid_amount;
        $isPaid = $remainingAmount <= 0;

        return (new MailMessage)
            ->subject('تأكيد استلام دفعة قسط')
            ->greeting("مرحباً {$notifiable->name}")
            ->line("نود إعلامكم بأنه تم استلام دفعة بقيمة " . number_format($this->amount, 2) . " ريال للقسط المستحق.")
            ->line("رقم الفاتورة: {$invoice->invoice_number}")
            ->line("تاريخ الاستحقاق: {$this->installment->due_date->format('Y-m-d')}")
            ->line($isPaid
                ? "تم سداد القسط بالكامل."
                : "المبلغ المتبقي: " . number_format($remainingAmount, 2) . " ريال")
            ->line("شكراً لكم على الدفع في الوقت المحدد.")
            ->action('عرض تفاصيل القسط', url("/invoice-installments/{$this->installment->id}"));
    }

    /**
     * الحصول على تمثيل قاعدة البيانات للإشعار.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function toArray($notifiable)
    {
        $invoice = $this->installment->invoice;
        $customer = $invoice->customer;
        $remainingAmount = $this->installment->amount - $this->installment->paid_amount;
        $isPaid = $remainingAmount <= 0;

        return [
            'title' => 'تأكيد استلام دفعة قسط',
            'message' => "تم استلام دفعة بقيمة " . number_format($this->amount, 2) . " ريال للقسط المستحق للفاتورة رقم {$invoice->invoice_number}.",
            'invoice_id' => $invoice->id,
            'installment_id' => $this->installment->id,
            'customer_id' => $customer->id,
            'amount' => $this->amount,
            'remaining_amount' => $remainingAmount,
            'is_paid' => $isPaid,
        ];
    }
}
